#drinc:exec/memory.g
#drinc:graphics/gfx.g
#drinc:graphics/rastport.g
#drinc:intuition/window.g
#drinc:intuition/image.g
#drinc:intuition/gadget.g
#drinc:devices/inputevent.g
#drinc:libraries/dos.g
#drinc:util.g

#text.g
#globals.g
#funcs.g

/*
 * Amiga MUD
 *
 * Copyright (c) 1997 by Chris Gray
 */

/*
 * edit.d - handle editing in the text window.
 */

uint MAX_EDIT_COLUMN = 160;

type
    EditLine_t = struct {
	*EditLine_t el_next;
	*EditLine_t el_prev;
	uint el_length;
	[MAX_EDIT_COLUMN] char el_text;
    };

uint LINE_SIZE = sizeof(EditLine_t) - MAX_EDIT_COLUMN * sizeof(char);

*EditLine_t FirstLine;
*EditLine_t CurrentLine;

ulong ScrollPot, ScrollBody;		/* so can avoid extra redraws */

uint CurrentLineNumber, CurrentColumnNumber, MaxLineNumber, MaxEditColumn;
uint WindowTopLine, PropFlags;
uint TextLines, TextInputTop, TextInputBottom;

uint BufferTotal;
[MAX_EDIT_COLUMN] char EditBuffer;
[100] char ErrorMessage;	/* needed for refresh events */

bool Unpacked, ErrorShown, HaveError, HadAnyError, FirstGetLine,
    EditParsing, EditAborting, IsRaw, HelpShown;

/*
 * ed_init - called once on startup
 */

proc ed_init()void:

    MaxEditColumn := TextColumns;
    if MaxEditColumn > MAX_EDIT_COLUMN then
	MaxEditColumn := MAX_EDIT_COLUMN;
    fi;
    PropFlags :=
	if OldStyle then
	    AUTOKNOB | FREEVERT
	else
	    AUTOKNOB | FREEVERT | PROPNEWLOOK
	fi;
    if Mode = md_oneWindow then
	TextLines := TextHalfLines;
	TextInputTop := TextHalfInputTop;
	TextInputBottom := TextHalfInputBottom;
    else
	TextLines := TextFullLines;
	TextInputTop := TextFullInputTop;
	TextInputBottom := TextFullInputBottom;
    fi;
corp;

/*
 * ed_freeBuffer - free up whatever is in the current edit buffer.
 */

proc ed_freeBuffer()void:
    register *EditLine_t el;

    while
	el := FirstLine;
	el ~= nil
    do
	FirstLine := el*.el_next;
	FreeMem(el, LINE_SIZE + el*.el_length * sizeof(char));
    od;
corp;

/*
 * ed_clearError - clear the error line.
 */

proc ed_clearError()void:

    SetAPen(TextRastPort, TextPen);
    RectFill(TextRastPort,
	LeftBorder, TextInputTop, RightEdge, TextInputBottom);
    ErrorShown := false;
corp;

/*
 * ed_showError - put an error comment into the error area.
 */

proc ed_showError(*char message)void:
    register *RastPort_t rp;
    ulong len;

    if ErrorShown then
	ed_clearError();
    fi;
    rp := TextRastPort;
    Move(rp, LeftBorder,
	TextInputBottom - (CHAR_HEIGHT - CHAR_BASELINE - 1));
    SetAPen(rp, ErasePen);
    SetBPen(rp, TextPen);
    len := CharsLen(message);
    if len > TextColumns then
	len := TextColumns;
    fi;
    Text(rp, message, len);
    SetAPen(rp, TextPen);
    SetBPen(rp, ErasePen);
    ErrorShown := true;
    CharsCopy(&ErrorMessage[0], message);
corp;

/*
 * ed_noMem - error utility
 */

proc ed_noMem()void:

    ed_showError("Out of memory");
corp;

/*
 * ed_notBound - error utility
 */

proc ed_notBound()void:

    ed_showError("That key performs no action");
corp;

/*
 * ed_repaint - repaint a portion of the viewable window.
 */

proc ed_repaint(register uint windowTop, windowBottom)void:
    register *EditLine_t el;
    register *RastPort_t rp;
    register uint l;

    rp := TextRastPort;
    l := TextTop + windowTop * CHAR_HEIGHT;
    SetAPen(rp, ErasePen);
    RectFill(rp, LeftBorder, l, RightEdge,
	     l + (windowBottom - windowTop + 1) * CHAR_HEIGHT - 1);
    SetAPen(rp, TextPen);
    Move(rp, LeftBorder, l + CHAR_BASELINE);
    el := FirstLine;
    l := WindowTopLine;
    while l ~= 0 and el ~= nil do
	l := l - 1;
	el := el*.el_next;
    od;
    l := windowTop;
    while l ~= 0 and el ~= nil do
	l := l - 1;
	el := el*.el_next;
    od;
    if el ~= nil then
	while
	    l := el*.el_length;
	    if l > TextColumns then
		l := TextColumns;
	    fi;
	    Text(rp, &el*.el_text[0], l);
	    el := el*.el_next;
	    windowTop ~= windowBottom and el ~= nil
	do
	    windowTop := windowTop + 1;
	    Move(rp, LeftBorder, rp*.rp_cp_y + CHAR_HEIGHT);
	od;
    fi;
corp;

/*
 * ed_refreshLine - redraw the line the cursor is on.
 */

proc ed_refreshLine()void:
    register *RastPort_t rp;
    register uint y;

    rp := TextRastPort;
    y := (CurrentLineNumber - WindowTopLine) * CHAR_HEIGHT + TextTop;
    SetAPen(rp, ErasePen);
    RectFill(rp, LeftBorder, y, RightEdge, y + (CHAR_HEIGHT - 1));
    SetAPen(rp, TextPen);
    Move(rp, LeftBorder, y + CHAR_BASELINE);
    if Unpacked then
	y := BufferTotal;
	if y > TextColumns then
	    y := TextColumns;
	fi;
	Text(rp, &EditBuffer[0], y);
    else
	y := CurrentLine*.el_length;
	if y > TextColumns then
	    y := TextColumns;
	fi;
	Text(rp, &CurrentLine*.el_text[0], y);
    fi;
corp;

/*
 * ed_unpack - unpack from CurrentLine to EditBuffer. Return the window X
 *	position of the current character.
 */

proc ed_unpack()uint:
    register *EditLine_t el;

    if not Unpacked then
	Unpacked := true;
	el := CurrentLine;
	BufferTotal := el*.el_length;
	BlockCopy(&EditBuffer[0], &el*.el_text[0], el*.el_length);
	if CurrentColumnNumber > el*.el_length then
	    CurrentColumnNumber := el*.el_length;
	fi;
    fi;
    CurrentColumnNumber * CHAR_WIDTH + LeftBorder
corp;

/*
 * ed_pack - pack from EditBuffer to CurrentLine.
 */

proc ed_pack()void:
    register *EditLine_t el, el2;

    if Unpacked then
	Unpacked := false;
	el2 := CurrentLine;
	if el2*.el_length ~= BufferTotal then
	    el := mudAlloc(LINE_SIZE + BufferTotal * sizeof(char), 0x0);
	    if el = nil then
		ed_noMem();
		ed_refreshLine();
	    else
		el*.el_prev := el2*.el_prev;
		el*.el_next := el2*.el_next;
		el*.el_length := BufferTotal;
		BlockCopy(&el*.el_text[0], &EditBuffer[0], el*.el_length);
		if el*.el_prev ~= nil then
		    el*.el_prev*.el_next := el;
		else
		    FirstLine := el;
		fi;
		if el*.el_next ~= nil then
		    el*.el_next*.el_prev := el;
		fi;
		CurrentLine := el;
		FreeMem(el2, LINE_SIZE + el2*.el_length * sizeof(char));
	    fi;
	else
	    BlockCopy(&el2*.el_text[0], &EditBuffer[0], el2*.el_length);
	fi;
    fi;
corp;

/*
 * ed_cursor - toggle/draw a cursor at the current position.
 */

proc ed_cursor()void:
    register *RastPort_t rp;
    register ulong column, row;

    rp := TextRastPort;
    column := CurrentColumnNumber;
    if column >= TextColumns then
	column := TextColumns - 1;
    fi;
    column := column * CHAR_WIDTH;
    column := column + LeftBorder;
    row := (CurrentLineNumber - WindowTopLine) * CHAR_HEIGHT + TextTop;
    SetDrMd(rp, COMPLEMENT);
    RectFill(rp, column, row,
	     column + (CHAR_WIDTH - 1), row + (CHAR_HEIGHT - 1));
    SetDrMd(rp, JAM1);
corp;

/*
 * ed_drawScroll - redraw the scrollbar on the right.
 */

proc ed_drawScroll()void:
    register ulong newPot, newBody;
    register uint max, pos;

    max := TextLines - 1;
    if MaxLineNumber <= max then
	newPot := 0;
	newBody := 0xffff;
    else
	max := MaxLineNumber - max;
	pos := WindowTopLine;
	if pos > max then
	    pos := max;
	fi;
	newPot := make(pos, ulong) * 0xffff / max;
	newBody := make(TextLines - 1, ulong) * 0xffff / MaxLineNumber;
    fi;
    if newPot ~= ScrollPot or newBody ~= ScrollBody then
	NewModifyProp(ScrollGadget, TextWindow, nil, PropFlags,
		      0, newPot, 0, newBody, 1);
	ScrollPot := newPot;
	ScrollBody := newBody;
    fi;
corp;

/*
 * ed_doScroll - used when scrolling the window.
 */

proc ed_doScroll(int delta)void:

    tx_doAnyScroll(0, delta * CHAR_HEIGHT,
		   LeftBorder, TextTop, RightEdge, TextInputTop - 1);
corp;

/*
 * ed_scrollWindow - the actual work of scrolling the window.
 */

proc ed_scrollWindow(register uint where)void:
    register uint delta;

    if CurrentLineNumber < where then
	while CurrentLineNumber < where and CurrentLine*.el_next ~= nil do
	    CurrentLineNumber := CurrentLineNumber + 1;
	    CurrentLine := CurrentLine*.el_next;
	od;
    else
	delta := where + TextLines - 2;
	if CurrentLineNumber > delta then
	    while CurrentLineNumber > delta and CurrentLine*.el_prev ~= nil do
		CurrentLineNumber := CurrentLineNumber - 1;
		CurrentLine := CurrentLine*.el_prev;
	    od;
	fi;
    fi;

    delta := WindowTopLine - where;
    if where < WindowTopLine and delta < TextLines - 2 then
	/* do a true scroll */
	WindowTopLine := where;
	ed_doScroll(- delta);
	ed_repaint(0, delta - 1);
    else
	delta := where - WindowTopLine;
	if where > WindowTopLine and delta < TextLines - 2 then
	    /* do a true scroll */
	    WindowTopLine := where;
	    ed_doScroll(delta);
	    ed_repaint(TextLines - 1 - delta, TextLines - 2);
	else
	    /* just redraw the whole thing */
	    WindowTopLine := where;
	    ed_repaint(0, TextLines - 2);
	fi;
    fi;
corp;

/*
 * ed_fillWindow - draw the stuff for the current editing window.
 */

proc ed_fillWindow()void:

    ScrollBody := 0;
    ScrollPot := 0;
    ed_drawScroll();
    ed_repaint(0, TextLines - 2);
    ErrorShown := true;
    ed_clearError();
    ed_cursor();
    HelpShown := false;
corp;

/*
 * ed_scroll2 - second level of handling a scrolling action while editing.
 */

proc ed_scroll2(uint vertPot)void:
    register uint space, where;

    if HelpShown then
	ed_fillWindow();
    fi;
    if ErrorShown then
	ed_clearError();
    fi;
    if MaxLineNumber > TextLines - 1 then
	/* scrolling is possible */
	space := MaxLineNumber - TextLines + 1;
	where := make(vertPot, ulong) * space / 0xffff;
	if where > space then
	    where := space;
	fi;
	if where ~= WindowTopLine then
	    ed_cursor();
	    ed_pack();
	    ed_scrollWindow(where);
	    ed_cursor();
	fi;
    fi;
corp;

/*
 * ed_scrollLines - a key-press scroll action.
 */

proc ed_scrollLines(bool scrollUp, scrollLines)void:
    uint lines, limit;

    if HelpShown then
	ed_fillWindow();
    fi;
    if ErrorShown then
	ed_clearError();
    fi;
    if MaxLineNumber > TextLines - 1 then
	/* buffer more than fills the window */
	ed_cursor();
	ed_pack();
	lines := if scrollLines then TextLines - 2 else 1 fi;
	if scrollUp then
	    /* scroll back towards the beginning of the buffer */
	    if lines > WindowTopLine then
		lines := WindowTopLine;
	    fi;
	    if lines > 0 then
		ed_scrollWindow(WindowTopLine - lines);
		ed_drawScroll();
	    fi;
	else
	    /* scroll forward towards the end of the buffer */
	    limit := MaxLineNumber - WindowTopLine - TextLines + 1;
	    if lines > limit then
		lines := limit;
	    fi;
	    if lines > 0 then
		ed_scrollWindow(WindowTopLine + lines);
		ed_drawScroll();
	    fi;
	fi;
	ed_cursor();
    fi;
corp;

/*
 * ed_errorStart - start of an error from parsing the buffer.
 */

proc ed_errorStart(uint column)void:
    register *RastPort_t rp;
    register uint delta;

    /* window to a valid place in the buffer for the current line. */
    delta := WindowTopLine + TextLines - 2;
    if CurrentLineNumber < WindowTopLine then
	delta := WindowTopLine - CurrentLineNumber;
	if delta < 5 then
	    ed_scrollWindow(WindowTopLine - delta);
	else
	    ed_scrollWindow(CurrentLineNumber);
	fi;
    elif CurrentLineNumber > delta then
	delta := CurrentLineNumber - delta;
	if delta < 5 then
	    ed_scrollWindow(WindowTopLine + delta);
	else
	    ed_scrollWindow(CurrentLineNumber);
	fi;
    fi;
    CurrentColumnNumber := column;
    /* display cursor at the error position (is not displayed currently) */
    ed_cursor();
    HadAnyError := true;
    HaveError := true;
    rp := TextRastPort;
    Move(rp, LeftBorder,
	TextInputBottom - (CHAR_HEIGHT - CHAR_BASELINE - 1));
    SetAPen(rp, ErasePen);
    SetBPen(rp, TextPen);
    ErrorShown := true;
corp;

/*
 * ed_errorString - some string for the parsing error.
 */

proc ed_errorString(register *char st)void:
    register *char p;
    register *RastPort_t rp;

    p := st;
    while p* ~= '\e' and p* ~= '\n' do
	p := p + sizeof(char);
    od;
    rp := TextRastPort;
    Text(rp, st, (p - st) / sizeof(char));
    if p* = '\n' then
	SetAPen(rp, TextPen);
	SetBPen(rp, ErasePen);
    fi;
corp;

/*
 * ed_scrollRegion - scroll a bottom region of the text window.
 */

proc ed_scrollRegion(int lines; uint firstLine)void:

    tx_doAnyScroll(0, lines * CHAR_HEIGHT,
		   LeftBorder, firstLine * CHAR_HEIGHT + TextTop,
		   RightEdge, TextInputBottom - CHAR_HEIGHT);
corp;

/*
 * ed_deleteLine - delete the line the cursor is on.
 */

proc ed_deleteLine()void:
    register *EditLine_t el;
    uint oldCLN;

    if MaxLineNumber = 1 then
	ed_showError("Can't delete only line");
    else
	ed_pack();
	MaxLineNumber := MaxLineNumber - 1;
	oldCLN := CurrentLineNumber;
	el := CurrentLine;
	if el = FirstLine then
	    FirstLine := el*.el_next;
	else
	    el*.el_prev*.el_next := el*.el_next;
	fi;
	if el*.el_next ~= nil then
	    el*.el_next*.el_prev := el*.el_prev;
	    CurrentLine := el*.el_next;
	else
	    CurrentLine := el*.el_prev;
	    CurrentLineNumber := CurrentLineNumber - 1;
	fi;
	FreeMem(el, LINE_SIZE + el*.el_length * sizeof(char));
	if CurrentLineNumber < WindowTopLine then
	    WindowTopLine := CurrentLineNumber;
	    ed_refreshLine();
	else
	    ed_scrollRegion(1, oldCLN - WindowTopLine);
	    if MaxLineNumber >= WindowTopLine + TextLines - 1 then
		ed_repaint(TextLines - 2, TextLines - 2);
	    fi;
	fi;
	ed_drawScroll();
    fi;
corp;

/*
 * ed_joinLines - join the current line with the next line.
 */

proc ed_joinLines()void:
    register *EditLine_t el;
    register uint l;

    if CurrentLineNumber = MaxLineNumber - 1 then
	ed_showError("No line to join with");
    else
	ignore ed_unpack();
	el := CurrentLine*.el_next;
	l := el*.el_length;
	if l + BufferTotal > MAX_EDIT_COLUMN then
	    ed_showError("Joined line would be too long");
	else
	    BlockCopy(&EditBuffer[BufferTotal], &el*.el_text[0], l);
	    BufferTotal := BufferTotal + l;
	    CurrentLine*.el_next := el*.el_next;
	    if el*.el_next ~= nil then
		el*.el_next*.el_prev := CurrentLine;
	    fi;
	    FreeMem(el, LINE_SIZE + l * sizeof(char));
	    MaxLineNumber := MaxLineNumber - 1;
	    ed_refreshLine();
	    l := WindowTopLine + TextLines - 2;
	    if CurrentLineNumber ~= l then
		ed_scrollRegion(1, CurrentLineNumber - WindowTopLine + 1);
		if MaxLineNumber > l then
		    ed_repaint(TextLines - 2, TextLines - 2);
		fi;
	    fi;
	    ed_drawScroll();
	fi;
    fi;
corp;

/*
 * ed_upOneLine - move the cursor up one line if possible.
 */

proc ed_upOneLine()void:

    if CurrentLineNumber > 0 then
	ed_pack();
	if CurrentLineNumber = WindowTopLine then
	    ed_scrollWindow(WindowTopLine - 1);
	    ed_drawScroll();
	fi;
	CurrentLine := CurrentLine*.el_prev;
	CurrentLineNumber := CurrentLineNumber - 1;
    else
	ed_showError("Beginning of buffer");
    fi;
corp;

/*
 * ed_downOneLine - move the cursor down one line if possible.
 */

proc ed_downOneLine()void:

    if CurrentLineNumber < MaxLineNumber - 1 then
	ed_pack();
	if CurrentLineNumber = WindowTopLine + TextLines - 2 then
	    ed_scrollWindow(WindowTopLine + 1);
	    ed_drawScroll();
	fi;
	CurrentLine := CurrentLine*.el_next;
	CurrentLineNumber := CurrentLineNumber + 1;
    else
	ed_showError("End of buffer");
    fi;
corp;

/*
 * ed_carriageReturn - process a carriage return key.
 */

proc ed_carriageReturn()void:
    register *EditLine_t el;
    register *char p;
    register uint l;

    /* Just insert a new line, and pre-indent it to the level of the non-blank
       line at or before the one the cursor was just on */
    ed_pack();
    el := CurrentLine;
    l := 0;
    while
	if el*.el_prev = nil then
	    false
	elif el*.el_length = 0 then
	    true
	else
	    l := el*.el_length;
	    p := &el*.el_text[0];
	    while l ~= 0 and p* = ' ' do
		l := l - 1;
		p := p + sizeof(char);
	    od;
	    l = 0
	fi
    do
	el := el*.el_prev;
    od;
    if l ~= 0 then
	l := el*.el_length - l;
    fi;
    el := mudAlloc(LINE_SIZE + l * sizeof(char), 0x0);
    if el = nil then
	ed_noMem();
    else
	el*.el_prev := CurrentLine;
	el*.el_next := CurrentLine*.el_next;
	if el*.el_next ~= nil then
	    el*.el_next*.el_prev := el;
	fi;
	CurrentLine*.el_next := el;
	el*.el_length := l;
	CurrentColumnNumber := l;
	p := &el*.el_text[0];
	while l ~= 0 do
	    l := l - 1;
	    p* := ' ';
	    p := p + sizeof(char);
	od;
	CurrentLine := el;
	CurrentLineNumber := CurrentLineNumber + 1;
	MaxLineNumber := MaxLineNumber + 1;
	if CurrentLineNumber = WindowTopLine + TextLines - 1 then
	    ed_scrollWindow(WindowTopLine + 1);
	    ed_refreshLine();
	else
	    ed_scrollRegion(-1, CurrentLineNumber - WindowTopLine);
	fi;
	ed_drawScroll();
    fi;
corp;

/*
 * ed_splitLine - split the current line at the current column.
 */

proc ed_splitLine()void:
    register *EditLine_t el;
    register uint l;

    ignore ed_unpack();
    l := BufferTotal - CurrentColumnNumber;
    el := mudAlloc(LINE_SIZE + l * sizeof(char), 0x0);
    if el = nil then
	ed_noMem();
    else
	el*.el_prev := CurrentLine;
	el*.el_next := CurrentLine*.el_next;
	CurrentLine*.el_next := el;
	if el*.el_next ~= nil then
	    el*.el_next*.el_prev := el;
	fi;
	BlockCopy(&el*.el_text[0], &EditBuffer[CurrentColumnNumber], l);
	el*.el_length := l;
	BufferTotal := CurrentColumnNumber;
	MaxLineNumber := MaxLineNumber + 1;
	l := CurrentLineNumber - WindowTopLine;
	ed_scrollRegion(-1, l);
	ed_refreshLine();
	l := l + 1;
	if l < TextLines - 1 then
	    ed_repaint(l, l);
	fi;
	ed_drawScroll();
    fi;
corp;

/*
 * ed_cursorToTop - move cursor to top of text window.
 */

proc ed_cursorToTop()void:
    register uint delta;

    ed_pack();
    delta := CurrentLineNumber - WindowTopLine;
    while delta ~= 0 do
	delta := delta - 1;
	CurrentLine := CurrentLine*.el_prev;
    od;
    CurrentLineNumber := WindowTopLine;
corp;

/*
 * ed_cursorToBottom - move cursor to bottom of text window.
 */

proc ed_cursorToBottom()void:
    register uint delta;

    ed_pack();
    delta := WindowTopLine + TextLines - 2 - CurrentLineNumber;
    while delta ~= 0 and CurrentLine*.el_next ~= nil do
	delta := delta - 1;
	CurrentLine := CurrentLine*.el_next;
	CurrentLineNumber := CurrentLineNumber + 1;
    od;
corp;

/*
 * ed_hide - hide the editing stuff, and go back to normal view.
 */

proc ed_hide()void:

    /* This is a bit dangerous, but we need to do it for when the user hides
       the edit window, which is done by the event handler calling this
       routine directly. In the other situations where this is called, there
       may not be a valid buffer, but 'Unpacked' should be false. */
    ed_pack();
    Editing := false;
    if Mode = md_oneWindow or Mode = md_twoWindow then
	tx_redraw(true);
    else
	doPictureMenu();
	if PictureShown then
	    doShowPicture();
	fi;
	if OldStyle or not PictureShown then
	    tx_redraw(true);
	fi;
    fi;
corp;

/*
 * ed_showHelp - show a help screen.
 */

proc ed_showHelp()void:
    *char HELP_TEXT =
"Control:\n"
"    ^C - exit with no changes           ^G - Go to next error\n"
"    ^Q - save changes, [parse] and Quit ^R - Reject further errors\n"
"    ALT - up & down arrow - scroll by one line\n"
"    SHIFT-ALT - up & down arrow - scroll by one windowfull\n"
"\n"
"Cursor Motion:\n"
"    ^A - beginning of line              ^B - Back one column\n"
"    ^E - End of line                    ^F - Forward one column\n"
"    ^I/TAB - forward to next tabstop    ^L - beginning of next Line\n"
"    ^N - Next line                      ^P - Previous line\n"
"    ^T - cursor to Top of window        ^Z - cursor to bottom of window\n"
"    arrow keys and shifted arrow keys also operate\n"
"\n"
"Editing:\n"
"    ^D - Delete current line            ^H/BS - delete previous character\n"
"    ^J - Join this line with next       ^K - Kill from cursor to line end\n"
"    ^M/CR - insert new line             ^S - Split line at current column\n"
"    ^U - delete from cursor to beginning of line\n"
"    ^W - delete previous Word           ^X - clear line\n"
"    ^Y - undo changes to line           DEL - DELete current character\n"
"\n"
"                      PRESS ANY KEY TO CONTINUE";
    register *RastPort_t rp;
    register *char p, q;
    register ulong i, len;
    bool tooSmall;

    tooSmall := TextLines < EDIT_HELP_HEIGHT or TextColumns < EDIT_HELP_WIDTH;
    if not tooSmall or HelpShown then
	rp := TextRastPort;
	SetAPen(rp, ErasePen);
	RectFill(rp, LeftBorder, TextTop, RightEdge, TextInputBottom);
	SetAPen(rp, TextPen);
	if tooSmall then
	    /* Doing this rather than using ed_showError is in fact easier,
	       since we might already be set up to be showing HELP when the
	       window is resized, and this is called - that leaves a mess! */
	    p := "Window too small for HELP\n\nPress any key:";
	else
	    p := HELP_TEXT;
	fi;
	i := 0;
	while p* ~= '\e' do
	    q := p;
	    while p* ~= '\e' and p* ~= '\n' do
		p := p + sizeof(char);
	    od;
	    Move(rp, LeftBorder, TextTop + CHAR_BASELINE + i * CHAR_HEIGHT);
	    len := p - q;
	    if len > TextColumns then
		len := TextColumns;
	    fi;
	    Text(rp, q, len);
	    i := i + 1;
	    if p* = '\n' then
		p := p + sizeof(char);
	    fi;
	od;
	HelpShown := true;
    else
	ed_showError("Window too small for HELP");
	ed_cursor();
    fi;
corp;

/*
 * ed_refresh - redraw the full contents of the edit window.
 */

proc ed_refresh()void:

    ScrollBody := 0;
    ScrollPot := 0;
    ed_drawScroll();
    if HelpShown then
	ed_showHelp();
    else
	ed_repaint(0, TextLines - 2);
	if Unpacked then
	    ed_refreshLine();
	fi;
	if ErrorShown then
	    ed_clearError();
	    ed_showError(&ErrorMessage[0]);
	else
	    ed_clearError();	/* clears 'ErrorShown' */
	fi;
	ed_cursor();
    fi;
corp;

/*
 * ed_redraw - called on NEWSIZE event - when window sizing has happened,
 *	and also below, in 'ed_show'.
 */

proc ed_redraw()void:

    ignore RemoveGadget(TextWindow, ScrollGadget);
    tx_setScrollStuff(true);
    ignore AddGadget(TextWindow, ScrollGadget, 0);
    RefreshGadgets(ScrollGadget, TextWindow, nil);
    /* Why do we do an ed_refresh here, instead of relying on the REFRESH
       event to do it? The reason is that we need the whole window
       redrawing with edit data, not just the part that Intuition
       thinks needs redrawing. This is why we have the icky 'SkipRefresh'
       flag in Intuition.d, to avoid an extra redraw of the part that
       Intuition thinks needs redrawing. */
    ed_refresh();
corp;

/*
 * ed_show - hide the normal stuff, and go to edit window.
 */

proc ed_show()void:

    if Mode = md_oneWindow or Mode = md_twoWindow then
	ed_refresh();
	Editing := true;
    else
	if PictureShown then
	    doHidePicture();
	    if OldStyle then
		ed_redraw();	    /* change height on scroll bar */
		tx_drawBorders(TextFullHeight);
	    fi;
	else
	    ed_refresh();
	fi;
	Editing := true;
	doPictureMenu();
    fi;
corp;

/*
 * ed_scrollInLine - utility to scroll left/right in given line.
 */

proc ed_scrollInLine(int delta; uint x, y)void:

    tx_doAnyScroll(delta * CHAR_WIDTH, 0, x, y,
	RightEdge, y + (CHAR_HEIGHT - 1));
corp;

/*
 * ed_cookChar - a normal keypress - handle it.
 */

proc ed_cookChar(char ch)void:
    char
	CNTL_A = '\(0x01)',
	CNTL_B = '\(0x02)',
	CNTL_C = '\(0x03)',
	CNTL_D = '\(0x04)',
	CNTL_E = '\(0x05)',
	CNTL_F = '\(0x06)',
	CNTL_G = '\(0x07)',
	CNTL_H = '\(0x08)',
	CNTL_I = '\(0x09)',
	CNTL_J = '\(0x0a)',
	CNTL_K = '\(0x0b)',
	CNTL_L = '\(0x0c)',
	CNTL_M = '\(0x0d)',
	CNTL_N = '\(0x0e)',
	CNTL_P = '\(0x10)',
	CNTL_Q = '\(0x11)',
	CNTL_R = '\(0x12)',
	CNTL_S = '\(0x13)',
	CNTL_T = '\(0x14)',
	CNTL_U = '\(0x15)',
	CNTL_W = '\(0x17)',
	CNTL_X = '\(0x18)',
	CNTL_Y = '\(0x19)',
	CNTL_Z = '\(0x1a)',
	DEL = '\(0x7f)';
    register *RastPort_t rp;
    register uint x, y, t;

    if HelpShown then
	ed_fillWindow();
	return;
    fi;
    if ErrorShown then
	ed_clearError();
    fi;
    rp := TextRastPort;
    y := (CurrentLineNumber - WindowTopLine) * CHAR_HEIGHT + TextTop;
    ed_cursor();
    case ch
    incase CNTL_A:
	/* cursor to beginning of line */
	CurrentColumnNumber := 0;
    incase CNTL_B:
	/* cursor back one column */
	if CurrentColumnNumber ~= 0 then
	    CurrentColumnNumber := CurrentColumnNumber - 1;
	fi;
    incase CNTL_C:
	/* abort the edit session */
	ed_pack();
	if EditParsing then
	    EditAborting := true;
	    editNextError(true);
	else
	    ed_freeBuffer();
	    editEnd(true);
	    ed_hide();
	    EditActive := false;
	    doEditWinState(false);
	fi;
    incase CNTL_D:
	/* delete the current line */
	ed_deleteLine();
    incase CNTL_E:
	/* Cursor to end of line */
	if Unpacked then
	    CurrentColumnNumber := BufferTotal;
	else
	    CurrentColumnNumber := CurrentLine*.el_length;
	fi;
    incase CNTL_F:
	/* cursor forward one column */
	if Unpacked then
	    t := BufferTotal;
	else
	    t := CurrentLine*.el_length;
	fi;
	if CurrentColumnNumber < t then
	    CurrentColumnNumber := CurrentColumnNumber + 1;
	fi;
    incase CNTL_G:
	/* go to next error */
	if HaveError then
	    ed_pack();
	    HaveError := false;
	    editNextError(false);
	    /* do not put cursor back on current line/column, since we may
	       be changing them after returning to 'editErrorEnd' */
	    return;
	else
	    ed_showError("No error to go to");
	fi;
    incase CNTL_H:
	/* delete character before cursor */
	x := ed_unpack();
	if CurrentColumnNumber ~= 0 then
	    if CurrentColumnNumber <= TextColumns then
		ed_scrollInLine(+ 1, x - CHAR_WIDTH, y);
	    fi;
	    if BufferTotal - 1 >= TextColumns and
		CurrentColumnNumber <= TextColumns
	    then
		Move(rp, RightEdge - CHAR_WIDTH, y + CHAR_BASELINE);
		Text(rp, &EditBuffer[TextColumns], 1);
	    fi;
	    BlockCopy(&EditBuffer[CurrentColumnNumber - 1],
		&EditBuffer[CurrentColumnNumber],
		(BufferTotal - CurrentColumnNumber) * sizeof(char));
	    BufferTotal := BufferTotal - 1;
	    CurrentColumnNumber := CurrentColumnNumber - 1;
	else
	    ed_showError("Nothing to backspace over");
	fi;
    incase CNTL_I:
	/* move forward to next 8 column boundary */
	if Unpacked then
	    t := BufferTotal;
	else
	    t := CurrentLine*.el_length;
	fi;
	CurrentColumnNumber := (CurrentColumnNumber + 8) & (-8);
	if CurrentColumnNumber > t then
	    CurrentColumnNumber := t;
	fi;
    incase CNTL_J:
	/* join this line with the next */
	ed_joinLines();
    incase CNTL_K:
	/* delete from cursor to end of line */
	x := ed_unpack();
	if CurrentColumnNumber ~= BufferTotal then
	    BufferTotal := CurrentColumnNumber;
	    if CurrentColumnNumber < TextColumns then
		SetAPen(rp, ErasePen);
		RectFill(rp, x, y, RightEdge, y + (CHAR_HEIGHT - 1));
		SetAPen(rp, TextPen);
	    fi;
	else
	    ed_showError("Nothing past here to delete");
	fi;
    incase CNTL_L:
	/* cursor to beginning of next line */
	CurrentColumnNumber := 0;
	ed_downOneLine();
    incase CNTL_M:
	/* Carriage return - insert a new line after the current one. Indent
	   it to the position of the first character on the current line. */
	ed_carriageReturn();
    incase CNTL_N:
	/* move to next line */
	ed_downOneLine();
    incase CNTL_P:
	/* move to previous line */
	ed_upOneLine();
    incase CNTL_Q:
	/* leave the editor */
	ed_pack();
	if HaveError then
	    /* hit when displaying an error - want to tell the parser to skip
	       errors after this one and end up calling 'editEnd' again. To
	       do that, we have to quickly exit here, and loop in the main
	       call to 'editEnd' while this situation has ocurred. */
	    editNextError(true);
	    /* we do not want the cursor put back, since then it would still
	       be there when we return from the 'editEnd' call below */
	    return;
	else
	    /* a normal, top-level hit of this key */
	    while
		CurrentLine := FirstLine;
		CurrentLineNumber := 0;
		CurrentColumnNumber := 0;
		FirstGetLine := true;
		EditParsing := true;
		editEnd(false);
		EditParsing := false;
		if EditAborting then
		    HaveError := false;
		    HadAnyError := false;
		    /* tell the server we are done editing */
		    editEnd(true);
		fi;
		HaveError
	    do
		/* We get in here if the user types a CNTL-Q from in the
		   middle of parsing from a CNTL-Q when paused at an error. */
		HaveError := false;
		HadAnyError := false;
	    od;
	    if not HadAnyError then
		ed_freeBuffer();
		ed_hide();
		EditActive := false;
		doEditWinState(false);
	    else
		/* We have just finished the very lengthy call into editEnd
		   which gave us errors and would have recursed into this
		   routine, and we want to leave the user editing. */
		HadAnyError := false;
		CurrentLine := FirstLine;
		CurrentLineNumber := 0;
		CurrentColumnNumber := 0;
		ed_scrollWindow(0);
	    fi;
	fi;
    incase CNTL_R:
	/* reject all further error messages and just continue editing */
	ed_pack();
	if EditParsing then
	    editNextError(true);
	    /* make the ^Q code just go back to editing */
	    HaveError := false;
	    /* do not re-display cursor */
	    return;
	else
	    ed_showError("Not currently parsing");
	fi;
    incase CNTL_S:
	/* split this line at the current column */
	ed_splitLine();
    incase CNTL_T:
	/* move cursor to top of window */
	ed_cursorToTop();
    incase CNTL_U:
	/* delete from previous to beginning of line */
	ignore ed_unpack();
	if CurrentColumnNumber ~= 0 then
	    BlockCopy(&EditBuffer[0], &EditBuffer[CurrentColumnNumber],
		      BufferTotal - CurrentColumnNumber);
	    BufferTotal := BufferTotal - CurrentColumnNumber;
	    CurrentColumnNumber := 0;
	    ed_refreshLine();
	fi;
    incase CNTL_W:
	/* delete the word to the left of the cursor */
	ignore ed_unpack();
	t := CurrentColumnNumber;
	if t ~= 0 then
	    t := t - 1;
	    if t ~= 0 and EditBuffer[t] = ' ' then
		t := t - 1;
	    fi;
	    while t ~= 0 and EditBuffer[t] ~= ' ' do
		t := t - 1;
	    od;
	    if EditBuffer[t] = ' ' then
		t := t + 1;
	    fi;
	    BlockCopy(&EditBuffer[t], &EditBuffer[CurrentColumnNumber],
		      BufferTotal - CurrentColumnNumber);
	    BufferTotal := BufferTotal - (CurrentColumnNumber - t);
	    CurrentColumnNumber := t;
	    ed_refreshLine();
	fi;
    incase CNTL_X:
	/* clear whole line */
	ignore ed_unpack();
	SetAPen(rp, ErasePen);
	RectFill(rp, LeftBorder, y, RightEdge, y + (CHAR_HEIGHT - 1));
	SetAPen(rp, TextPen);
	CurrentColumnNumber := 0;
	BufferTotal := 0;
    incase CNTL_Y:
	/* put the line back to the way it was */
	if Unpacked then
	    Unpacked := false;
	    ed_refreshLine();
	fi;
    incase CNTL_Z:
	/* move cursor to bottom of window */
	ed_cursorToBottom();
    incase DEL:
	/* delete the current character */
	x := ed_unpack();
	if CurrentColumnNumber ~= BufferTotal then
	    if CurrentColumnNumber < TextColumns then
		ed_scrollInLine(+ 1, x, y);
	    fi;
	    if BufferTotal - 1 >= TextColumns and
		CurrentColumnNumber < TextColumns
	    then
		Move(rp, RightEdge - CHAR_WIDTH, y + CHAR_BASELINE);
		Text(rp, &EditBuffer[TextColumns], 1);
	    fi;
	    BlockCopy(&EditBuffer[CurrentColumnNumber],
		&EditBuffer[CurrentColumnNumber + 1],
		(BufferTotal - CurrentColumnNumber - 1) * sizeof(char));
	    BufferTotal := BufferTotal - 1;
	fi;
    default:
	/* regular character - insert into line */
	x := ed_unpack();
	if (ch >= IECODE_ASCII_FIRST + '\e' and
	    ch <= IECODE_ASCII_LAST + '\e' or
	    ch >= IECODE_LATIN1_FIRST + '\e' and
	    ch <= IECODE_LATIN1_LAST + '\e')
	then
	    if BufferTotal = MAX_EDIT_COLUMN then
		ed_showError("Line too long");
	    else
		if CurrentColumnNumber < TextColumns then
		    if CurrentColumnNumber ~= BufferTotal then
			ed_scrollInLine(- 1, x, y);
		    fi;
		    Move(rp, x, y + CHAR_BASELINE);
		    Text(rp, &ch, 1);
		fi;
		if CurrentColumnNumber ~= BufferTotal then
		    BlockCopyB(&EditBuffer[BufferTotal],
			&EditBuffer[BufferTotal - 1],
			BufferTotal - CurrentColumnNumber);
		fi;
		EditBuffer[CurrentColumnNumber] := ch;
		BufferTotal := BufferTotal + 1;
		CurrentColumnNumber := CurrentColumnNumber + 1;
	    fi;
	else
	    ed_notBound();
	fi;
    esac;
    if Editing then
	ed_cursor();
    fi;
corp;

/*
 * ed_specialKey - a special (editing) keypress - handle it.
 */

proc ed_specialKey(uint ch)void:
    register uint delta;

    if HelpShown then
	ed_fillWindow();
	return;
    fi;
    if ErrorShown then
	ed_clearError();
    fi;
    ed_cursor();
    case ch
    incase KEY_LEFT:
	if CurrentColumnNumber ~= 0 then
	    CurrentColumnNumber := CurrentColumnNumber - 1;
	fi;
    incase KEY_RIGHT:
	if Unpacked then
	    delta := BufferTotal;
	else
	    delta := CurrentLine*.el_length;
	fi;
	if CurrentColumnNumber < delta then
	    CurrentColumnNumber := CurrentColumnNumber + 1;
	fi;
    incase KEY_UP:
	ed_upOneLine();
    incase KEY_DOWN:
	ed_downOneLine();
    incase KEY_SHIFT | KEY_LEFT:
	CurrentColumnNumber := 0;
    incase KEY_SHIFT | KEY_RIGHT:
	if Unpacked then
	    CurrentColumnNumber := BufferTotal;
	else
	    CurrentColumnNumber := CurrentLine*.el_length;
	fi;
    incase KEY_SHIFT | KEY_UP:
	ed_cursorToTop();
    incase KEY_SHIFT | KEY_DOWN:
	ed_cursorToBottom();
    incase KEY_HELP:
	ed_pack();
	ed_showHelp();
	/* do not draw cursor */
	return;
    default:
	ed_notBound();
    esac;
    ed_cursor();
corp;

/*
 * ed_start - called at the start of each edit session.
 */

proc ed_start()void:

    FirstLine := nil;
    CurrentLine := nil;
    CurrentColumnNumber := 0;
    MaxLineNumber := 0;
corp;

/*
 * ed_text - called when some text has arrived from the server. It will
 *	simply be a bunch of characters. It will end in a newline.
 */

proc ed_text(register *char data; register uint len)bool:
    register *EditLine_t el;
    register *char q;
    register uint lineLen;
    *char endSave;

    while len ~= 0 do
	endSave := data;
	lineLen := 0;
	while len ~= 0 and data* ~= '\n' and data* ~= '\e' and
	    lineLen ~= MAX_EDIT_COLUMN
	do
	    data := data + sizeof(char);
	    len := len - 1;
	    lineLen := lineLen + 1;
	od;
	el := mudAlloc(LINE_SIZE + lineLen * sizeof(char), 0x0);
	if el = nil then
	    while
		el := FirstLine;
		el ~= nil
	    do
		FirstLine := el*.el_next;
		FreeMem(el, LINE_SIZE + el*.el_length * sizeof(char));
	    od;
	    return(false);
	fi;
	el*.el_prev := CurrentLine;
	if CurrentLine ~= nil then
	    CurrentLine*.el_next := el;
	fi;
	el*.el_next := nil;
	CurrentLine := el;
	if FirstLine = nil then
	    FirstLine := el;
	fi;
	el*.el_length := lineLen;
	MaxLineNumber := MaxLineNumber + 1;
	q := &el*.el_text[0];
	data := endSave;
	while lineLen ~= 0 do
	    lineLen := lineLen - 1;
	    q* := data*;
	    q := q + sizeof(char);
	    data := data + sizeof(char);
	od;
	if len ~= 0 and (data* = '\n' or data* = '\e') then
	    data := data + sizeof(char);	/* skip the newline/\e */
	    len := len - 1;
	    if len = 1 and data* = '\e' then
		len := 0;
	    fi;
	fi;
    od;
    true
corp;

/*
 * ed_bufferString - part of a line from cooking an edited string.
 */

proc ed_bufferString(register *char buf; register uint len)bool:
    register *char p;
    bool overFlow;

    overFlow := false;
    if CurrentColumnNumber + len > MAX_EDIT_COLUMN then
	len := MAX_EDIT_COLUMN - CurrentColumnNumber;
	overFlow := true;
    fi;
    p := &EditBuffer[CurrentColumnNumber];
    CurrentColumnNumber := CurrentColumnNumber + len;
    while len ~= 0 do
	len := len - 1;
	p* := buf*;
	p := p + sizeof(char);
	buf := buf + sizeof(char);
    od;
    buf := buf - sizeof(char);
    if buf* = '\n' or overFlow then
	len := CurrentColumnNumber;
	CurrentColumnNumber := 0;
	ed_text(&EditBuffer[0], len)
    else
	true
    fi
corp;

/*
 * ed_stringText - cook a string to be edited.
 */

proc ed_stringText(*char buf; uint len)bool:

    editCook(buf, len, MaxEditColumn, ed_bufferString)
corp;

/*
 * ed_go - actually start editing.
 */

proc ed_go(*char prompt)void:
    register *EditLine_t el;

    if FirstLine = nil then
	el := mudAlloc(LINE_SIZE, 0x0);
	el*.el_next := nil;
	el*.el_prev := nil;
	el*.el_length := 0;
	FirstLine := el;
    fi;
    CurrentLine := FirstLine;
    CurrentLineNumber := 0;
    CurrentColumnNumber := 0;
    WindowTopLine := 0;
    HaveError := false;
    HadAnyError := false;
    EditParsing := false;
    EditAborting := false;
    tx_toggleCursor();
    EditActive := true;
    doEditWinState(true);
    Unpacked := false;
    HelpShown := false;
    ed_show();
    ed_showError(prompt);
corp;

/*
 * ed_getLine - get the next line from the edit buffer.
 */

proc ed_getLine(register *char buff; register uint maxLen; *uint pGotLen)bool:
    register *EditLine_t el;
    register *char p;
    register uint len, len2;

    el := CurrentLine;
    if el = nil then
	return(false);
    fi;
    if FirstGetLine then
	FirstGetLine := false;
    else
	el := el*.el_next;
	if el = nil then
	    return(false);
	fi;
	CurrentLine := el;
	CurrentLineNumber := CurrentLineNumber + 1;
    fi;
    CurrentColumnNumber := 0;
    len := el*.el_length;
    p := &el*.el_text[0];
    if len > maxLen then
	len := maxLen;
    fi;
    len2 := len;
    while len ~= 0 do
	len := len - 1;
	buff* := p*;
	buff := buff + sizeof(char);
	p := p + sizeof(char);
    od;
    /* add a trailing space - this helps cursor positioning when the token for
       the error is at the end of a line */
    if len2 < maxLen then
	len2 := len2 + 1;
	buff* := ' ';
    fi;
    pGotLen* := len2;
    true
corp;

/*
 * ed_setIsRaw - allow main.c to control raw/nonraw in ed_normalize.
 */

proc ed_setIsRaw(bool isRaw)void:

    IsRaw := isRaw;
corp;

/*
 * ed_normalize - editing a string - take the supplied bufferful of stuff and
 *	add it to an allocated buffer in a normalized form.
 */

proc ed_normalize(register *char buf; uint bLen; bool first;
		  register *char allocBuf; uint allocLen;
		  *uint pExtraLen, pColumn)uint:
    register uint len, column, extraLen;
    register char ch;

    len := 0;
    extraLen := pExtraLen*;
    column := pColumn*;
    if not first and buf* > ' ' and allocLen ~= 0 then
	ch := (allocBuf - sizeof(char))*;
	if ch ~= ' ' and ch ~= '\n' then
	    if extraLen ~= 0 then
		extraLen := extraLen - 1;
		allocBuf* := ' ';
		allocBuf := allocBuf + sizeof(char);
		allocLen := allocLen - 1;
		len := len + 1;
		column := column + 1;
	    fi;
	fi;
    fi;
    while bLen ~= 0 and allocLen ~= 0 do
	bLen := bLen - 1;
	ch := buf*;
	if ch >= ' ' then
	    if ch = '\\' and bLen ~= 0 and not IsRaw then
		bLen := bLen - 1;
		buf := buf + sizeof(char);
		if buf* = 'n' then
		    allocBuf* := '\n';
		elif buf* = 't' then
		    allocBuf* := '\t';
		else
		    allocBuf* := buf*;
		fi;
		extraLen := extraLen + 1;
	    else
		allocBuf* := ch;
	    fi;
	    allocBuf := allocBuf + sizeof(char);
	    allocLen := allocLen - 1;
	    len := len + 1;
	    column := column + 1;
	elif ch = '\n' then
	    if IsRaw then
		allocBuf* := ch;
		allocBuf := allocBuf + sizeof(char);
		allocLen := allocLen - 1;
		len := len + 1;
		column := 0;
	    else
		if (not first or len ~= 0) and bLen ~= 0 then
		    ch := (allocBuf - sizeof(char))*;
		    if ch ~= ' ' and ch ~= '\n' then
			allocBuf* := ' ';
			allocBuf := allocBuf + sizeof(char);
			allocLen := allocLen - 1;
			len := len + 1;
			column := column + 1;
		    else
			extraLen := extraLen + 1;
		    fi;
		else
		    extraLen := extraLen + 1;
		fi;
	    fi;
	    first := false;
	elif ch = '\t' then
	    /* An editor can put tabs in the file to save file space on
	       a sequence of blanks. We have trouble here properly
	       expanding them again. We do what we can without a lot of
	       hassle, and at least will always put in at least one
	       space. I am sure someone will complain! */
	    extraLen := extraLen + 1;
	    while
		allocBuf* := ' ';
		allocBuf := allocBuf + sizeof(char);
		allocLen := allocLen - 1;
		len := len + 1;
		extraLen := extraLen - 1;
		column := column + 1;
		column & 7 ~= 0 and extraLen ~= 0
	    do
	    od;
	fi;
	buf := buf + sizeof(char);
    od;
    /* Want a newline added to the end of the line in raw mode, to match
       what an external editor will do. */
    if IsRaw and allocLen ~= 0 then
	if len = 0 or (allocBuf - sizeof(char))* ~= '\n' then
	    allocBuf* := '\n';
	    len := len + 1;
	    column := 0;
	fi;
    fi;
    pExtraLen* := extraLen;
    pColumn* := column;
    len
corp;

/*
 * ed_getString - return the whole edit buffer as a big string.
 */

proc ed_getString(**char pBuff; *uint pLen, pAllocLen)void:
    register *EditLine_t el;
    register *char buff;
    register uint allocLen, len, l;
    uint extraLen, column;
    bool first;

    el := FirstLine;
    allocLen := 0;
    while el ~= nil do
	allocLen := allocLen + el*.el_length;
	allocLen := allocLen + 1;
	el := el*.el_next;
    od;
    if allocLen > 3700 then
	allocLen := 3700;
    fi;
    extraLen := allocLen / 10;
    if extraLen < 50 then
	extraLen := 50;
    fi;
    allocLen := allocLen + extraLen;
    buff := mudAlloc(allocLen * sizeof(char), 0x0);
    pBuff* := buff;
    if buff = nil then
	return;
    fi;
    pAllocLen* := allocLen;
    len := 0;
    first := true;
    while
	el := FirstLine;
	el ~= nil
    do
	FirstLine := el*.el_next;
	column := 0;
	l := ed_normalize(&el*.el_text[0], el*.el_length, first,
			  buff, allocLen, &extraLen, &column);
	if l ~= 0 then
	    first := false;
	fi;
	len := len + l;
	buff := buff + l * sizeof(char);
	allocLen := allocLen - l;
	FreeMem(el, LINE_SIZE + el*.el_length * sizeof(char));
    od;
    pLen* := len;
corp;
