#drinc:exec/ports.g
#drinc:exec/io.g
#drinc:exec/memory.g
#drinc:graphics/gfx.g
#drinc:graphics/view.g
#drinc:graphics/rastport.g
#drinc:graphics/text.g

#drinc:util.g

#/Include/MUD.g
#/Include/Effects.g
#graphics.g
#globals.g
#text.g
#funcs.g
#iff.g
#soundFuncs.g

/*
 * Amiga MUD
 *
 * Copyright (c) 1997 by Chris Gray
 */

/*
 * effects.d - implement the various effects for the Amiga client.
 */

uint
    ICON_WIDTH = 16,
    ICON_HEIGHT = 16,
    MAX_ICONS = (LORES_GRAPHICS_WIDTH / ICON_WIDTH) *
		(LORES_GRAPHICS_HEIGHT / ICON_HEIGHT);

type
    IconData_t = [ICON_WIDTH * ICON_HEIGHT / 16] uint,

    IconCache_t = struct {
	*IconCache_t ic_next;
	*IconCache_t ic_prev;
	ulong ic_key;
	*IconData_t ic_data;
    },

    IconSlot_t = struct {
	*IconCache_t is_icon;
	BitMap_t is_save;
	bool is_alloced;
    },

    FileType_t = enum {
	ft_backGround,
	ft_image,
	ft_brush,
	ft_sound,
	ft_music,
	ft_instrument
    },

    FileCache_t = struct {
	*FileCache_t fc_next;
	*FileCache_t fc_prev;
	*char fc_filePath;
	arbptr fc_ptr;
	uint fc_lockCount;
	FileType_t fc_type;
    },

    Region_t = struct {
	*Region_t rg_next;
	*char rg_text;
	int rg_x1, rg_y1, rg_x2, rg_y2;
	uint rg_length;
	uint rg_number;
    },

    EffectCache_t = struct {
	*EffectCache_t ec_next;
	*EffectCache_t ec_prev;
	ulong ec_key;
	*byte ec_data;
	*ulong ec_calls;
	uint ec_length;
	byte ec_callsCount;
	byte ec_useCount;
    },

    TileList_t = struct {
	*TileList_t tl_next;
	BitMap_t tl_bitMap;
	uint tl_id;
	byte tl_width;
	byte tl_height;
    };

IconData_t DefaultCharacterIcon := (
    0b0000000000000000,
    0b0000000000000000,
    0b0000001111000000,
    0b0000110000110000,
    0b0001000000001000,
    0b0001000000001000,
    0b0010011001100100,
    0b0010000000000100,
    0b0010000000000100,
    0b0010010000100100,
    0b0001001111001000,
    0b0001000000001000,
    0b0000110000110000,
    0b0000001111000000,
    0b0000000000000000,
    0b0000000000000000
);

IconData_t DefaultMonsterIcon := (
    0b0000111111100000,
    0b0111100000111100,
    0b0100000000000100,
    0b0101110001110100,
    0b1101010001010110,
    0b1000110001100010,
    0b1010000000001010,
    0b1001001110010010,
    0b1000101010100010,
    0b1000000000000010,
    0b1000010001000010,
    0b1100110101100110,
    0b0110011111001100,
    0b0011000000011000,
    0b0001111111110000,
    0b0000000000000000
);

IconData_t CurrentCursor := _chip (
    0b1000000000000001,
    0b0100000000000010,
    0b0010000000000100,
    0b0001000000001000,
    0b0000100000010000,
    0b0000010000100000,
    0b0000001001000000,
    0b0000000110000000,
    0b0000000110000000,
    0b0000001001000000,
    0b0000010000100000,
    0b0000100000010000,
    0b0001000000001000,
    0b0010000000000100,
    0b0100000000000010,
    0b1000000000000001,
);

bool FindError, Ignoring, Filling;

[MAX_ICONS] IconSlot_t Icons;
*IconCache_t IconCache;

*FileCache_t FileCache, FileCacheTail, DefaultImage;
*char LastFileName;
FileType_t LastFileType;

*Region_t Regions, Buttons;

*EffectCache_t EffectCache, EffectCacheTail;

*TileList_t TileList;

*char ButtonText;
int ButtonX1, ButtonY1, ButtonX2, ButtonY2;
uint ButtonNumber, ButtonLength, ColourMax;
[7] uint ButtonColours;
bool ButtonActive;

int SavedX, SavedY;
ushort SavedPen, IconPen, CursorPen;
BitMap_t CursorSave;
uint CursorX, CursorY, CursorWidth, CursorHeight;
bool CursorShown;

[GRAPHICS_COLOURS] uint ColourMapping;

/*
 * freeString - handy utility.
 */

proc freeString(register *char p)void:
    register *char q;

    q := p;
    while q* ~= '\e' do
	q := q + sizeof(char);
    od;
    q := q + sizeof(char);
    FreeMem(p, (q - p) / sizeof(char));
corp;

/*
 * freeFileCache - free one file cache entry.
 */

proc freeFileCache(register *FileCache_t fc)void:
    register *char p;

    case fc*.fc_type
    incase ft_backGround:
    incase ft_image:
    incase ft_brush:
	iffFreeILBM(fc*.fc_ptr);
    incase ft_sound:
    incase ft_instrument:
	iffFree8SVX(fc*.fc_ptr);
    incase ft_music:
	iffFreeSMUS(fc*.fc_ptr);
    esac;
    freeString(fc*.fc_filePath);
    FreeMem(fc, sizeof(FileCache_t));
corp;

/*
 * freeEffect - free one effect cache entry.
 */

proc freeEffect(register *EffectCache_t ec)void:
    register *EffectCache_t prev, next;
    register *EffectCache_t ec2 @ prev;
    register *ulong calls @ next;
    register ulong key;
    register byte callsCount;

    callsCount := ec*.ec_callsCount;
    calls := ec*.ec_calls;
    while callsCount ~= 0 do
	callsCount := callsCount - 1;
	key := calls*;
	calls := calls + sizeof(ulong);
	ec2 := EffectCache;
	while ec2 ~= nil and ec2*.ec_key ~= key do
	    ec2 := ec2*.ec_next;
	od;
	if ec2 ~= nil then
	    ec2*.ec_useCount := ec2*.ec_useCount - 1;
	fi;
    od;
    callsCount := ec*.ec_callsCount;
    if callsCount ~= 0 then
	FreeMem(ec*.ec_calls, make(callsCount, ulong) * sizeof(ulong));
    fi;
    FreeMem(ec*.ec_data, ec*.ec_length);
    prev := ec*.ec_prev;
    next := ec*.ec_next;
    if prev ~= nil then
	prev*.ec_next := next;
    else
	EffectCache := next;
    fi;
    if next ~= nil then
	next*.ec_prev := prev;
    else
	EffectCacheTail := prev;
    fi;
    FreeMem(ec, sizeof(EffectCache_t));
corp;

proc freeAllEffects()void:

    while EffectCache ~= nil do
	freeEffect(EffectCache);
    od;
corp;

/*
 * clearButtons - remove all mouse-click buttons from list and display.
 */

proc clearButtons()void:
    register *Region_t rg;
    ushort savedPen;

    savedPen := GraphicsRastPort*.rp_FgPen;
    setGraphicsAPen(if MapColours then ColourMapping[0] else 0 fi);
    while
	rg := Buttons;
	rg ~= nil
    do
	Buttons := rg*.rg_next;
	RectFill(GraphicsRastPort, rg*.rg_x1, rg*.rg_y1, rg*.rg_x2, rg*.rg_y2);
	freeString(rg*.rg_text);
	FreeMem(rg, sizeof(Region_t));
    od;
    SetAPen(GraphicsRastPort, savedPen);
corp;

/*
 * clearRegions - remove all mouse-select regions from the list.
 */

proc clearRegions()void:
    register *Region_t rg;

    while
	rg := Regions;
	rg ~= nil
    do
	Regions := rg*.rg_next;
	FreeMem(rg, sizeof(Region_t));
    od;
corp;

/*
 * initColourMapping - set up our colour mapping table.
 */

proc initColourMapping()void:
    [GRAPHICS_COLOURS] uint colourTable;
    [MAX_COLOURS] uint currentColours;
    register uint i;

    getColourMap(&colourTable[0], true);
    getColourMap(&currentColours[0], false);
    iffCreateMapping(&ColourMapping[0], GRAPHICS_COLOURS, &colourTable[0],
		     ColourCount, &currentColours[0]);
    for i from 6 downto 0 do
	ButtonColours[i] := ColourMapping[ButtonColours[i]];
    od;
corp;

/*
 * effectsDefaults - set up the default values.
 */

proc effectsDefaults()void:

    FindError := false;
    Ignoring := false;
    Filling := false;
    /* See intuition.d, GraphicsColourTable, for which colours these are */
    ButtonColours[0] := 1;	/* background of letters */
    ButtonColours[1] := 9;	/* letters */
    ButtonColours[2] := 2;	/* outer border colour */
    ButtonColours[3] := 3;	/* middle border colour */
    ButtonColours[4] := 4;	/* inner border colour */
    ButtonColours[5] := 9;	/* background of highlighted letters */
    ButtonColours[6] := 1;	/* highlighted letters */
    ButtonActive := false;
    SavedX := GXOffset;
    SavedY := GYOffset;
    SavedPen := GraphicsRastPort*.rp_FgPen;
    CursorShown := false;
    LastFileName := nil;
    LastFileType := ft_backGround;
    if MapColours then
	initColourMapping();
	ColourMax := GRAPHICS_COLOURS;
	IconPen := ColourMapping[4];	/* WHITE */
	CursorPen := ColourMapping[6];	/* RED */
	setGraphicsAPen(ColourMapping[1]);
    else
	ColourMax := ColourCount;
	IconPen := 4;			/* WHITE */
	CursorPen := 6; 		/* RED */
	setGraphicsAPen(1);
    fi;
corp;

/*
 * effectsTerm - close down the stuff here.
 */

proc effectsTerm(bool partial)void:
    register *IconSlot_t is;
    register *IconCache_t ic @ is;
    register *FileCache_t fc @ is;
    register *TileList_t tl @ is;
    register uint i;

    if LastFileName ~= nil then
	freeString(LastFileName);
    fi;
    is := &Icons[0];
    for i from MAX_ICONS - 1 downto 0 do
	if is*.is_alloced then
	    freeBitMap(&is*.is_save, ICON_WIDTH, ICON_HEIGHT);
	    is*.is_alloced := false;
	fi;
	is*.is_icon := nil;
	is := is + sizeof(IconSlot_t);
    od;
    while
	ic := IconCache;
	ic ~= nil
    do
	IconCache := ic*.ic_next;
	FreeMem(ic*.ic_data, sizeof(IconData_t));
	FreeMem(ic, sizeof(IconCache_t));
    od;
    while
	tl := TileList;
	tl ~= nil
    do
	TileList := tl*.tl_next;
	freeBitMap(&tl*.tl_bitMap, tl*.tl_width, tl*.tl_height);
	FreeMem(tl, sizeof(TileList_t));
    od;
    clearButtons();
    clearRegions();

    /* Do soundTerm before freeFileCache, since it does unlockEffectsFile's */
    soundTerm(partial);
    if partial then
	effectsDefaults();
    else
	freeBitMap(&CursorSave, ICON_WIDTH, ICON_HEIGHT);
	while
	    fc := FileCache;
	    fc ~= nil
	do
	    FileCache := fc*.fc_next;
	    freeFileCache(fc);
	od;
	freeAllEffects();
    fi;
corp;

/*
 * effectsInit - set up the stuff here.
 */

proc effectsInit()bool:
    register uint i;

    if not soundInit() then
	return(false);
    fi;
    effectsDefaults();
    if ICON_WIDTH ~= sizeof(uint) * 8 then
	error("ICON_WIDTH ~= 16 (sizeof(uint) * 8)");
    fi;
    for i from MAX_ICONS - 1 downto 0 do
	Icons[i].is_icon := nil;
	Icons[i].is_alloced := false;
    od;
    IconCache := nil;
    FileCache := nil;
    FileCacheTail := nil;
    EffectCache := nil;
    EffectCacheTail := nil;
    TileList := nil;
    DefaultImage := nil;
    Regions := nil;
    Buttons := nil;
    if not allocBitMap(&CursorSave, Depth, ICON_WIDTH, ICON_HEIGHT,
		       "cursor save")
    then
	effectsTerm(false);
	false
    else
	true
    fi
corp;

/*
 * stringCopy - make a copy of a string.
 */

proc stringCopy(*char st)*char:
    register *char p, q;
    register ulong len;

    p := st;
    len := 1;
    while p* ~= '\e' do
	p := p + sizeof(char);
	len := len + 1;
    od;
    p := mudAlloc(len, 0x0);
    if p = nil then
	nil
    else
	q := st;
	st := p;
	while len ~= 0 do
	    len := len - 1;
	    p* := q*;
	    p := p + sizeof(char);
	    q := q + sizeof(char);
	od;
	st
    fi
corp;

/*
 * effectDir - the name of the directory for the given effect type.
 */

proc effectDir(FileType_t ft)*char:

    case ft
    incase ft_backGround:
	"BackGrounds"
    incase ft_image:
	"Images"
    incase ft_brush:
	"Brushes"
    incase ft_sound:
	"Sounds"
    incase ft_music:
	"Music"
    incase ft_instrument:
	"Instruments"
    default:
	""
    esac
corp;

/*
 * expandFileName - return a properly expanded filename.
 */

proc expandFileName(*char name; FileType_t ft)*char:
    register *char p, q;
    *char start;

    if LastFileName ~= nil then
	freeString(LastFileName);
    fi;
    LastFileName := stringCopy(name);
    LastFileType := ft;
    q := effectDir(ft);
    p := mudAlloc((CharsLen(q) + CharsLen(name) + 11) * sizeof(char), 0x0);
    if p = nil then
	noMemory("file path");
	nil
    else
	start := p;
	q := "AmigaMUD:";
	while q* ~= '\e' do
	    p* := q*;
	    p := p + sizeof(char);
	    q := q + sizeof(char);
	od;
	q := effectDir(ft);
	while q* ~= '\e' do
	    p* := q*;
	    p := p + sizeof(char);
	    q := q + sizeof(char);
	od;
	p* := '/';
	p := p + sizeof(char);
	q := name;
	while q* ~= '\e' do
	    p* := q*;
	    p := p + sizeof(char);
	    q := q + sizeof(char);
	od;
	p* := '\e';
	start
    fi
corp;

/*
 * findEffect - find the effect with the given key.
 *	Move it to the head of the effect list.
 */

proc findEffect(register ulong key)*EffectCache_t:
    register *EffectCache_t ec, prev, next;

    ec := EffectCache;
    while ec ~= nil and ec*.ec_key ~= key do
	ec := ec*.ec_next;
    od;
    if ec ~= nil then
	prev := ec*.ec_prev;
	if prev ~= nil then
	    next := ec*.ec_next;
	    prev*.ec_next := next;
	    if next ~= nil then
		next*.ec_prev := prev;
	    else
		EffectCacheTail := prev;
	    fi;
	    next := EffectCache;
	    ec*.ec_next := next;
	    ec*.ec_prev := nil;
	    next*.ec_prev := ec;
	    EffectCache := ec;
	fi;
    fi;
    ec
corp;

/*
 * removeCursor - remove the current cursor.
 */

proc removeCursor()void:

    BltBitMapRastPort(&CursorSave, 0, 0,
		      GraphicsRastPort, CursorX, CursorY,
		      CursorWidth, CursorHeight, 0xc0);
corp;

/*
 * showCursor - show the cursor at the new position - it is not now shown.
 */

proc showCursor()void:
    RastPort_t rp;
    ushort savedPen;

    InitRastPort(&rp);
    rp.rp_BitMap := &CursorSave;
    ignore ClipBlit(GraphicsRastPort, CursorX, CursorY, &rp, 0, 0,
		    CursorWidth, CursorHeight, 0xc0);
/*
    ignore BltBitMap(GraphicsRastPort*.rp_BitMap, CursorX, CursorY,
		     &CursorSave, 0, 0,
		     CursorWidth, CursorHeight, 0xc0, 0xffffffff, nil);
*/
    savedPen := GraphicsRastPort*.rp_FgPen;
    setGraphicsAPen(CursorPen);
    BltPattern(GraphicsRastPort, &CurrentCursor[0], CursorX, CursorY,
	       CursorX + CursorWidth - 1, CursorY + CursorHeight - 1,
	       ICON_WIDTH / 8);
    SetAPen(GraphicsRastPort, savedPen);
corp;

/*
 * doPlaceCursor - the actual effect to place the cursor.
 */

proc doPlaceCursor(uint x, y)void:

    if x < GraphicsWidth and y < GraphicsHeight then
	if CursorShown then
	    removeCursor();
	    CursorShown := false;
	fi;
	CursorX := x + GXOffset;
	CursorY := y + GYOffset;
	if x <= GraphicsWidth - ICON_WIDTH then
	    x := ICON_WIDTH;
	else
	    x := GraphicsWidth - x;
	fi;
	CursorWidth := x;
	if y <= GraphicsHeight - ICON_HEIGHT then
	    y := ICON_HEIGHT;
	else
	    y := GraphicsHeight - y;
	fi;
	CursorHeight := y;
	showCursor();
	CursorShown := true;
    fi;
corp;

/*
 * setNewCursor - set in a new cursor pattern.
 */

proc setNewCursor(register *byte p)void:
    register *byte q;
    register uint i;

    if CursorShown then
	removeCursor();
    fi;
    q := pretend(&CurrentCursor[0], *byte);
    for i from sizeof(IconData_t) - 1 downto 0 do
	q* := p*;
	q := q + 1;
	p := p + 1;
    od;
    if CursorShown then
	showCursor();
    fi;
corp;

/*
 * addButton - add a new mouse-button to the display and button list.
 */

proc addButton(uint x, y, n; *char st; uint len)void:
    register *RastPort_t rp;
    register *Region_t rg @ rp;
    register uint i;
    uint x2, y2;

    rg := Buttons;
    while rg ~= nil and rg*.rg_number ~= n do
	rg := rg*.rg_next;
    od;
    if rg = nil then
	rg := mudAlloc(sizeof(Region_t), 0x0);
	if rg ~= nil then
	    rg*.rg_text := stringCopy(st);
	    if rg*.rg_text ~= nil then
		x := x + GXOffset;
		y := y + GYOffset;
		x2 := x + len * CHAR_WIDTH + 7;
		y2 := y + (CHAR_HEIGHT + 7);
		rg*.rg_next := Buttons;
		Buttons := rg;
		rg*.rg_x1 := x;
		rg*.rg_y1 := y;
		rg*.rg_x2 := x2;
		rg*.rg_y2 := y2;
		rg*.rg_length := len;
		rg*.rg_number := n;
		rp := GraphicsRastPort;
		for i from 0 upto 2 do
		    setGraphicsAPen(ButtonColours[i + 2]);
		    Move(rp, x + i, y + i);
		    Draw(rp, x2 - i, y + i);
		    Draw(rp, x2 - i, y2 - i);
		    Draw(rp, x + i, y2 - i);
		    Draw(rp, x + i, y + i);
		od;
		setGraphicsAPen(ButtonColours[0]);
		RectFill(rp, x + 3, y + 3, x2 - 3, y2 - 3);
		Move(rp, x + 4, y + (4 + CHAR_BASELINE));
		setGraphicsAPen(ButtonColours[1]);
		Text(rp, st, len);
	    else
		FreeMem(rg, sizeof(Region_t));
		noMemory("button text");
	    fi;
	else
	    noMemory("mouse button");
	fi;
    fi;
corp;

/*
 * eraseButton - remove a mouse button from the list and display.
 */

proc eraseButton(uint n)void:
    register **Region_t rgp;
    register *Region_t rg;

    rgp := &Buttons;
    while
	rg := rgp*;
	rg ~= nil and rg*.rg_number ~= n
    do
	rgp := &rg*.rg_next;
    od;
    if rg ~= nil then
	setGraphicsAPen(if MapColours then ColourMapping[0] else 0 fi);
	RectFill(GraphicsRastPort, rg*.rg_x1, rg*.rg_y1, rg*.rg_x2, rg*.rg_y2);
	rgp* := rg*.rg_next;
	freeString(rg*.rg_text);
	FreeMem(rg, sizeof(Region_t));
    fi;
corp;

/*
 * addRegion - add a mouse-select region to the list. Sort by number.
 */

proc addRegion(uint x, y, width, height, n)void:
    register **Region_t rgp;
    register *Region_t rg;

    rgp := &Regions;
    while
	rg := rgp*;
	rg ~= nil and rg*.rg_number < n
    do
	rgp := &rg*.rg_next;
    od;
    if rg = nil or rg*.rg_number ~= n then
	rg := mudAlloc(sizeof(Region_t), 0x0);
	if rg ~= nil then
	    x := x + GXOffset;
	    y := y + GYOffset;
	    rg*.rg_next := rgp*;
	    rgp* := rg;
	    rg*.rg_x1 := x;
	    rg*.rg_y1 := y;
	    rg*.rg_x2 := x + width;
	    rg*.rg_y2 := y + height;
	    rg*.rg_number := n;
	else
	    noMemory("mouse region");
	fi;
    fi;
corp;

/*
 * eraseRegion - remove a mouse-select region from the list.
 */

proc eraseRegion(uint n)void:
    register **Region_t rgp;
    register *Region_t rg;

    rgp := &Regions;
    while
	rg := rgp*;
	rg ~= nil and rg*.rg_number ~= n
    do
	rgp := &rg*.rg_next;
    od;
    if rg ~= nil then
	rgp* := rg*.rg_next;
	FreeMem(rg, sizeof(Region_t));
    fi;
corp;

/*
 * mouseCheck - check for a mouse hit on a button or a region.
 *	Return true if we hit a button.
 */

proc mouseCheck(register int mouseX, mouseY; MouseCheck_t mc)void:
    register *Region_t rg;
    register *RastPort_t rp;
    ushort savedAPen, savedBPen;

    if mc = mc_up then
	if ButtonActive then
	    unHighLightButton();
	    sendButton(ButtonNumber);
	fi;
	return;
    fi;

    if mc = mc_move then
	if ButtonActive then
	    if mouseX < ButtonX1 or mouseX > ButtonX2 or
		mouseY < ButtonY1 or mouseY > ButtonY2
	    then
		/* pointer has gone off of the button - unhighlight it */
		unHighLightButton();
		/* Drop through to check for another button now hit. */
	    else
		return;
	    fi;
	else
	    return;
	fi;
    fi;

    /* mc_down */
    rg := Buttons;
    while rg ~= nil do
	/* When mouse buttons were set up, the upper limit was adjusted
	   to be inside the range we wish to check for. */
	if mouseX >= rg*.rg_x1 and mouseX <= rg*.rg_x2 and
	    mouseY >= rg*.rg_y1 and mouseY <= rg*.rg_y2
	then
	    rp := GraphicsRastPort;
	    savedAPen := rp*.rp_FgPen;
	    savedBPen := rp*.rp_BgPen;
	    ButtonX1 := rg*.rg_x1;
	    ButtonY1 := rg*.rg_y1;
	    ButtonX2 := rg*.rg_x2;
	    ButtonY2 := rg*.rg_y2;
	    ButtonNumber := rg*.rg_number;
	    ButtonText := rg*.rg_text;
	    ButtonLength := rg*.rg_length;
	    setGraphicsAPen(ButtonColours[5]);
	    RectFill(rp, rg*.rg_x1 + 3, rg*.rg_y1 + 3,
		     rg*.rg_x2 - 3, rg*.rg_y2 - 3);
	    Move(rp, rg*.rg_x1 + 4, rg*.rg_y1 + (4 + CHAR_BASELINE));
	    setGraphicsAPen(ButtonColours[6]);
	    setGraphicsBPen(ButtonColours[5]);
	    Text(rp, rg*.rg_text, rg*.rg_length);
	    ButtonActive := true;
	    SetAPen(rp, savedAPen);
	    SetBPen(rp, savedBPen);
	    return;
	fi;
	rg := rg*.rg_next;
    od;
    rg := Regions;
    while rg ~= nil do
	/* We compare >= X1 and < X2, since the upper bounds are outside
	   the region, whereas the lower bounds are included in the region. */
	if mouseX >= rg*.rg_x1 and mouseX < rg*.rg_x2 and
	    mouseY >= rg*.rg_y1 and mouseY < rg*.rg_y2
	then
	    sendRegion(rg*.rg_number, mouseX - rg*.rg_x1, mouseY - rg*.rg_y1);
	    return;
	fi;
	rg := rg*.rg_next;
    od;
corp;

/*
 * unHighLightButton - mouse action causes button to be de-selected.
 */

proc unHighLightButton()void:
    register *RastPort_t rp;
    ushort savedPen;

    rp := GraphicsRastPort;
    savedPen := rp*.rp_FgPen;
    ButtonActive := false;
    setGraphicsAPen(ButtonColours[0]);
    RectFill(rp, ButtonX1 + 3, ButtonY1 + 3, ButtonX2 - 3, ButtonY2 - 3);
    Move(rp, ButtonX1 + 4, ButtonY1 + (4 + CHAR_BASELINE));
    setGraphicsAPen(ButtonColours[1]);
    Text(rp, ButtonText, ButtonLength);
    SetAPen(rp, savedPen);
corp;

/*
 * saveBackground - save the background prior to inserting an icon.
 */

proc saveBackground(uint col, row; register *IconSlot_t is)void:
    RastPort_t rp;

    InitRastPort(&rp);
    rp.rp_BitMap := &is*.is_save;
    ignore ClipBlit(GraphicsRastPort, col + GXOffset, row + GYOffset,
		    &rp, 0, 0, ICON_WIDTH, ICON_HEIGHT, 0xc0);
/*
    ignore BltBitMap(GraphicsRastPort*.rp_BitMap,
		     col + GXOffset, row + GYOffset, &is*.is_save, 0, 0,
		     ICON_WIDTH, ICON_HEIGHT, 0xc0, 0xffffffff, nil);
*/
corp;

/*
 * restoreBackground - same, except copy the other direction.
 */

proc restoreBackground(register uint i; register *IconSlot_t is)void:

    BltBitMapRastPort(&is*.is_save, 0, 0, GraphicsRastPort,
		     i % (GraphicsWidth / ICON_WIDTH) * ICON_WIDTH + GXOffset,
		     i / (GraphicsWidth / ICON_WIDTH) * ICON_HEIGHT + GYOffset,
		     ICON_WIDTH, ICON_HEIGHT, 0xc0);
corp;

/*
 * iconInsert - insert an icon into the display.
 */

proc iconInsert(uint col, row; *IconData_t iData)void:
    ushort savedPen;

    col := col + GXOffset;
    row := row + GYOffset;
    savedPen := GraphicsRastPort*.rp_FgPen;
    setGraphicsAPen(IconPen);
    BltPattern(GraphicsRastPort, &iData*[0], col, row,
	       col + (ICON_WIDTH - 1), row + (ICON_HEIGHT - 1),
	       ICON_WIDTH / 8);
    SetAPen(GraphicsRastPort, savedPen);
corp;

/*
 * drawIcon - save background and draw a particular icon.
 */

proc drawIcon(register uint which)void:
    register *IconSlot_t is;
    register *IconCache_t ic @ is;
    register uint row, col @ which;

    is := &Icons[which];
    row := (which / (GraphicsWidth / ICON_WIDTH)) * ICON_HEIGHT;
    col := (which % (GraphicsWidth / ICON_WIDTH)) * ICON_WIDTH;
    saveBackground(col, row, is);
    ic := is*.is_icon;
    iconInsert(col, row, ic*.ic_data);
corp;

/*
 * newIcon - a new image for an icon has arrived.
 *	Note that the data in this case will NOT be word aligned.
 */

proc newIcon(register ulong key; register *byte p)void:
    register *IconCache_t ic;
    register *byte q;
    register *IconSlot_t is @ q;
    register uint i;
    bool isNewIcon, first;

    ic := IconCache;
    while ic ~= nil and ic*.ic_key ~= key do
	ic := ic*.ic_next;
    od;
    isNewIcon := false;
    if ic = nil then
	/* never seen it before - just add it to the cache */
	isNewIcon := true;
	ic := mudAlloc(sizeof(IconCache_t), 0x0);
	if ic = nil then
	    noMemory("icon");
	    return;
	fi;
	ic*.ic_data := mudAlloc(sizeof(IconData_t), MEMF_CHIP);
	if ic*.ic_data = nil then
	    noMemory("icon data");
	    return;
	fi;
	ic*.ic_next := IconCache;
	ic*.ic_prev := nil;
	ic*.ic_key := key;
	if IconCache ~= nil then
	    IconCache*.ic_prev := ic;
	fi;
	IconCache := ic;
    fi;
    if ic ~= nil then
	/* Byte-copy the icon data from the effects transaction. */
	q := pretend(ic*.ic_data, *byte);
	for i from sizeof(IconData_t) - 1 downto 0 do
	    q* := p*;
	    q := q + 1;
	    p := p + 1;
	od;
	if not isNewIcon then
	    /* we may have a reference to this one - update any we find */
	    first := true;
	    is := &Icons[0];
	    for i from 0 upto MAX_ICONS - 1 do
		if is*.is_icon = ic then
		    if first then
			first := false;
			if CursorShown then
			    removeCursor();
			fi;
		    fi;
		    restoreBackground(i, is);
		    drawIcon(i);
		fi;
		is := is + sizeof(IconSlot_t);
	    od;
	    if not first then
		if CursorShown then
		    showCursor();
		fi;
	    fi;
	fi;
    fi;
corp;

/*
 * showIcon - show a character icon on this client's screen
 */

proc showIcon(register ulong key; bool isMonster)void:
    register uint i;
    register *IconCache_t ic;
    register *IconSlot_t is;

    is := &Icons[0];
    for i from 0 upto MAX_ICONS - 1 do
	if is*.is_icon = nil then
	    ic := IconCache;
	    while ic ~= nil and ic*.ic_key ~= key do
		ic := ic*.ic_next;
	    od;
	    if ic = nil then
		ic := mudAlloc(sizeof(IconCache_t), 0x0);
		if ic = nil then
		    noMemory("icon");
		    return;
		fi;
		ic*.ic_data := mudAlloc(sizeof(IconData_t), MEMF_CHIP);
		if ic*.ic_data = nil then
		    noMemory("icon data");
		    FreeMem(ic, sizeof(IconCache_t));
		    return;
		fi;
		ic*.ic_next := IconCache;
		ic*.ic_prev := nil;
		ic*.ic_key := key;
		ic*.ic_data* :=
		    if isMonster then
			DefaultMonsterIcon
		    else
			DefaultCharacterIcon
		    fi;
		if IconCache ~= nil then
		    IconCache*.ic_prev := ic;
		fi;
		IconCache := ic;
	    fi;
	    if is*.is_alloced or
		allocBitMap(&is*.is_save, Depth,
			    ICON_WIDTH, ICON_HEIGHT, "icon background")
	    then
		is*.is_alloced := true;
		is*.is_icon := ic;
		if CursorShown then
		    removeCursor();
		fi;
		drawIcon(i);
		if CursorShown then
		    showCursor();
		fi;
	    fi;
	    return;
	fi;
	is := is + sizeof(IconSlot_t);
    od;
corp;

/*
 * removeIcon - remove an icon from this client's screen
 */

proc removeIcon(register ulong key)void:
    register uint i;
    register *IconSlot_t is;

    is := &Icons[0];
    for i from 0 upto MAX_ICONS - 1 do
	if is*.is_icon ~= nil and is*.is_icon*.ic_key = key then
	    if CursorShown then
		removeCursor();
	    fi;
	    restoreBackground(i, is);
	    if CursorShown then
		showCursor();
	    fi;
	    is*.is_icon := nil;
	    return;
	fi;
	is := is + sizeof(IconSlot_t);
    od;
corp;

/*
 * deleteIcon - remove an icon from this client's screen and also remove it
 *	from the icon cache.
 */

proc deleteIcon(register ulong key)void:
    register uint i;
    register *IconSlot_t is;
    register *IconCache_t ic, prevIc @ is;

    is := &Icons[0];
    for i from 0 upto MAX_ICONS - 1 do
	ic := is*.is_icon;
	if ic ~= nil and ic*.ic_key = key then
	    if CursorShown then
		removeCursor();
	    fi;
	    restoreBackground(i, is);
	    if CursorShown then
		showCursor();
	    fi;
	    is*.is_icon := nil;
	    prevIc := ic*.ic_prev;
	    if prevIc ~= nil then
		prevIc*.ic_next := ic*.ic_next;
	    else
		IconCache := ic*.ic_next;
	    fi;
	    if ic*.ic_next ~= nil then
		ic*.ic_next*.ic_prev := prevIc;
	    fi;
	    FreeMem(ic, sizeof(IconCache_t));
	    return;
	fi;
	is := is + sizeof(IconSlot_t);
    od;
corp;

/*
 * resetIcons - reset to indicate that no icons are present.
 */

proc resetIcons()void:
    register uint i;
    register *IconSlot_t is;

    is := &Icons[0];
    for i from MAX_ICONS - 1 downto 0 do
	is*.is_icon := nil;
	is := is + sizeof(IconSlot_t);
    od;
corp;

/*
 * redrawIcons - go through and redraw all icons that should be present.
 */

proc redrawIcons()void:
    register uint i;
    register *IconSlot_t is;

    if CursorShown then
	removeCursor();
    fi;
    /* Draw them in forward order - it looks odd to to it in reverse. */
    is := &Icons[0];
    for i from 0 upto MAX_ICONS - 1 do
	if is*.is_icon ~= nil then
	    drawIcon(i);
	fi;
	is := is + sizeof(IconSlot_t);
    od;
    if CursorShown then
	showCursor();
    fi;
corp;

/*
 * undrawIcons - go through and undraw all icons.
 */

proc undrawIcons()void:
    register uint i;
    register *IconSlot_t is;

    if CursorShown then
	removeCursor();
    fi;
    /* We need 'i' to correspond to 'is', so use forward order. */
    is := &Icons[0];
    for i from 0 upto MAX_ICONS - 1 do
	if is*.is_icon ~= nil then
	    restoreBackground(i, is);
	fi;
	is := is + sizeof(IconSlot_t);
    od;
    if CursorShown then
	showCursor();
    fi;
corp;

/*
 * pixelsToBitMap - reformat an array of pixels into the bitplanes of a BitMap.
 */

proc pixelsToBitMap(*BitMap_t bm; *byte p; uint w, h)void:
    register *byte pip;
    register *uint plp;
    register uint result;
    register byte temp, mask, pos;
    register uint j, i, plane;

    if MapColours then
	/* Remap the pens of the tile */
	pip := p;
	for i from h - 1 downto 0 do
	    for j from w - 1 downto 0 do
		temp := pip*;
		if temp >= GRAPHICS_COLOURS then
		    temp := 0;
		else
		    temp := ColourMapping[temp];
		fi;
		pip* := temp;
		pip := pip + 1;
	    od;
	od;
    fi;

    mask := 0x01;
    for plane from 0 upto bm*.bm_Depth - 1 do
	pip := p;
	plp := bm*.bm_Planes[plane];
	result := 0;
	pos := 16;
	for i from h - 1 downto 0 do
	    for j from w - 1 downto 0 do
		result := result << 1;
		temp := pip*;
		pip := pip + 1;
		if temp & mask ~= 0 then
		    result := result | 1;
		fi;
		pos := pos - 1;
		if pos = 0 then
		    plp* := result;
		    plp := plp + sizeof(uint);
		    result := 0;
		    pos := 16;
		fi;
	    od;
	    if pos ~= 16 then
		result := result << pos;
		plp* := result;
		plp := plp + sizeof(uint);
		result := 0;
		pos := 16;
	    fi;
	od;
	mask := mask << 1;
    od;
corp;

/*
 * freeOneCachedFile - free one cached file if we can.
 */

proc freeOneCachedFile()bool:
    register *FileCache_t fc, prev, next;

    fc := FileCacheTail;
    while fc ~= nil do
	prev := fc*.fc_prev;
	if fc*.fc_lockCount = 0 then
	    next := fc*.fc_next;
	    if prev ~= nil then
		prev*.fc_next := next;
	    else
		FileCache := next;
	    fi;
	    if next ~= nil then
		next*.fc_prev := prev;
	    else
		FileCacheTail := prev;
	    fi;
	    freeFileCache(fc);
	    return(true);
	fi;
	fc := prev;
    od;
    false
corp;

/*
 * flushCachedImages - we are loading a new background in a mode where
 *	we have control over the palette. That means that all cached
 *	images and brushes are now mapped incorrectly, and need to be
 *	flushed, so that they will be remapped on their next use.
 */

proc flushCachedImages()void:
    register *FileCache_t fc, prev, next;

    fc := FileCache;
    while fc ~= nil do
	next := fc*.fc_next;
	if fc*.fc_type = ft_image or fc*.fc_type = ft_brush then
	    if fc*.fc_lockCount = 0 then
		prev := fc*.fc_prev;
		if prev ~= nil then
		    prev*.fc_next := next;
		else
		    FileCache := next;
		fi;
		if next ~= nil then
		    next*.fc_prev := prev;
		else
		    FileCacheTail := prev;
		fi;
		freeFileCache(fc);
	    else
		SavedPen := GraphicsRastPort*.rp_FgPen;
		SetAPen(GraphicsRastPort, TextPen);
		putString("::Cached image ");
		putString(fc*.fc_filePath);
		putString(" locked - not remapped.::\n");
		SetAPen(GraphicsRastPort, SavedPen);
	    fi;
	fi;
	fc := next;
    od;
corp;

/*
 * findCachedFile - find the given file in the file cache.
 */

proc findCachedFile(*char fileName; FileType_t ft)*FileCache_t:
    register *char p, q;
    register *FileCache_t fc, prev @ p, next @ q;

    fc := FileCache;
    while fc ~= nil do
	if fc*.fc_type = ft then
	    p := fileName;
	    q := fc*.fc_filePath;
	    while p* = q* and p* ~= '\e' do
		p := p + sizeof(char);
		q := q + sizeof(char);
	    od;
	    if p* = q* then
		/* do LRU - move it to the front of the list */
		prev := fc*.fc_prev;
		if prev ~= nil then
		    next := fc*.fc_next;
		    prev*.fc_next := next;
		    if next ~= nil then
			next*.fc_prev := prev;
		    else
			FileCacheTail := prev;
		    fi;
		    fc*.fc_prev := nil;
		    next := FileCache;
		    next*.fc_prev := fc;
		    fc*.fc_next := next;
		    FileCache := fc;
		fi;
		return(fc);
	    fi;
	fi;
	fc := fc*.fc_next;
    od;
    fc
corp;

/*
 * addFileToCache - add a new file to the file cache.
 */

proc addFileToCache(*char fileName; arbptr p; FileType_t ft)*FileCache_t:
    register *FileCache_t fc, next;

    fc := mudAlloc(sizeof(FileCache_t), 0x0);
    if fc ~= nil then
	fc*.fc_filePath := stringCopy(fileName);
	if fc*.fc_filePath = nil then
	    FreeMem(fc, sizeof(FileCache_t));
	    fc := nil;
	    noMemory("file path");
	else
	    next := FileCache;
	    fc*.fc_next := next;
	    fc*.fc_prev := nil;
	    fc*.fc_ptr := p;
	    fc*.fc_lockCount := 0;
	    fc*.fc_type := ft;
	    if next = nil then
		FileCacheTail := fc;
	    else
		next*.fc_prev := fc;
	    fi;
	    FileCache := fc;
	fi;
    else
	noMemory("cached file");
    fi;
    fc
corp;

/*
 * loadILBM - load an ILBM file (either brush or image).
 */

proc loadILBM(*char fileName; FileType_t ft)*FileCache_t:
    register *FileCache_t fc;
    register arbptr im;
    *char path;

    fc := findCachedFile(fileName, ft);
    if fc = nil then
	path := expandFileName(fileName, ft);
	if path ~= nil then
	    im := iffLoadILBM(path, ft - ft_backGround + pt_backGround);
	    freeString(path);
	    if im ~= nil then
		fc := addFileToCache(fileName, im, ft);
		if fc = nil then
		    iffFreeILBM(im);
		fi;
	    fi;
	fi;
    fi;
    fc
corp;

/*
 * load8SVX - load an 8SVX file (either sound or instrument).
 *	Note that this routine is called from iffLoadSMUS also.
 */

proc load8SVX(*char fileName)*FileCache_t:
    register *FileCache_t fc;
    register arbptr sn;
    *char path;

    fc := findCachedFile(fileName, ft_sound);
    if fc = nil then
	path := expandFileName(fileName, ft_sound);
	if path ~= nil then
	    sn := iffLoad8SVX(path);
	    freeString(path);
	    if sn ~= nil then
		fc := addFileToCache(fileName, sn, ft_sound);
		if fc = nil then
		    iffFree8SVX(sn);
		fi;
	    fi;
	fi;
    fi;
    fc
corp;

/*
 * loadSMUS - load a SMUS file.
 */

proc loadSMUS(*char fileName)*FileCache_t:
    register *FileCache_t fc;
    register arbptr mu;
    *char path;

    fc := findCachedFile(fileName, ft_music);
    if fc = nil then
	path := expandFileName(fileName, ft_music);
	if path ~= nil then
	    mu := iffLoadSMUS(path);
	    freeString(path);
	    if mu ~= nil then
		fc := addFileToCache(fileName, mu, ft_music);
		if fc = nil then
		    iffFreeSMUS(mu);
		fi;
	    fi;
	fi;
    fi;
    fc
corp;

/*
 * loadInstrument - called from SMUS stuff to load an instrument 8SVX.
 */

proc loadInstrument(*char fileName)arbptr:
    register *FileCache_t fc;
    register arbptr sn;
    *char path;

    fc := findCachedFile(fileName, ft_instrument);
    if fc = nil then
	path := expandFileName(fileName, ft_instrument);
	if path ~= nil then
	    sn := iffLoad8SVX(path);
	    freeString(path);
	    if sn ~= nil then
		fc := addFileToCache(fileName, sn, ft_instrument);
		if fc = nil then
		    iffFree8SVX(sn);
		fi;
	    fi;
	fi;
    fi;
    if fc = nil then
	nil
    else
	fc*.fc_ptr
    fi
corp;

/*
 * doAMove - do an absolute move.
 */

proc doAMove(register uint x, y)void:

    if x >= GraphicsWidth then
	x := GraphicsWidth - 1;
    fi;
    if y >= GraphicsHeight then
	y := GraphicsHeight - 1;
    fi;
    x := x + GXOffset;
    y := y + GYOffset;
    if Filling then
	ignore AreaMove(GraphicsRastPort, x, y);
    fi;
    Move(GraphicsRastPort, x, y);
corp;

/*
 * doRMove - do a relative move.
 */

proc doRMove(register uint dx, dy)void:
    register *RastPort_t rp;

    rp := GraphicsRastPort;
    dx := dx + rp*.rp_cp_x;
    dx := dx - GXOffset;
    dy := dy + rp*.rp_cp_y;
    dy := dy - GYOffset;
    if dx = 0xffff then
	dx := 0;
    elif dx >= GraphicsWidth then
	dx := GraphicsWidth - 1;
    fi;
    if dy = 0xffff then
	dy := 0;
    elif dy >= GraphicsHeight then
	dy := GraphicsHeight - 1;
    fi;
    dx := dx + GXOffset;
    dy := dy + GYOffset;
    if Filling then
	ignore AreaMove(rp, dx, dy);
    fi;
    Move(rp, dx, dy);
corp;

/*
 * doADraw - do an absolute draw.
 */

proc doADraw(register uint x, y)void:

    if x >= GraphicsWidth then
	x := GraphicsWidth - 1;
    fi;
    if y >= GraphicsHeight then
	y := GraphicsHeight - 1;
    fi;
    x := x + GXOffset;
    y := y + GYOffset;
    if Filling then
	ignore AreaDraw(GraphicsRastPort, x, y);
    fi;
    Draw(GraphicsRastPort, x, y);
corp;

/*
 * doRDraw - do a relative draw.
 */

proc doRDraw(register uint dx, dy)void:
    *RastPort_t rp;

    rp := GraphicsRastPort;
    dx := dx + rp*.rp_cp_x;
    dx := dx - GXOffset;
    dy := dy + rp*.rp_cp_y;
    dy := dy - GYOffset;
    if dx = 0xffff then
	dx := 0;
    elif dx >= GraphicsWidth then
	dx := GraphicsWidth - 1;
    fi;
    if dy = 0xffff then
	dy := 0;
    elif dy >= GraphicsHeight then
	dy := GraphicsHeight - 1;
    fi;
    dx := dx + GXOffset;
    dy := dy + GYOffset;
    if Filling then
	ignore AreaDraw(rp, dx, dy);
    fi;
    Draw(rp, dx, dy);
corp;

/*
 * doRectangle - do the work of drawing a rectangle.
 */

proc doRectangle(register uint dx, dy; uint fill)void:
    register *RastPort_t rp;
    register uint x, y;

    rp := GraphicsRastPort;
    x := rp*.rp_cp_x - GXOffset;
    y := rp*.rp_cp_y - GYOffset;
    /* Shrink the width/height values, since the Amiga routines
       want pixel coordinates, not width/height. */
    if dx ~= 0 then
	dx := dx - 1;
    fi;
    if dy ~= 0 then
	dy := dy - 1;
    fi;
    if dx >= GraphicsWidth - x then
	dx := GraphicsWidth - x - 1;
    fi;
    if dy >= GraphicsHeight - y then
	dy := GraphicsHeight - y - 1;
    fi;
    x := x + GXOffset;
    y := y + GYOffset;
    if fill ~= 0 then
	RectFill(rp, x, y, x + dx, y + dy);
    else
	Draw(rp, x + dx, y);
	Draw(rp, x + dx, y + dy);
	Draw(rp, x, y + dy);
	Draw(rp, x, y);
    fi;
corp;

/*
 * doCircle - draw a circle.
 */

proc doCircle(register uint r; uint fill)void:
    register *RastPort_t rp;
    register uint x, y;

    rp := GraphicsRastPort;
    if r ~= 0 then
	r := r - 1;
    fi;
    x := rp*.rp_cp_x - GXOffset;
    y := rp*.rp_cp_y - GYOffset;
    if x + r < GraphicsWidth and y + r < GraphicsHeight and x >= r and y >= r
    then
	x := x + GXOffset;
	y := y + GYOffset;
	if fill ~= 0 then
	    zapArea();
	    ignore AreaCircle(rp, x, y, r);
	    ignore AreaEnd(rp);
	else
	    DrawCircle(rp, x, y, r);
	fi;
    fi;
corp;

/*
 * doEllipse - draw an ellipse.
 */

proc doEllipse(register uint r1, r2; uint fill)void:
    register *RastPort_t rp;
    register uint x, y;

    rp := GraphicsRastPort;
    if r1 ~= 0 then
	r1 := r1 - 1;
    fi;
    if r2 ~= 0 then
	r2 := r2 - 1;
    fi;
    x := rp*.rp_cp_x - GXOffset;
    y := rp*.rp_cp_y - GYOffset;
    if x + r1 < GraphicsWidth and y + r2 < GraphicsHeight and
	x >= r1 and y >= r2
    then
	x := x + GXOffset;
	y := y + GYOffset;
	if fill ~= 0 then
	    zapArea();
	    ignore AreaEllipse(rp, x, y, r1, r2);
	    ignore AreaEnd(rp);
	else
	    DrawEllipse(rp, x, y, r1, r2);
	fi;
    fi;
corp;

/*
 * doEffects1 - 2nd level handler for special effects.
 */

proc doEffects1(register *byte p; register uint len)void:
    register *RastPort_t rp;
    register *FileCache_t fc;
    register *EffectCache_t ec @ fc;
    register *TileList_t tl @ fc;
    register ulong u, u2;
    ulong u3, u4, u5, u6, u7, u8;
    *char cp;
    register uint w @ u, w2 @ u2;
    register byte b @ u, b2 @ u2;
    uint w3 @ u3, w4 @ u4, w5 @ u5, w6 @ u6;
    EffectType_t et;
    byte fill;
    ushort savedPen;
    register bool ignoring;

    rp := GraphicsRastPort;
    ignoring := Ignoring;
    while len ~= 0 do
	et := p* + ef_null;
	p := p + 1;
	len := len - 1;
	case et
	incase ef_Else:
	    ignoring := not ignoring;
	incase ef_Fi:
	    ignoring := false;
	incase ef_IfFound:
	    ignoring := FindError;
	incase ef_FailText:
	    cp := pretend(p, *char);
	    while p* ~= 0 do
		p := p + 1;
		len := len - 1;
	    od;
	    p := p + 1;
	    len := len - 1;
	    if not ignoring then
		if LastFileName ~= nil then
		    SavedPen := rp*.rp_FgPen;
		    SetAPen(rp, TextPen);
		    putChar('[');
		    putString(effectDir(LastFileType));
		    putChar('/');
		    putString(LastFileName);
		    putString("] ");
		    putString(cp);
		    newLine();
		    SetAPen(rp, SavedPen);
		fi;
	    fi;
	incase ef_Abort:
	    b2 := p*;
	    p := p + 1;
	    u := p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    len := len - (1 + 4);
	    if not ignoring then
		abortSound(b2, u);
	    fi;
	incase ef_Call:
	    u := p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    len := len - 4;
	    ec := findEffect(u);
	    if ec ~= nil and not ignoring then
		Ignoring := ignoring;
		ec*.ec_useCount := ec*.ec_useCount + 1;
		doEffects1(ec*.ec_data, ec*.ec_length);
		ec*.ec_useCount := ec*.ec_useCount - 1;
		ignoring := Ignoring;
		if ec*.ec_key = 0 and ec*.ec_useCount = 0 then
		    /* never cache effect 0 - it is reused all the time. */
		    freeEffect(ec);
		fi;
	    fi;
	incase ef_PlaceCursor:
	    w := p*;
	    p := p + 1;
	    w := w << 8;
	    w := w | p*;
	    p := p + 1;
	    w2 := p*;
	    p := p + 1;
	    w2 := w2 << 8;
	    w2 := w2 | p*;
	    p := p + 1;
	    if not ignoring then
		doPlaceCursor(w, w2);
	    fi;
	    len := len - (2 + 2);
	incase ef_PlaceCursorShort:
	    w := p*;
	    p := p + 1;
	    w2 := p*;
	    p := p + 1;
	    if not ignoring then
		doPlaceCursor(w, w2);
	    fi;
	    len := len - (1 + 1);
	incase ef_RemoveCursor:
	    if not ignoring and CursorShown then
		removeCursor();
		CursorShown := false;
	    fi;
	incase ef_SetCursorPen:
	    w := p*;
	    p := p + 1;
	    w := w << 8;
	    w := w | p*;
	    p := p + 1;
	    if not ignoring then
		if w < ColourMax then
		    if MapColours then
			w := ColourMapping[w];
		    fi;
		    CursorPen := w;
		    if CursorShown then
			removeCursor();
			showCursor();
		    fi;
		fi;
	    fi;
	    len := len - 2;
	incase ef_SetCursorPattern:
	    if not ignoring then
		setNewCursor(p);
	    fi;
	    p := p + 32;
	    len := len - 32;
	incase ef_AddButton:
	    w := p*;
	    p := p + 1;
	    w := w << 8;
	    w := w | p*;
	    p := p + 1;
	    w3 := w;
	    w := p*;
	    p := p + 1;
	    w := w << 8;
	    w := w | p*;
	    p := p + 1;
	    w4 := w;
	    w2 := p*;
	    p := p + 1;
	    w2 := w2 << 8;
	    w2 := w2 | p*;
	    p := p + 1;
	    len := len - (2 + 2 + 2);
	    cp := pretend(p, *char);
	    w := 0;
	    while p* ~= 0 do
		p := p + 1;
		w := w + 1
	    od;
	    len := len - w;
	    p := p + 1;
	    len := len - 1;
	    if not ignoring then
		if w3 + w * CHAR_WIDTH + 8 <= GraphicsWidth and
		    w4 + CHAR_HEIGHT + 8 <= GraphicsHeight
		then
		    addButton(w3, w4, w2, cp, w);
		fi;
	    fi;
	incase ef_EraseButton:
	    w := p*;
	    p := p + 1;
	    w := w << 8;
	    w := w | p*;
	    p := p + 1;
	    len := len - 2;
	    if not ignoring then
		eraseButton(w);
	    fi;
	incase ef_ClearButtons:
	    if not ignoring then
		clearButtons();
	    fi;
	incase ef_AddRegion:
	    w := p*;
	    p := p + 1;
	    w := w << 8;
	    w := w | p*;
	    p := p + 1;
	    w3 := w;
	    w2 := p*;
	    p := p + 1;
	    w2 := w2 << 8;
	    w2 := w2 | p*;
	    p := p + 1;
	    w4 := w2;
	    w := p*;
	    p := p + 1;
	    w := w << 8;
	    w := w | p*;
	    p := p + 1;
	    w5 := w;
	    w := p*;
	    p := p + 1;
	    w := w << 8;
	    w := w | p*;
	    p := p + 1;
	    w2 := p*;
	    p := p + 1;
	    w2 := w2 << 8;
	    w2 := w2 | p*;
	    p := p + 1;
	    len := len - (2 + 2 + 2 + 2 + 2);
	    if not ignoring then
		if w3 < GraphicsWidth and w4 < GraphicsHeight and
		    w3 <= w5 and w4 <= w
		then
		    addRegion(w3, w4, w5, w, w2);
		fi;
	    fi;
	incase ef_EraseRegion:
	    w := p*;
	    p := p + 1;
	    w := w << 8;
	    w := w | p*;
	    p := p + 1;
	    len := len - 2;
	    if not ignoring then
		eraseRegion(w);
	    fi;
	incase ef_ClearRegions:
	    if not ignoring then
		clearRegions();
	    fi;
	incase ef_SetButtonPen:
	    w := p*;
	    p := p + 1;
	    w := w << 8;
	    w := w | p*;
	    p := p + 1;
	    w2 := p*;
	    p := p + 1;
	    w2 := w2 << 8;
	    w2 := w2 | p*;
	    p := p + 1;
	    len := len - (2 + 2);
	    if not ignoring then
		if w < 7 and w2 < ColourMax then
		    if MapColours then
			ButtonColours[w] := ColourMapping[w2];
		    else
			ButtonColours[w] := w2;
		    fi;
		fi;
	    fi;
	incase ef_SetPen:
	    w := p*;
	    p := p + 1;
	    w := w << 8;
	    w := w | p*;
	    p := p + 1;
	    len := len - 2;
	    if not ignoring then
		if w < ColourMax then
		    if MapColours then
			setGraphicsAPen(ColourMapping[w]);
		    else
			setGraphicsAPen(w);
		    fi;
		fi;
	    fi;
	incase ef_SetColour:
	    w := p*;
	    p := p + 1;
	    w := w << 8;
	    w := w | p*;
	    p := p + 1;
	    w2 := p*;
	    p := p + 1;
	    w2 := w2 << 8;
	    w2 := w2 | p*;
	    p := p + 1;
	    len := len - (2 + 2);
	    if not ignoring then
		if w < ColourMax then
		    if MapColours then
			ColourMapping[w] := iffMapColour(w2);
		    else
			setGraphicsPen(w, w2);
		    fi;
		fi;
	    fi;
	incase ef_ResetColours:
	    if not ignoring then
		if MapColours then
		    initColourMapping();
		else
		    resetColours();
		fi;
	    fi;
	incase ef_Clear:
	    if not ignoring then
		setGraphicsRast(if MapColours then ColourMapping[0] else 0 fi);
	    fi;
	incase ef_Pixel:
	    if not ignoring then
		ignore WritePixel(rp, rp*.rp_cp_x, rp*.rp_cp_y);
	    fi;
	incase ef_AMove:
	    w := p*;
	    p := p + 1;
	    w := w << 8;
	    w := w | p*;
	    p := p + 1;
	    w2 := p*;
	    p := p + 1;
	    w2 := w2 << 8;
	    w2 := w2 | p*;
	    p := p + 1;
	    len := len - (2 + 2);
	    if not ignoring then
		doAMove(w, w2);
	    fi;
	incase ef_AMoveShort:
	    w := p*;
	    p := p + 1;
	    w2 := p*;
	    p := p + 1;
	    len := len - (1 + 1);
	    if not ignoring then
		doAMove(w, w2);
	    fi;
	incase ef_RMove:
	    w := p*;
	    p := p + 1;
	    w := w << 8;
	    w := w | p*;
	    p := p + 1;
	    w2 := p*;
	    p := p + 1;
	    w2 := w2 << 8;
	    w2 := w2 | p*;
	    p := p + 1;
	    len := len - (2 + 2);
	    if not ignoring then
		doRMove(w, w2);
	    fi;
	incase ef_RMoveShort:
	    /* want the values sign-extended */
	    w := make(p*, short);
	    p := p + 1;
	    w2 := make(p*, short);
	    p := p + 1;
	    len := len - (1 + 1);
	    if not ignoring then
		doRMove(w, w2);
	    fi;
	incase ef_ADraw:
	    w := p*;
	    p := p + 1;
	    w := w << 8;
	    w := w | p*;
	    p := p + 1;
	    w2 := p*;
	    p := p + 1;
	    w2 := w2 << 8;
	    w2 := w2 | p*;
	    p := p + 1;
	    len := len - (2 + 2);
	    if not ignoring then
		doADraw(w, w2);
	    fi;
	incase ef_ADrawShort:
	    w := p*;
	    p := p + 1;
	    w2 := p*;
	    p := p + 1;
	    len := len - (1 + 1);
	    if not ignoring then
		doADraw(w, w2);
	    fi;
	incase ef_RDraw:
	    w := p*;
	    p := p + 1;
	    w := w << 8;
	    w := w | p*;
	    p := p + 1;
	    w2 := p*;
	    p := p + 1;
	    w2 := w2 << 8;
	    w2 := w2 | p*;
	    p := p + 1;
	    len := len - (2 + 2);
	    if not ignoring then
		doRDraw(w, w2);
	    fi;
	incase ef_RDrawShort:
	    /* want the values sign-extended */
	    w := make(p*, short);
	    p := p + 1;
	    w2 := make(p*, short);
	    p := p + 1;
	    len := len - (1 + 1);
	    if not ignoring then
		doRDraw(w, w2);
	    fi;
	incase ef_Rectangle:
	    w := p*;
	    p := p + 1;
	    w := w << 8;
	    w := w | p*;
	    p := p + 1;
	    w2 := p*;
	    p := p + 1;
	    w2 := w2 << 8;
	    w2 := w2 | p*;
	    p := p + 1;
	    fill := p*;
	    p := p + 1;
	    len := len - (2 + 2 + 1);
	    if not ignoring then
		doRectangle(w, w2, fill);
	    fi;
	incase ef_Circle:
	    w := p*;
	    p := p + 1;
	    w := w << 8;
	    w := w | p*;
	    p := p + 1;
	    fill := p*;
	    p := p + 1;
	    len := len - (2 + 1);
	    if not ignoring then
		doCircle(w, fill);
	    fi;
	incase ef_Ellipse:
	    w := p*;
	    p := p + 1;
	    w := w << 8;
	    w := w | p*;
	    p := p + 1;
	    w2 := p*;
	    p := p + 1;
	    w2 := w2 << 8;
	    w2 := w2 | p*;
	    p := p + 1;
	    fill := p*;
	    p := p + 1;
	    len := len - (2 + 2 + 1);
	    if not ignoring then
		doEllipse(w, w2, fill);
	    fi;
	incase ef_PolygonStart:
	    if not ignoring then
		Filling := true;
	    fi;
	incase ef_PolygonEnd:
	    if not ignoring then
		ignore AreaEnd(rp);
		Filling := false;
	    fi;
	incase ef_DefineTile:
	    w2 := p*;
	    p := p + 1;
	    w2 := w2 << 8;
	    w2 := w2 | p*;
	    p := p + 1;
	    w3 := w2;
	    b := p*;
	    p := p + 1;
	    b2 := p*;
	    p := p + 1;
	    len := len - (2 + 1 + 1);
	    tl := mudAlloc(sizeof(TileList_t), 0x0);
	    if tl ~= nil then
		if allocBitMap(&tl*.tl_bitMap, Depth, b, b2, "tile buffer")
		then
		    tl*.tl_next := TileList;
		    TileList := tl;
		    tl*.tl_id := w3;
		    tl*.tl_width := b;
		    tl*.tl_height := b2;
		    pixelsToBitMap(&tl*.tl_bitMap, p, b, b2);
		else
		    FreeMem(tl, sizeof(TileList_t));
		    noMemory("tile data");
		fi;
	    else
		noMemory("tile");
	    fi;
	    w := make(b, uint) * make(b2, uint);
	    p := p + w;
	    len := len - w;
	incase ef_DisplayTile:
	    w2 := p*;
	    p := p + 1;
	    w2 := w2 << 8;
	    w2 := w2 | p*;
	    p := p + 1;
	    len := len - 2;
	    tl := TileList;
	    while tl ~= nil and tl*.tl_id ~= w2 do
		tl := tl*.tl_next;
	    od;
	    if tl ~= nil then
		/* Nuts, have to compensate for the adds in 'showBitMap' */
		showBitMap(&tl*.tl_bitMap, 0, 0, tl*.tl_width, tl*.tl_height,
		    tl*.tl_width, tl*.tl_height,
		    rp*.rp_cp_x - GXOffset, rp*.rp_cp_y - GYOffset);
	    else
		SavedPen := rp*.rp_FgPen;
		SetAPen(rp, TextPen);
		putString("::Tile ");
		putUnsigned(w2);
		putString(" not found!::\n");
		SetAPen(rp, SavedPen);
	    fi;
	incase ef_SetTextColour:
	    w := p*;
	    p := p + 1;
	    w := w << 8;
	    w := w | p*;
	    p := p + 1;
	    w2 := p*;
	    p := p + 1;
	    w2 := w2 << 8;
	    w2 := w2 | p*;
	    p := p + 1;
	    len := len - (2 + 2);
	    if not ignoring then
		setTextPen(w, w2);
	    fi;
	incase ef_Text:
	    cp := pretend(p, *char);
	    w := 0;
	    while p* ~= 0 do
		p := p + 1;
		w := w + 1
	    od;
	    len := len - w;
	    p := p + 1;
	    len := len - 1;
	    if not ignoring then
		if rp*.rp_cp_x - GXOffset + w * CHAR_WIDTH <=
			make(GraphicsWidth, int) and
		    rp*.rp_cp_y - GYOffset >= CHAR_BASELINE and
		    rp*.rp_cp_y - GYOffset + (CHAR_HEIGHT - CHAR_BASELINE) <=
			make(GraphicsHeight, int)
		then
		    Text(rp, cp, w);
		fi;
	    fi;
	incase ef_LoadBackGround:
	    cp := pretend(p, *char);
	    while p* ~= 0 do
		p := p + 1;
		len := len - 1;
	    od;
	    p := p + 1;
	    len := len - 1;
	    if not ignoring then
		fc := loadILBM(cp, ft_backGround);
		if fc = nil then
		    FindError := true;
		else
		    FindError := false;
		    iffLoadBackGround(fc*.fc_ptr);
		fi;
	    fi;
	incase ef_SetImage:
	    cp := pretend(p, *char);
	    while p* ~= 0 do
		p := p + 1;
		len := len - 1;
	    od;
	    p := p + 1;
	    len := len - 1;
	    fc := loadILBM(cp, ft_image);
	    if fc = nil then
		FindError := true;
	    else
		FindError := false;
	    fi;
	    DefaultImage := fc;
	incase ef_ShowImage:
	    cp := pretend(p, *char);
	    while p* ~= 0 do
		p := p + 1;
		len := len - 1;
	    od;
	    p := p + 1;
	    len := len - 1;
	    u := p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u2 := u;	/* imageX */
	    u := p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u3 := u;	 /* imageY */
	    u := p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u4 := u;	 /* imageWidth */
	    u := p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u5 := u;	 /* imageHeight */
	    u := p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u6 := u;	/* displayX */
	    u := p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u7 := u;	/* displayY */
	    u := p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u8 := u;	/* displayWidth */
	    u := p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    /* u is displayHeight */
	    len := len - 4 * 8;
	    if not ignoring then
		if cp* = '\e' then
		    fc := DefaultImage;
		else
		    fc := loadILBM(cp, ft_image);
		    FindError := fc = nil;
		fi;
		if fc ~= nil then
		    iffShowImage(fc*.fc_ptr, u2, u3, u4, u5, u6, u7, u8, u);
		fi;
	    fi;
	incase ef_ShowImagePixels:
	    cp := pretend(p, *char);
	    while p* ~= 0 do
		p := p + 1;
		len := len - 1;
	    od;
	    p := p + 1;
	    len := len - 1;
	    w := p*;
	    p := p + 1;
	    w := w << 8;
	    w := w | p*;
	    p := p + 1;
	    w2 := p*;
	    p := p + 1;
	    w2 := w2 << 8;
	    w2 := w2 | p*;
	    p := p + 1;
	    w3 := w;
	    w4 := w2;
	    w := p*;
	    p := p + 1;
	    w := w << 8;
	    w := w | p*;
	    p := p + 1;
	    w2 := p*;
	    p := p + 1;
	    w2 := w2 << 8;
	    w2 := w2 | p*;
	    p := p + 1;
	    w5 := w;
	    w6 := w2;
	    w := p*;
	    p := p + 1;
	    w := w << 8;
	    w := w | p*;
	    p := p + 1;
	    w2 := p*;
	    p := p + 1;
	    w2 := w2 << 8;
	    w2 := w2 | p*;
	    p := p + 1;
	    len := len - (2 + 2 + 2 + 2 + 2 + 2);
	    if not ignoring then
		if cp* = '\e' then
		    fc := DefaultImage;
		else
		    fc := loadILBM(cp, ft_image);
		    FindError := fc = nil;
		fi;
		if fc ~= nil then
		    /* Do *not* want to trim width/height here. */
		    iffShowImagePixels(fc*.fc_ptr, w3, w4, w5, w6, w, w2);
		fi;
	    fi;
	incase ef_ShowBrush:
	    cp := pretend(p, *char);
	    while p* ~= 0 do
		p := p + 1;
		len := len - 1;
	    od;
	    p := p + 1;
	    len := len - 1;
	    w := p*;
	    p := p + 1;
	    w := w << 8;
	    w := w | p*;
	    p := p + 1;
	    w2 := p*;
	    p := p + 1;
	    w2 := w2 << 8;
	    w2 := w2 | p*;
	    p := p + 1;
	    len := len - (2 + 2);
	    if not ignoring then
		fc := loadILBM(cp, ft_brush);
		if fc = nil then
		    FindError := true;
		else
		    FindError := false;
		    iffShowBrush(fc*.fc_ptr, w, w2);
		fi;
	    fi;
	incase ef_ScrollRectangle:
	    w := p*;
	    p := p + 1;
	    w := w << 8;
	    w := w | p*;
	    p := p + 1;
	    w2 := p*;
	    p := p + 1;
	    w2 := w2 << 8;
	    w2 := w2 | p*;
	    p := p + 1;
	    w3 := w;
	    w4 := w2;
	    w := p*;
	    p := p + 1;
	    w := w << 8;
	    w := w | p*;
	    p := p + 1;
	    w2 := p*;
	    p := p + 1;
	    w2 := w2 << 8;
	    w2 := w2 | p*;
	    p := p + 1;
	    w5 := w;
	    w6 := w2;
	    w := p*;
	    p := p + 1;
	    w := w << 8;
	    w := w | p*;
	    p := p + 1;
	    w2 := p*;
	    p := p + 1;
	    w2 := w2 << 8;
	    w2 := w2 | p*;
	    p := p + 1;
	    len := len - (2 + 2 + 2 + 2 + 2 + 2);
	    if not ignoring then
		/* Reduce these, to get outer pixel values for region */
		if w ~= 0 then
		    w := w - 1;
		fi;
		if w2 ~= 0 then
		    w2 := w2 - 1;
		fi;
		w := w + w5;
		w2 := w2 + w6;
		if w <= GraphicsWidth and w2 <= GraphicsHeight then
		    /* Need to set BgPen, since that is what ScrollRaster
		       fills vacated space with. */
		    savedPen := rp*.rp_BgPen;
		    if MapColours then
			setGraphicsBPen(ColourMapping[0]);
		    else
			setGraphicsBPen(0);
		    fi;
		    ScrollRaster(rp, make(w3, int), make(w4, int),
				 w5 + GXOffset, w6 + GYOffset,
				 w + GXOffset, w2 + GYOffset);
		    SetBPen(rp, savedPen);
		fi;
	    fi;
	incase ef_SetIconPen:
	    w := p*;
	    p := p + 1;
	    w := w << 8;
	    w := w | p*;
	    p := p + 1;
	    len := len - 2;
	    if not ignoring then
		if w < ColourMax then
		    if MapColours then
			w := ColourMapping[w];
		    fi;
		    if w ~= IconPen then
			IconPen := w;
			undrawIcons();
			redrawIcons();
		    fi;
		fi;
	    fi;
	incase ef_NewIcon:
	    u := p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    cp := pretend(p, *char);
	    p := p + sizeof(IconData_t);
	    len := len - (4 + sizeof(IconData_t));
	    if not ignoring then
		newIcon(u, pretend(cp, *byte));
	    fi;
	incase ef_ShowIcon:
	    u := p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    b2 := p*;
	    p := p + 1;
	    len := len - (4 + 1);
	    if not ignoring then
		showIcon(u, b2 + false);
	    fi;
	incase ef_RemoveIcon:
	    u := p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    len := len - 4;
	    if not ignoring then
		removeIcon(u);
	    fi;
	incase ef_DeleteIcon:
	    u := p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    len := len - 4;
	    if not ignoring then
		deleteIcon(u);
	    fi;
	incase ef_ResetIcons:
	    if not ignoring then
		resetIcons();
	    fi;
	incase ef_RedrawIcons:
	    if not ignoring then
		redrawIcons();
	    fi;
	incase ef_UndrawIcons:
	    if not ignoring then
		undrawIcons();
	    fi;
	incase ef_SoundVolume:
	    w := p*;
	    p := p + 1;
	    w := w << 8;
	    w := w | p*;
	    p := p + 1;
	    len := len - 2;
	    if not ignoring then
		setSoundVolume(w);
	    fi;
	incase ef_PlaySound:
	    cp := pretend(p, *char);
	    while p* ~= 0 do
		p := p + 1;
		len := len - 1;
	    od;
	    p := p + 1;
	    u := p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    w2 := p*;
	    p := p + 1;
	    w2 := w2 << 8;
	    w2 := w2 | p*;
	    p := p + 1;
	    len := len - (1 + 4 + 2);
	    if not ignoring then
		fc := load8SVX(cp);
		if fc = nil then
		    FindError := true;
		else
		    FindError := false;
		    playSound(fc*.fc_ptr, fc, u, w2);
		fi;
	    fi;
	incase ef_Params:
	    if ignoring then
		p := p + (2 + 2 + 1 + 1 + 1);
	    else
		w := p*;
		p := p + 1;
		w := w << 8;
		w := w | p*;
		p := p + 1;
		w3 := w;
		w := p*;
		p := p + 1;
		w := w << 8;
		w := w | p*;
		p := p + 1;
		w4 := w;
		b := p*;
		p := p + 1;
		b2 := p*;
		p := p + 1;
		setVoiceParams(w3, w4, b, b2, p*);
		p := p + 1;
	    fi;
	    len := len - (2 + 2 + 1 + 1 + 1);
	incase ef_VReset:
	    if not ignoring then
		resetVoice();
	    fi;
	incase ef_VoiceVolume:
	    w := p*;
	    p := p + 1;
	    w := w << 8;
	    w := w | p*;
	    p := p + 1;
	    len := len - 2;
	    if not ignoring then
		setVoiceVolume(w);
	    fi;
	incase ef_Narrate:
	    cp := pretend(p, *char);
	    w2 := 0;
	    while p* ~= 0 do
		p := p + 1;
		w2 := w2 + 1;
	    od;
	    len := len - w2;
	    p := p + 1;
	    u := p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    len := len - (1 + 4);
	    if not ignoring then
		doNarrate(cp, w2, u);
	    fi;
	incase ef_MusicVolume:
	    w := p*;
	    p := p + 1;
	    w := w << 8;
	    w := w | p*;
	    p := p + 1;
	    len := len - 2;
	    if not ignoring then
		setMusicVolume(w);
	    fi;
	incase ef_PlaySong:
	    cp := pretend(p, *char);
	    while p* ~= 0 do
		p := p + 1;
		len := len - 1;
	    od;
	    p := p + 1;
	    u := p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    u := u << 8;
	    u := u | p*;
	    p := p + 1;
	    len := len - (1 + 4);
	    if not ignoring then
		fc := loadSMUS(cp);
		if fc = nil then
		    FindError := true;
		else
		    FindError := false;
		    playMusic(fc*.fc_ptr, fc, u);
		fi;
	    fi;
	esac;
    od;
    Ignoring := ignoring;
corp;

/*
 * doEffects - top level handler for special effects.
 */

proc doEffects(*byte p; uint len)void:
    register *RastPort_t rp;
    uint savedX, savedY;
    ushort savedPen;

    rp := GraphicsRastPort;

    savedPen := rp*.rp_FgPen;
    savedX := rp*.rp_cp_x;
    savedY := rp*.rp_cp_y;
    SetAPen(rp, SavedPen);
    Move(rp, SavedX, SavedY);

    doEffects1(p, len);

    SavedPen := rp*.rp_FgPen;
    SavedX := rp*.rp_cp_x;
    SavedY := rp*.rp_cp_y;
    SetAPen(rp, savedPen);
    Move(rp, savedX, savedY);
corp;

/*
 * defineEffect - define a new effect routine.
 */

proc defineEffect(register ulong key; *byte data; uint len)void:
    register *EffectCache_t ec;
    register *ulong calls;
    register uint callsLen;
    register byte callsCount;

    ec := findEffect(key);
    /* Silently ignore the case where we already have this effect. This
       situation can happen because the server can get in a state where it
       doesn't think we have an effect when we do have it. This can happen
       if the user asks to flush an effect, and thus we send off a flushEffect
       request, WHILE the server is sending us something that uses that
       effect. The server will OK the request (as it always does) and return
       it to us, but we will not act upon it since we will have, inbetween
       the times, got the effect that references the one in question, and
       will have incremented that one's useCount. Got that? */
    if ec = nil then
	ec := mudAlloc(sizeof(EffectCache_t), 0x0);
	if ec ~= nil then
	    callsCount := data*;
	    data := data + 1;
	    len := len - 1;
	    callsLen := make(callsCount, uint) * sizeof(ulong);
	    ec*.ec_callsCount := callsCount;
	    if callsCount ~= 0 then
		calls := mudAlloc(callsLen, 0x0);
		if calls = nil then
		    FreeMem(ec, sizeof(EffectCache_t));
		    noMemory("effects refs");
		    return;
		fi;
		ec*.ec_calls := calls;
		BlockCopy(calls, data, callsLen);
		data := data + callsLen;
		len := len - callsLen;
	    fi;
	    ec*.ec_data := mudAlloc(len, 0x0);
	    if ec*.ec_data = nil then
		if callsCount ~= 0 then
		    FreeMem(calls, callsLen);
		fi;
		FreeMem(ec, sizeof(EffectCache_t));
		noMemory("effect data");
		return;
	    fi;
	    BlockCopy(ec*.ec_data, data, len);
	    ec*.ec_next := EffectCache;
	    ec*.ec_prev := nil;
	    ec*.ec_key := key;
	    ec*.ec_length := len;
	    ec*.ec_useCount := 0;
	    if EffectCache = nil then
		EffectCacheTail := ec;
	    else
		EffectCache*.ec_prev := ec;
	    fi;
	    EffectCache := ec;
	    while callsCount ~= 0 do
		callsCount := callsCount - 1;
		key := calls*;
		calls := calls + sizeof(ulong);
		ec := EffectCache;
		while ec ~= nil and ec*.ec_key ~= key do
		    ec := ec*.ec_next;
		od;
		if ec ~= nil then
		    ec*.ec_useCount := ec*.ec_useCount + 1;
		fi;
	    od;
	else
	    noMemory("effect");
	fi;
    fi;
corp;

/*
 * flushEffect - remove the indicated effect from the cache.
 */

proc flushEffect(ulong key)void:
    register *EffectCache_t ec;

    if key = 0 then
	freeAllEffects();
    else
	ec := findEffect(key);
	if ec ~= nil and ec*.ec_useCount = 0 then
	    freeEffect(ec);
	fi;
    fi;
corp;

/*
 * freeOneCachedEffect - try to free one effect to free up some space.
 */

proc freeOneCachedEffect()void:
    register *EffectCache_t ec;

    ec := EffectCacheTail;
    while ec ~= nil and ec*.ec_useCount ~= 0 do
	ec := ec*.ec_prev;
    od;
    if ec ~= nil then
	sendReleaseEffect(ec*.ec_key);
    fi;
corp;

/*
 * dumpEffectsCaches - dump the effects caches.
 */

proc dumpEffectsCaches()void:
    *FileCache_t fc;
    *EffectCache_t ec @ fc;
    *TileList_t tl @ fc;
    ushort savedPen;

    savedPen := GraphicsRastPort*.rp_FgPen;
    SetAPen(GraphicsRastPort, TextPen);
    putString("Files:\n");
    fc := FileCache;
    while fc ~= nil do
	putString("  ");
	putString(effectDir(fc*.fc_type));
	putChar('/');
	putString(fc*.fc_filePath);
	putString(" >");
	putUnsigned(fc*.fc_lockCount);
	putChar('<');
	newLine();
	fc := fc*.fc_next;
    od;
    putString("Effects:");
    ec := EffectCache;
    while ec ~= nil do
	putChar(' ');
	putUnsigned(ec*.ec_key);
	putChar('/');
	putUnsigned(ec*.ec_length);
	if ec*.ec_useCount ~= 0 then
	    putChar('>');
	    putUnsigned(ec*.ec_useCount);
	    putChar('<');
	fi;
	if ec*.ec_callsCount ~= 0 then
	    putChar('(');
	    putUnsigned(ec*.ec_callsCount);
	    putChar(')');
	fi;
	ec := ec*.ec_next;
    od;
    newLine();
    putString("Tiles:");
    tl := TileList;
    while tl ~= nil do
	putChar(' ');
	putUnsigned(tl*.tl_id);
	tl := tl*.tl_next;
    od;
    newLine();
    SetAPen(GraphicsRastPort, savedPen);
corp;

/*
 * handleEffectSignal - handle an asynchronous completion of an effect event.
 */

proc handleEffectSignal(ulong bits)void:

    handleSoundSignal(bits);
corp;

/*
 * lockEffectFile - bump the lock counter for an effect file.
 */

proc lockEffectFile(arbptr p)void:
    *FileCache_t fc @ p;

    fc*.fc_lockCount := fc*.fc_lockCount + 1;
corp;

/*
 * unlockEffectFile - decrement the lock counter for an effect file.
 */

proc unlockEffectFile(arbptr p)void:
    *FileCache_t fc @ p;

    fc*.fc_lockCount := fc*.fc_lockCount - 1;
corp;

/*
 * displayIntro - put up our introduction picture.
 */

proc displayIntro()void:
    *FileCache_t fc;

    fc := loadILBM("Intro", ft_backGround);
    if fc ~= nil then
	iffLoadBackGround(fc*.fc_ptr);
	WaitBlit();
	freeFileCache(fc);
	FileCache := nil;
	FileCacheTail := nil;
    fi;
corp;
