#drinc:exec/memory.g
#drinc:exec/nodes.g
#drinc:exec/ports.g
#drinc:exec/io.g
#drinc:graphics/gfx.g
#drinc:graphics/view.g
#drinc:graphics/rastport.g
#drinc:graphics/scale.g
#drinc:devices/audio.g
#drinc:hardware/blit.g
#drinc:util.g

#/Include/MUD.g
#graphics.g
#iff.g
#iffILBM.g
#iff8SVX.g
#iffSMUS.g
#sound.g
#funcs.g
#soundFuncs.g
#globals.g

/*
 * Amiga MUD
 *
 * Copyright (c) 1997 by Chris Gray
 */

/*
 * iff.d - top-level stuff for handling IFFs.
 */

type
    Image_t = struct {
	BitMap_t im_bitMap;
	PLANEPTR im_maskPlane;
	[MAX_COLOURS] uint im_colourMap;
	*uint im_origColourMap;
	uint im_colourCount;
	uint im_origColourCount;
	uint im_origDepth;
	uint im_width, im_height;
	ushort im_transparentColour;
	PictureType_t im_pictureType;
	bool im_hadBMHD, im_hadCMAP, im_hadBODY;
	BCompression_t im_compression;
	BMasking_t im_masking;
    };

bool CurrentIsILBM, CurrentIs8SVX;

proc readCat(*IFFContext_t context; IFFID_t subId)bool:

    IFFReadCatTail(context)
corp;

proc readList(*IFFContext_t context; IFFID_t subId)bool:

    IFFReadListTail(context)
corp;

proc readForm(*IFFContext_t context; IFFID_t subId)bool:
    bool wasILBM, was8SVX, res;

    wasILBM := CurrentIsILBM;
    was8SVX := CurrentIs8SVX;
    CurrentIsILBM := subId = ID_BILBM;
    CurrentIs8SVX := subId = ID_S8SVX;
    res := IFFReadFormTail(context);
    CurrentIsILBM := wasILBM;
    CurrentIs8SVX := was8SVX;
    res
corp;

proc readProp(*IFFContext_t context; IFFID_t subId)bool:

    IFFReadPropTail(context)
corp;

proc freeBitMap(register *BitMap_t bm; uint w, h)void:
    register uint i;

    for i from bm*.bm_Depth - 1 downto 0 do
	if bm*.bm_Planes[i] ~= nil then
	    FreeRaster(bm*.bm_Planes[i], w, h);
	    bm*.bm_Planes[i] := nil;
	fi;
    od;
corp;

proc allocBitMap(register *BitMap_t bm; uint d, w, h; *char desc)bool:
    register PLANEPTR p;
    register uint i;
    register bool ok;

    InitBitMap(bm, d, w, h);
    ok := true;
    for i from d - 1 downto 0 do
	while
	    p := AllocRaster(w, h);
	    p = nil and freeOneCachedFile()
	do
	od;
	if p = nil and ok then
	    noMemory(desc);
	    ok := false;
	fi;
	bm*.bm_Planes[i] := p;
    od;
    if not ok then
	freeBitMap(bm, w, h);
    fi;
    ok
corp;

proc readILBMOther(register *IFFContext_t context; IFFID_t id)bool:
    BitMapHeader_t bmh;
    register *Image_t im;
    register *BitMap_t bm;
    PLANEPTR p;
    register uint i;
    uint w, h;
    bool ok;

    im := pretend(context*.iff_client*.iff_userStructure, *Image_t);
    case id
    incase ID_BBMHD:
	if im*.im_hadBMHD then
	    /* We keep the first one - arbitrary choice! */
	    IFFSkipChunk(context)
	else
	    if IFFRead(context, &bmh, sizeof(BitMapHeader_t)) then
		im*.im_hadBMHD := true;
		im*.im_origDepth := bmh.bmh_nPlanes;
		im*.im_width := bmh.bmh_w;
		im*.im_height := bmh.bmh_h;
		im*.im_transparentColour := bmh.bmh_transparentColor;
		im*.im_compression := bmh.bmh_compression;
		im*.im_masking := bmh.bmh_masking;
		if im*.im_pictureType = pt_brush then
		    if bmh.bmh_masking = bmsk_hasMask or
			bmh.bmh_masking = bmsk_hasTransParentColour
		    then
			true
		    else
			IFFErrorContext(context,
					"ILBM BMHD no masking for brush");
			false
		    fi
		else
		    if bmh.bmh_masking = bmsk_hasMask then
			IFFErrorContext(context,
					"ILBM BMHD masking for non brush");
			false
		    else
			im*.im_masking := bmsk_none;
			true
		    fi
		fi
	    else
		IFFErrorContext(context, "ILBM BMHD read");
		false
	    fi
	fi
    incase ID_BCMAP:
	if im*.im_hadCMAP then
	    /* We keep the first one - arbitrary choice! */
	    IFFSkipChunk(context)
	else
	    if IFFReadBCMAP(context, &im*.im_origColourMap,
			    &im*.im_origColourCount)
	    then
		/* alloc IFFReadBCMAP to silently reject a map */
		im*.im_hadCMAP := im*.im_origColourMap ~= nil;
		true
	    else
		IFFErrorContext(context, "ILBM CMAP read");
		false
	    fi
	fi
    incase ID_BBODY:
	if CurrentIsILBM and not im*.im_hadBODY then
	    im*.im_hadBODY := true;
	    im*.im_maskPlane := nil;
	    bm := &im*.im_bitMap;
	    w := im*.im_width;
	    h := im*.im_height;
	    ok := allocBitMap(bm, im*.im_origDepth, w, h, "image plane");
	    if ok then
		if im*.im_masking ~= bmsk_none then
		    while
			p := AllocRaster(w, h);
			p = nil and freeOneCachedFile()
		    do
		    od;
		    if p = nil then
			noMemory("image mask");
			ok := false;
		    fi;
		    im*.im_maskPlane := p;
		fi;
	    fi;
	    if ok then
		ok := IFFReadBBODY(context, bm, im*.im_maskPlane, w, h,
				   im*.im_compression, im*.im_masking);
		if im*.im_masking = bmsk_hasTransParentColour then
		    IFFMakeMaskPlane(bm, p, w, im*.im_transparentColour);
		fi;
	    else
		ignore IFFSkipChunk(context);
	    fi;
	    if not ok then
		freeBitMap(bm, w, h);
		if im*.im_maskPlane ~= nil then
		    FreeRaster(im*.im_maskPlane, w, h);
		fi;
	    fi;
	    ok
	else
	    IFFSkipChunk(context)
	fi
    default:
	IFFSkipChunk(context)
    esac
corp;

proc iffFreeILBM(register arbptr ptr)void:
    register *Image_t im @ ptr;

    if im*.im_origColourMap ~= nil then
	FreeMem(im*.im_origColourMap, im*.im_origColourCount * sizeof(uint));
    fi;
    if im*.im_hadBODY then
	freeBitMap(&im*.im_bitMap, im*.im_width, im*.im_height);
	if im*.im_maskPlane ~= nil then
	    FreeRaster(im*.im_maskPlane, im*.im_width, im*.im_height);
	fi;
    fi;
    FreeMem(im, sizeof(Image_t));
corp;

proc iffDoRemap(*Image_t im)void:
    BitMap_t tempBitMap;
    *uint mapping;
    uint count;

    count := im*.im_origColourCount;
    mapping := mudAlloc(count * sizeof(uint), 0x0);
    if mapping = nil then
	noMemory("colour remap table");
	return;
    fi;
    iffCreateMapping(mapping, count, im*.im_origColourMap,
		     ColourCount, &im*.im_colourMap[0]);
    if im*.im_origDepth = Depth then
	/* Same depths - can do it in place */
	iffPerformMapping(mapping, &im*.im_bitMap, &im*.im_bitMap,
			  im*.im_width, im*.im_height);
    else
	/* Different depths - need a new BitMap_t of the proper depth. */
	if allocBitMap(&tempBitMap, Depth,
		       im*.im_width, im*.im_height, "colour remap bitmap")
	then
	    iffPerformMapping(mapping, &tempBitMap, &im*.im_bitMap,
			      im*.im_width, im*.im_height);
	    freeBitMap(&im*.im_bitMap, im*.im_width, im*.im_height);
	    im*.im_bitMap := tempBitMap;
	fi;
    fi;
    FreeMem(mapping, count * sizeof(uint));
corp;

proc iffRemapImage(register *Image_t im)void:
    register *uint p1, p2;
    register uint count, min, i;
    bool ok;

    getColourMap(&im*.im_colourMap[0], false);
    min := ColourCount;
    im*.im_colourCount := min;
    if (Mode = md_oneScreen or Mode = md_twoScreen) and
	im*.im_origDepth = Depth
    then
	if im*.im_origColourMap = nil then
	    /* Nothing to do - is right depth, and has no colourMap. */
	    return;
	fi;
	count := im*.im_origColourCount;
	if count > 1 << im*.im_origDepth then
	    count := 1 << im*.im_origDepth;
	fi;
	/* there might be fewer than ColourCount pens provided */
	ok := true;
	if im*.im_pictureType ~= pt_backGround then
	    /* If it is background, always use the new colourMap */
	    if count < min then
		min := count;
	    fi;
	    p1 := im*.im_origColourMap;
	    p2 := &im*.im_colourMap[0];
	    for i from min - 1 downto 0 do
		if p1* ~= p2* then
		    ok := false;
		fi;
		p1 := p1 + sizeof(uint);
		p2 := p2 + sizeof(uint);
	    od;
	fi;
	if ok then
	    /* Just copy over the new colourMap, then free it. */
	    BlockCopy(&im*.im_colourMap[0], im*.im_origColourMap,
		      count * sizeof(uint));
	    FreeMem(im*.im_origColourMap,
		    im*.im_origColourCount * sizeof(uint));
	    im*.im_origColourMap := nil;
	    im*.im_colourCount := count;
	    return;
	fi;
    else
	if im*.im_origColourMap = nil then
	    /* File had no colourMap, so just use ours */
	    if im*.im_origDepth > Depth then
		/* chop down any out-of-range pen specs */
		iffClipPens(&im*.im_bitMap, im*.im_width, im*.im_height);
	    fi;
	    return;
	fi;
    fi;
    /* Sigh, have to actually remap the colours */
    if im*.im_pictureType = pt_backGround then
	/* We are doing a background, and would normally end up loading its
	   colour map as the new colourmap. However, the current colour
	   map may not be as good as the default one for that purpose, so
	   use the default one instead of the current one. We can only do
	   this if we are not on the Workbench screen, however! */
	getColourMap(&im*.im_colourMap[0],
		     Mode = md_oneScreen or Mode = md_twoScreen);
    fi;
    iffDoRemap(im);
corp;

proc iffLoadILBM(*char name; PictureType_t pt)arbptr:
    register *Image_t im;
    IFFClient_t client;
    bool ok;

    im := mudAlloc(sizeof(Image_t), 0x0);
    if im ~= nil then
	client.iff_readForm := readForm;
	client.iff_readList := readList;
	client.iff_readCat := readCat;
	client.iff_readProp := readProp;
	client.iff_readOther := readILBMOther;
	client.iff_userStructure := im;
	im*.im_origColourMap := nil;
	im*.im_maskPlane := nil;
	im*.im_width := GraphicsWidth;
	im*.im_height := GraphicsHeight;
	im*.im_pictureType := pt;
	im*.im_hadBMHD := false;
	im*.im_hadCMAP := false;
	im*.im_hadBODY := false;
	im*.im_compression := bcmp_byteRun1;
	im*.im_masking := bmsk_none;
	CurrentIsILBM := false;
	if IFFReadFile(name, &client) then
	    if not im*.im_hadBODY then
		IFFErrorClient(&client, "no ILBM found for image");
		ok := false;
	    else
		iffRemapImage(im);
		ok := true;
	    fi;
	else
	    ok := false;
	fi;
	if not ok then
	    iffFreeILBM(im);
	    im := nil;
	fi;
    else
	noMemory("cached image");
    fi;
    im
corp;

proc doBitMapScale(*BitMap_t srcBitMap; uint imageWidth, imageHeight,
		   imageX, imageY, displayWidth, displayHeight,
		   displayX, displayY)void:
    BitScaleArgs_t bsa;
    BitMap_t tempBitMap;

    bsa.bsa_Flags := 0x0;
    bsa.bsa_SrcX := imageX;
    bsa.bsa_SrcY := imageY;
    bsa.bsa_SrcWidth := imageWidth;
    bsa.bsa_SrcHeight := imageHeight;
    bsa.bsa_XSrcFactor := imageWidth;
    bsa.bsa_YSrcFactor := imageHeight;
    bsa.bsa_XDestFactor := displayWidth;
    bsa.bsa_YDestFactor := displayHeight;
    bsa.bsa_SrcBitMap := srcBitMap;
    if Mode = md_oneWindow or Mode = md_twoWindow then
	if allocBitMap(&tempBitMap, Depth, displayWidth, displayHeight,
		       "scaling bitmap") then
	    bsa.bsa_DestX := 0;
	    bsa.bsa_DestY := 0;
	    bsa.bsa_DestBitMap := &tempBitMap;
	    BitMapScale(&bsa);
	    BltBitMapRastPort(&tempBitMap, 0, 0, GraphicsRastPort,
			      displayX + GXOffset, displayY + GYOffset,
			      displayWidth, displayHeight, 0xc0);
	    WaitBlit();
	    freeBitMap(&tempBitMap, displayWidth, displayHeight);
	fi;
    else
	bsa.bsa_DestX := displayX + GXOffset;
	bsa.bsa_DestY := displayY + GYOffset;
	bsa.bsa_DestBitMap := GraphicsRastPort*.rp_BitMap;
	BitMapScale(&bsa);
    fi;
corp;

proc iffLoadBackGround(register arbptr ptr)void:
    register *Image_t im @ ptr;
    register uint width, height, count @ height;
    register bool ok @ width;
    [MAX_COLOURS] uint colourMap;

    width := im*.im_width;
    height := im*.im_height;
    if not OS2 or width = GraphicsWidth and height = GraphicsHeight then
	if width > GraphicsWidth then
	    width := GraphicsWidth;
	fi;
	if height > GraphicsHeight then
	    height := GraphicsHeight;
	fi;
	BltBitMapRastPort(&im*.im_bitMap, 0, 0, GraphicsRastPort,
			  GXOffset, GYOffset, width, height, 0xc0);
    else
	doBitMapScale(&im*.im_bitMap, width, height, 0, 0,
		      GraphicsWidth, GraphicsHeight, 0, 0);
    fi;
    if im*.im_hadCMAP then
	if Mode = md_oneScreen or Mode = md_twoScreen then
	    getColourMap(&colourMap[0], false);
	    count := im*.im_colourCount;
	    ok := true;
	    while count ~= 0 and ok do
		count := count - 1;
		if colourMap[count] ~= im*.im_colourMap[count] then
		    ok := false;
		fi;
	    od;
	    if not ok then
		/* Palette will be different, so flush all cached images and
		   brushes that are based on that palette. */
		flushCachedImages();
	    fi;
	    setGraphicsPalette(&im*.im_colourMap[0], im*.im_colourCount);
	fi;
    fi;
corp;

proc showBitMap(*BitMap_t bm;
	uint imageX, imageY, imageWidth, imageHeight, fullWidth, fullHeight;
	register int displayX, displayY)void:

    if displayX < 0 then
	imageX := imageX - displayX;
	imageWidth := imageWidth + displayX;
	displayX := 0;
    fi;
    if displayY < 0 then
	imageY := imageY - displayY;
	imageHeight := imageHeight + displayY;
	displayY := 0;
    fi;
    if imageX >= fullWidth or imageY >= fullHeight then
	complain1("image not shown - image offset out of range");
    elif displayX >= make(GraphicsWidth, int) or
	displayY >= make(GraphicsHeight, int)
    then
	complain1("image not shown - display offset out of range");
    else
	if imageX + imageWidth > fullWidth then
	    imageWidth := fullWidth - imageX;
	fi;
	if imageY + imageHeight > fullHeight then
	    imageHeight := fullHeight - imageY;
	fi;
	if displayX + imageWidth > make(GraphicsWidth, int) then
	    imageWidth := GraphicsWidth - displayX;
	fi;
	if displayY + imageHeight > make(GraphicsHeight, int) then
	    imageHeight := GraphicsHeight - displayY;
	fi;
	if imageWidth = 0 or imageHeight = 0 then
	    complain1("image not shown - clipped region is empty");
	else
	    BltBitMapRastPort(bm, imageX, imageY, GraphicsRastPort,
			      displayX + GXOffset, displayY + GYOffset,
			      imageWidth, imageHeight, 0xc0);
	fi;
    fi;
corp;

proc iffShowImagePixels(register arbptr ptr;
			uint imageX, imageY, imageWidth, imageHeight;
			register int displayX, displayY)void:
    register *Image_t im @ ptr;

    showBitMap(&im*.im_bitMap, imageX, imageY, imageWidth, imageHeight,
	       im*.im_width, im*.im_height, displayX, displayY);
corp;

proc roundTo(long x; uint rang)uint:

    x := x * rang;
    if x < 0 then
	x := x - (1 << (FRAC_BITS - 1));
    else
	x := x + (1 << (FRAC_BITS - 1));
    fi;
    x / (1 << FRAC_BITS)
corp;

proc iffShowImage(register arbptr ptr;
		  long imageXL, imageYL, imageWidthL, imageHeightL,
		    displayXL, displayYL, displayWidthL, displayHeightL)void:
    *Image_t im @ ptr;
    uint imageX, imageY, imageWidth, imageHeight, displayWidth, displayHeight;
    int displayX, displayY;

    displayX := displayXL * GraphicsWidth / (1 << FRAC_BITS);
    if displayX < 0 then
	imageX := imageX - displayX;
	imageWidth := imageWidth + displayX;
	displayX := 0;
    fi;
    displayY := displayYL * GraphicsHeight / (1 << FRAC_BITS);
    if displayY < 0 then
	imageY := imageY - displayY;
	imageHeight := imageHeight + displayY;
	displayY := 0;
    fi;
    imageX := imageXL * im*.im_width / (1 << FRAC_BITS);
    imageY := imageYL * im*.im_height / (1 << FRAC_BITS);
    imageWidth := roundTo(imageWidthL, im*.im_width);
    imageHeight := roundTo(imageHeightL, im*.im_height);
    displayWidth := roundTo(displayWidthL, GraphicsWidth);
    displayHeight := roundTo(displayHeightL, GraphicsHeight);
    if imageX >= im*.im_width or imageY >= im*.im_height then
	complain1("image not shown - image offset out of range");
    elif displayX >= make(GraphicsWidth, int) or
	displayY >= make(GraphicsHeight, int)
    then
	complain1("image not shown - display offset out of range");
    else
	if imageX + imageWidth > im*.im_width then
	    imageWidth := im*.im_width - imageX;
	fi;
	if imageY + imageHeight > im*.im_height then
	    imageHeight := im*.im_height - imageY;
	fi;
	if displayX + displayWidth > make(GraphicsWidth, int) then
	    displayWidth := GraphicsWidth - displayX;
	fi;
	if displayY + displayHeight > make(GraphicsHeight, int) then
	    displayHeight := GraphicsHeight - displayY;
	fi;
	if imageWidth = 0 or imageHeight = 0 then
	    complain1("image not shown - clipped image region is empty");
	elif displayWidth = 0 or displayHeight = 0 then
	    complain1("image not shown - clipped display region is empty");
	else
	    if imageWidth = displayWidth and imageHeight = displayHeight then
		showBitMap(&im*.im_bitMap, imageX, imageY,
			   imageWidth, imageHeight,
			   im*.im_width, im*.im_height, displayX, displayY);
	    else
		if not OS2 then
		    /* Cannot use BitMapScale - it does not exist */
		    if displayWidth < imageWidth then
			imageWidth := displayWidth;
		    fi;
		    if displayHeight < imageHeight then
			imageHeight := displayHeight;
		    fi;
		    showBitMap(&im*.im_bitMap, imageX, imageY,
			       imageWidth, imageHeight,
			       im*.im_width, im*.im_height,
			       displayX, displayY);
		else
		    doBitMapScale(&im*.im_bitMap, imageWidth, imageHeight,
				  imageX, imageY, displayWidth, displayHeight,
				  displayX, displayY);
		fi;
	    fi;
	fi;
    fi;
corp;

proc iffShowBrush(register arbptr ptr; register int displayX, displayY)void:
    register *Image_t im @ ptr;
    uint xOffset, yOffset, width, height;

    if im*.im_maskPlane = nil then
	showBitMap(&im*.im_bitMap, 0, 0, im*.im_width, im*.im_height,
		   im*.im_width, im*.im_height, displayX, displayY);
    else
	xOffset := 0;
	yOffset := 0;
	width := im*.im_width;
	height := im*.im_height;
	if displayX < 0 then
	    xOffset := xOffset - displayX;
	    width := width + displayX;
	    displayX := 0;
	fi;
	if displayY < 0 then
	    yOffset := yOffset - displayY;
	    height := height + displayY;
	    displayY := 0;
	fi;
	if displayX >= make(GraphicsWidth, int) or
	    displayY >= make(GraphicsHeight, int)
	then
	    complain1("brush not shown - display offset out of range");
	elif xOffset >= width or yOffset >= height then
	    complain1("brush not shown - brush offset out of range");
	else
	    if displayX + width > make(GraphicsWidth, int) then
		width := GraphicsWidth - displayX;
	    fi;
	    if displayY + height > make(GraphicsHeight, int) then
		height := GraphicsHeight - displayY;
	    fi;
	    if width = 0 or height = 0 then
		complain1("brush not shown - clipped region empty");
	    else
		BltMaskBitMapRastPort(&im*.im_bitMap, xOffset, yOffset,
				      GraphicsRastPort,
				      displayX + GXOffset, displayY + GYOffset,
				      width, height, ABC | ABNC | ANBC,
				      im*.im_maskPlane);
	    fi;
	fi;
    fi;
corp;

proc iffReadATAKRLSE(register *IFFContext_t context;
		     *uint pLength; **EGPoint_t pPoints)bool:
    register uint count;
    register *EGPoint_t points;

    count := context*.iff_totalRemaining / sizeof(EGPoint_t);
    if count * sizeof(EGPoint_t) ~= context*.iff_totalRemaining then
	IFFErrorContext(context, "ATAK/RLSE data length");
	return(false);
    fi;
    if pPoints* ~= nil then
	FreeMem(pPoints*, pLength*);
	pPoints* := nil;
	pLength* := 0;
    fi;
    points := mudAlloc(context*.iff_totalRemaining, 0x0);
    if points = nil then
	noMemory("instrument ATAK/RLSE");
	return(false);
    fi;
    pPoints* := points;
    pLength* := count;
    IFFRead(context, points, context*.iff_totalRemaining)
corp;

proc read8SVXOther(register *IFFContext_t context; IFFID_t id)bool:
    register *Sound_t sn;
    register *short buffer;
    ulong len;

    sn := pretend(context*.iff_client*.iff_userStructure, *Sound_t);
    case id
    incase ID_SVHDR:
	IFFRead(context, &sn*.sn_v8h, sizeof(Voice8Header_t))
    incase ID_SBODY:
	if CurrentIs8SVX then
	    sn*.sn_hadBODY := true;
	    len := sn*.sn_v8h.v8h_oneShotHiSamples +
		   sn*.sn_v8h.v8h_repeatHiSamples;
	    len := len * ((1 << sn*.sn_v8h.v8h_ctOctave) - 1);
	    buffer := mudAlloc(len, MEMF_CHIP);
	    sn*.sn_data := buffer;
	    sn*.sn_bufferLength := len;
	    if buffer = nil then
		noMemory("sound samples");
		false
	    elif IFFReadSBODY(context, buffer, &len, &sn*.sn_v8h) then
		sn*.sn_sampleLength := len;
		true
	    else
		IFFErrorContext(context, "8SVX other BODY read");
		false
	    fi
	else
	    IFFSkipChunk(context)
	fi
    incase ID_SATAK:
	iffReadATAKRLSE(context, &sn*.sn_attackLength, &sn*.sn_attack)
    incase ID_SRLSE:
	iffReadATAKRLSE(context, &sn*.sn_releaseLength, &sn*.sn_release)
    default:
	IFFSkipChunk(context)
    esac
corp;

proc iffFree8SVX(register arbptr p)void:
    register *Sound_t sn @ p;

    if sn*.sn_attack ~= nil then
	FreeMem(sn*.sn_attack, sn*.sn_attackLength * sizeof(EGPoint_t));
    fi;
    if sn*.sn_release ~= nil then
	FreeMem(sn*.sn_release, sn*.sn_releaseLength * sizeof(EGPoint_t));
    fi;
    if sn*.sn_data ~= nil then
	FreeMem(sn*.sn_data, sn*.sn_bufferLength);
    fi;
    FreeMem(sn, sizeof(Sound_t));
corp;

proc iffLoad8SVX(*char name)arbptr:
    register *Sound_t sn;
    IFFClient_t client;
    bool ok;

    sn := mudAlloc(sizeof(Sound_t), 0x0);
    if sn ~= nil then
	client.iff_readForm := readForm;
	client.iff_readList := readList;
	client.iff_readCat := readCat;
	client.iff_readProp := readProp;
	client.iff_readOther := read8SVXOther;
	client.iff_userStructure := sn;
	sn*.sn_data := nil;
	sn*.sn_hadBODY := false;
	sn*.sn_v8h.v8h_oneShotHiSamples := 0;
	sn*.sn_v8h.v8h_repeatHiSamples := 0;
	sn*.sn_v8h.v8h_samplesPerHiCycle := 0;
	sn*.sn_v8h.v8h_samplesPerSecond := 8363;
	sn*.sn_v8h.v8h_ctOctave := 0;
	sn*.sn_v8h.v8h_compression := scmp_none;
	sn*.sn_v8h.v8h_volume := UNITY;
	sn*.sn_attackLength := 0;
	sn*.sn_releaseLength := 0;
	sn*.sn_attack := nil;
	sn*.sn_release := nil;
	if IFFReadFile(name, &client) then
	    if sn*.sn_hadBODY then
		ok := true;
	    else
		IFFErrorClient(&client, "no 8SVX found for sound");
		ok := false;
	    fi;
	else
	    ok := false;
	fi;
	if not ok then
	    iffFree8SVX(sn);
	    sn := nil;
	else
	    if sn*.sn_v8h.v8h_samplesPerSecond = 0 then
		sn*.sn_v8h.v8h_samplesPerSecond := 8363;
	    fi;
	    if sn*.sn_v8h.v8h_volume = 0 then
		sn*.sn_v8h.v8h_volume := UNITY;
	    fi;
	fi;
    else
	noMemory("cached sound");
    fi;
    sn
corp;

proc readSMUSOther(register *IFFContext_t context; IFFID_t id)bool:
    register *Music_t mu;
    register *Instrument_t ins;
    register *Track_t tr @ ins;
    RefInstrument_t ri;

    mu := pretend(context*.iff_client*.iff_userStructure, *Music_t);
    case id
    incase ID_MSHDR:
	IFFRead(context, &mu*.mu_ssh, sizeof(SScoreHeader_t))
    incase ID_MINS1:
	if mu*.mu_instrumentCount = MAX_INSTRUMENTS then
	    IFFSkipChunk(context)
	else
	    if IFFRead(context, &ri, sizeof(RefInstrument_t)-sizeof(char)) then
		ins := &mu*.mu_instruments[mu*.mu_instrumentCount];
		ins*.ins_nameLen := context*.iff_totalRemaining + 1;
		ins*.ins_name := mudAlloc(ins*.ins_nameLen, 0x0);
		if ins*.ins_name = nil then
		    noMemory("instrument name");
		    false
		else
		    mu*.mu_instrumentCount := mu*.mu_instrumentCount + 1;
		    IFFRead(context, ins*.ins_name,context*.iff_totalRemaining)
		fi
	    else
		IFFErrorContext(context, "SMUS INS1 read");
		false
	    fi
	fi
    incase ID_MTRAK:
	if mu*.mu_trackCount = MAX_TRACKS then
	    IFFSkipChunk(context)
	else
	    tr := &mu*.mu_tracks[mu*.mu_trackCount];
	    tr*.tr_length := context*.iff_totalRemaining / sizeof(SEvent_t);
	    tr*.tr_events := mudAlloc(tr*.tr_length * sizeof(SEvent_t), 0x0);
	    if tr*.tr_events = nil then
		noMemory("music TRAK data");
		false
	    else
		mu*.mu_trackCount := mu*.mu_trackCount + 1;
		tr*.tr_instrument := 0;
		if IFFRead(context, tr*.tr_events,
			   tr*.tr_length * sizeof(SEvent_t))
		then
		    true
		else
		    IFFErrorContext(context, "SMUS TRAK read");
		    false
		fi
	    fi
	fi
    default:
	IFFSkipChunk(context)
    esac
corp;

proc iffFreeSMUS(register arbptr p)void:
    register *Music_t mu @ p;
    register *Instrument_t ins;
    register *Track_t tr @ ins;
    register uint i;

    ins := &mu*.mu_instruments[0];
    i := 0;
    while i ~= mu*.mu_instrumentCount do
	FreeMem(ins*.ins_name, ins*.ins_nameLen);
	ins := ins + sizeof(Instrument_t);
	i := i + 1;
    od;
    tr := &mu*.mu_tracks[0];
    i := 0;
    while i ~= mu*.mu_trackCount do
	FreeMem(tr*.tr_events, tr*.tr_length * sizeof(SEvent_t));
	tr := tr + sizeof(Track_t);
	i := i + 1;
    od;
    FreeMem(mu, sizeof(Music_t));
corp;

proc iffLoadSMUS(*char name)arbptr:
    IFFClient_t client;
    *Music_t mu;

    mu := mudAlloc(sizeof(Music_t), 0x0);
    if mu ~= nil then
	mu*.mu_instrumentCount := 0;
	mu*.mu_trackCount := 0;
	mu*.mu_ssh.ssh_tempo := 240 * 128;
	mu*.mu_ssh.ssh_volume := 60;
	mu*.mu_ssh.ssh_ctTrack := 0;
	client.iff_readForm := readForm;
	client.iff_readList := readList;
	client.iff_readCat := readCat;
	client.iff_readProp := readProp;
	client.iff_readOther := readSMUSOther;
	client.iff_userStructure := mu;
	if not IFFReadFile(name, &client) then
	    iffFreeSMUS(mu);
	    mu := nil;
	fi;
    else
	noMemory("cached music");
    fi;
    mu
corp;
