#drinc:libraries/dos.g
#iff.g
#funcs.g
#globals.g

/*
 * Amiga MUD
 *
 * Copyright (c) 1997 by Chris Gray
 */

/*
 * IFFRead.d - general IFF input code.
 */

/*
 * IFFRead - attempt to read some bytes from an IFF context.
 */

proc IFFErrorClient(*IFFClient_t client; *char m)void:
    complain3(client*.iff_fileName, ": ", m);
corp;

proc IFFErrorContext(*IFFContext_t context; *char m)void:
    complain3(context*.iff_client*.iff_fileName, ": ", m);
corp;

proc IFFRead(register *IFFContext_t context; arbptr buffer;
	     register ulong length)bool:
    register *IFFClient_t client;
    register *byte p1, p2 @ context;
    register ulong actual;

    if length > context*.iff_totalRemaining then
	IFFErrorContext(context, "length");
	return(false);
    fi;
    context*.iff_totalRemaining := context*.iff_totalRemaining - length;
    context*.iff_totalRead := context*.iff_totalRead + length;
    client := context*.iff_client;
    if client*.iff_remaining ~= 0 then
	if client*.iff_remaining > length then
	    actual := length;
	    client*.iff_remaining := client*.iff_remaining - length;
	else
	    actual := client*.iff_remaining;
	    client*.iff_remaining := 0;
	fi;
	p1 := &client*.iff_data[client*.iff_position];
	p2 := buffer;
	client*.iff_position := client*.iff_position + actual;
	length := length - actual;
	buffer := pretend(buffer, *byte) + actual;
	while actual ~= 0 do
	    actual := actual - 1;
	    p2* := p1*;
	    p2 := p2 + 1;
	    p1 := p1 + 1;
	od;
    fi;
    if length ~= 0 then
	if length >= IFF_BUFFER_SIZE then
	    if Read(client*.iff_file, buffer, length) ~= length then
		IFFErrorContext(context, "data read");
		return(false);
	    fi;
	else
	    client*.iff_position := 0;
	    client*.iff_remaining :=
		Read(client*.iff_file, &client*.iff_data[0], IFF_BUFFER_SIZE);
	    if client*.iff_remaining < length then
		IFFErrorContext(context, "data read");
		return(false);
	    fi;
	    p1 := &client*.iff_data[0];
	    p2 := buffer;
	    client*.iff_position := length;
	    client*.iff_remaining := client*.iff_remaining - length;
	    while length ~= 0 do
		length := length - 1;
		p2* := p1*;
		p2 := p2 + 1;
		p1 := p1 + 1;
	    od;
	fi;
    fi;
    true
corp;

/*
 * IFFSkip - skip some bytes.
 */

proc IFFSkip(register *IFFContext_t context; register ulong amount)bool:
    register *IFFClient_t client;

    if amount > context*.iff_totalRemaining then
	IFFErrorContext(context, "skip amount");
	return(false);
    fi;
    context*.iff_totalRemaining := context*.iff_totalRemaining - amount;
    context*.iff_totalRead := context*.iff_totalRead + amount;
    client := context*.iff_client;
    if amount <= client*.iff_remaining then
	client*.iff_remaining := client*.iff_remaining - amount;
	client*.iff_position := client*.iff_position + amount;
    else
	if Seek(client*.iff_file, amount - client*.iff_remaining,
		OFFSET_CURRENT) = -1
	then
	    IFFErrorContext(context, "skip Seek");
	    return(false);
	fi;
	client*.iff_remaining := 0;
	client*.iff_position := 0;
    fi;
    true
corp;

/*
 * IFFSkipChunk - skip the current chunk.
 */

proc IFFSkipChunk(*IFFContext_t context)bool:

    IFFSkip(context, context*.iff_totalRemaining)
corp;

/*
 * iffReadCompound - common little utility to save code-space.
 */

proc iffReadCompound(register *IFFClient_t client;
		     register *IFFContext_t context; IFFID_t id, subId)bool:

    case id
    incase ID_FORM:
	client*.iff_readForm(context, subId)
    incase ID_LIST:
	client*.iff_readList(context, subId)
    incase ID_CAT:
	client*.iff_readCat(context, subId)
    default:
	IFFErrorClient(client, "unknown compound");
	false
    esac
corp;

/*
 * IFFReadFile - open and read an IFF file.
 */

proc IFFReadFile(*char fileName; register *IFFClient_t client)bool:
    IFFContext_t context;
    IFFHeader2_t header;
    bool ok;

    client*.iff_fileName := fileName;
    client*.iff_file := Open(fileName, MODE_READONLY);
    if client*.iff_file = 0 then
	if not EffectsQuiet then
	    IFFErrorClient(client, "can't open");
	fi;
	return(false);
    fi;
    context.iff_client := client;
    context.iff_totalRemaining := sizeof(IFFHeader2_t);
    context.iff_totalRead := 0;
    client*.iff_position := 0;
    client*.iff_remaining := 0;
    ok := true;
    if IFFRead(&context, &header, sizeof(IFFHeader2_t)) then
	context.iff_parent := nil;
	/* allow for junk at the end of the file */
	context.iff_totalRemaining :=
		header.iff_header.iff_chunkLength - sizeof(IFFID_t);
	ok := iffReadCompound(client, &context, header.iff_header.iff_id,
			      header.iff_subId);
	if ok then
	    if context.iff_totalRemaining ~= 0 then
		IFFErrorClient(client, "FORM/LIST/CAT top level structure");
		ok := false;
	    fi;
	else
	    IFFErrorClient(client, "FORM/LIST/CAT top level read");
	fi;
    else
	IFFErrorClient(client, "file header");
	ok := false;
    fi;
    Close(client*.iff_file);
    ok
corp;

/*
 * IFFReadGroupClose - end of a recursive group for reading.
 */

proc IFFReadGroupClose(register *IFFContext_t context)bool:
    register *IFFContext_t parent;

    if context*.iff_totalRemaining = 0 then
	parent := context*.iff_parent;
	if parent ~= nil then
	    parent*.iff_totalRemaining :=
		parent*.iff_totalRemaining - context*.iff_totalRead;
	    parent*.iff_totalRead :=
		parent*.iff_totalRead + context*.iff_totalRead;
	fi;
	true
    else
	IFFErrorContext(context, "FORM/LIST/CAT end");
	false
    fi
corp;

/*
 * IFFReadCatTail - read the remainder of an IFF CAT
 */

proc IFFReadCatTail(register *IFFContext_t parent)bool:
    IFFHeader2_t header;
    IFFContext_t child;
    register *IFFClient_t client;

    client := parent*.iff_client;
    child.iff_parent := parent;
    child.iff_client := client;
    while parent*.iff_totalRemaining ~= 0 do
	if not IFFRead(parent, &header, sizeof(IFFHeader2_t)) then
	    IFFErrorContext(parent, "CAT data read");
	    return(false);
	fi;
	if header.iff_header.iff_chunkLength - sizeof(IFFID_t) >
		parent*.iff_totalRemaining
	then
	    IFFErrorContext(parent, "CAT data length");
	    return(false);
	fi;
	child.iff_totalRemaining :=
		header.iff_header.iff_chunkLength - sizeof(IFFID_t);
	child.iff_totalRead := sizeof(IFFID_t);
	if not iffReadCompound(client, &child, header.iff_header.iff_id,
			       header.iff_subId)
	then
	    IFFErrorContext(parent, "CAT body");
	    return(false);
	fi;
    od;
    IFFReadGroupClose(parent)
corp;

/*
 * IFFReadListTail - read the remainder of an IFF LIST
 */

proc IFFReadListTail(register *IFFContext_t parent)bool:
    IFFHeader2_t header;
    IFFContext_t child;
    register *IFFClient_t client;
    register bool hadNonProp;

    hadNonProp := false;
    client := parent*.iff_client;
    child.iff_parent := parent;
    child.iff_client := client;
    while parent*.iff_totalRemaining ~= 0 do
	if not IFFRead(parent, &header, sizeof(IFFHeader2_t)) then
	    IFFErrorClient(client, "LIST header read");
	    return(false);
	fi;
	if header.iff_header.iff_chunkLength - sizeof(IFFID_t) >
		parent*.iff_totalRemaining
	then
	    IFFErrorClient(client, "LIST data length");
	    return(false);
	fi;
	child.iff_totalRemaining :=
		header.iff_header.iff_chunkLength - sizeof(IFFID_t);
	child.iff_totalRead := sizeof(IFFID_t);
	if header.iff_header.iff_id = ID_PROP then
	    if hadNonProp then
		IFFErrorClient(client, "LIST - PROP after non-PROP");
		return(false);
	    fi;
	    hadNonProp := true;
	    if not client*.iff_readProp(&child, header.iff_subId) then
		IFFErrorClient(client, "LIST PROP body read");
		return(false);
	    fi;
	else
	    if not iffReadCompound(client, &child, header.iff_header.iff_id,
				   header.iff_subId)
	    then
		IFFErrorClient(client, "LIST body read");
		return(false);
	    fi;
	fi;
    od;
    IFFReadGroupClose(parent)
corp;

/*
 * IFFReadFormTail - read the remainder of an IFF FORM
 */

proc IFFReadFormTail(register *IFFContext_t parent)bool:
    IFFHeader_t header;
    IFFContext_t child;
    IFFID_t subId;
    register *IFFContext_t childPtr;
    register *IFFClient_t client;

    client := parent*.iff_client;
    childPtr := &child;
    child.iff_parent := parent;
    child.iff_client := client;
    while parent*.iff_totalRemaining ~= 0 do
	if not IFFRead(parent, &header, sizeof(IFFHeader_t)) then
	    IFFErrorClient(client, "FORM header read");
	    return(false);
	fi;
	if header.iff_chunkLength > parent*.iff_totalRemaining then
	    IFFErrorClient(client, "FORM data length");
	    return(false);
	fi;
	child.iff_totalRemaining := header.iff_chunkLength;
	child.iff_totalRead := 0;
	case header.iff_id
	incase ID_FORM:
	    if not IFFRead(childPtr, &subId, sizeof(IFFID_t)) or
		not client*.iff_readForm(childPtr, subId)
	    then
		IFFErrorClient(client, "FORM sub-FORM read");
		return(false);
	    fi;
	incase ID_LIST:
	    if not IFFRead(childPtr, &subId, sizeof(IFFID_t)) or
		not client*.iff_readList(childPtr, subId)
	    then
		IFFErrorClient(client, "FORM sub-LIST read");
		return(false);
	    fi;
	incase ID_CAT:
	    if not IFFRead(childPtr, &subId, sizeof(IFFID_t)) or
		not client*.iff_readCat(childPtr, subId)
	    then
		IFFErrorClient(client, "FORM sub-CAT read");
		return(false);
	    fi;
	incase ID_PROP:
	    if not IFFRead(childPtr, &subId, sizeof(IFFID_t)) or
		not client*.iff_readProp(childPtr, subId)
	    then
		IFFErrorClient(client, "FORM PROP read");
		return(false);
	    fi;
	default:
	    if not client*.iff_readOther(childPtr, header.iff_id) then
		IFFErrorClient(client, "FORM other header read");
		return(false);
	    fi;
	    if header.iff_chunkLength & 1 ~= 0 then
		child.iff_totalRemaining := child.iff_totalRemaining + 1;
		if not IFFSkip(childPtr, 1) then
		    IFFErrorClient(client, "FORM pad skip");
		    return(false);
		fi;
	    fi;
	    if not IFFReadGroupClose(childPtr) then
		IFFErrorClient(client, "FORM group close");
		return(false);
	    fi;
	esac;
    od;
    IFFReadGroupClose(parent)
corp;

/*
 * IFFReadPropTail - read the remainder of an IFF PROP
 */

proc IFFReadPropTail(register *IFFContext_t parent)bool:
    IFFHeader_t header;
    IFFContext_t child;
    register *IFFClient_t client;
    register *IFFContext_t childPtr;

    client := parent*.iff_client;
    childPtr := &child;
    child.iff_parent := parent;
    child.iff_client := client;
    while parent*.iff_totalRemaining ~= 0 do
	if not IFFRead(parent, &header, sizeof(IFFHeader_t)) then
	    IFFErrorClient(client, "PROP header read");
	    return(false);
	fi;
	if header.iff_chunkLength > parent*.iff_totalRemaining then
	    IFFErrorClient(client, "PROP data length");
	    return(false);
	else
	    child.iff_totalRemaining := header.iff_chunkLength;
	    child.iff_totalRead := 0;
	    if not client*.iff_readOther(childPtr, header.iff_id) then
		IFFErrorClient(client, "PROP body");
		return(false);
	    fi;
	    if header.iff_chunkLength & 1 ~= 0 then
		if not IFFSkip(childPtr, 1) then
		    IFFErrorClient(client, "PROP pad skip");
		    return(false);
		fi;
	    fi;
	    if not IFFReadGroupClose(childPtr) then
		IFFErrorClient(client, "PROP close");
		return(false);
	    fi;
	fi;
    od;
    IFFReadGroupClose(parent)
corp;
