#drinc:exec/miscellaneous.g
#drinc:exec/nodes.g
#drinc:exec/lists.g
#drinc:exec/memory.g
#drinc:exec/ports.g
#drinc:exec/tasks.g
#drinc:exec/libraries.g
#drinc:graphics/gfx.g
#drinc:graphics/gfxBase.g
#drinc:graphics/rastport.g
#drinc:intuition/intuitionBase.g
#drinc:intuition/miscellaneous.g
#drinc:intuition/intuiText.g
#drinc:intuition/requester.g
#drinc:intuition/window.g
#drinc:intuition/intuiMessage.g
#drinc:libraries/dos.g
#drinc:libraries/dosextens.g
#drinc:workbench/workbench.g
#drinc:workbench/icon.g
#drinc:workbench/startup.g
#drinc:util.g

#drinc:AmiTCP/socket.g

#/Include/MUD.g
#/Include/Request.g
#/Include/MUDLib.g
#/Include/MUDLibPrivate.g
#text.g
#globals.g
#funcs.g

/*
 * Amiga MUD
 *
 * Copyright (c) 1997 by Chris Gray
 */

/*
 * main.d - startup and mainline code.
 */

bool FORCE_CLI = true , FORCE_WB = false;

ulong
    STACK_NEEDED = 10000,
    INPUT_BUFFER_SIZE = 512,
    CMD_BUF_LENGTH = 100,
    NAME_BUFFER_LEN = 100,
    DIAL_STRING_LEN = 256,
    DEFAULT_EDITOR_STACK = 9996,
    DEFAULT_OUTPUT_HISTORY_SIZE = 5000,
    DEFAULT_INPUT_HISTORY_SIZE = 1024;

/* These are in /Include/types.g, but we do not include that. */
ulong
    BOOLEAN_FALSE = 0,
    BOOLEAN_TRUE = 1;

uint DEFAULT_IP_PORT = 6667;

*char EDIT_FILE_NAME = "T:MUD.edit.TMP";
ulong EDIT_FILE_NAME_LEN = 14;
ulong EDIT_OVERHEAD = 14;

uint ERROR_LEN = 150;
ulong DEFAULT_RETRY_LIMIT = 10;

MUDState_t MS;

*Task_t MyTask;
*Process_t MyProcess;
*Window_t SaveWindowPtr;
*MsgPort_t ServerPort, MyPort;
*Request_t InitialRequest;
Handle_t StdOut, EditFH;
ulong ChildSignal, ChildSignalBit;
ulong EditorStackSize;

ulong OutputHistorySize, InputHistorySize;

ulong ErrorPos;
ulong MUDFlags;
ulong SerialUnit, RetryLimit, SerialBits, ServerBit, EffectsBits;

*WBStartup_t StartupMessage;
*char ForceCharacterName;
*char ForcePassword;

ulong EditMax, EditPos;

*Request_t CurrentRq;
Request_t RqBuf;
ulong HeaderLen, Expecting;
long IPSocket;
uint IPPort;
bool MoreHeader;

[ERROR_LEN] char ErrorBuffer;

[INPUT_BUFFER_SIZE] char EditBuffer;
[CMD_BUF_LENGTH] char CmdBuffer;

bool FromWorkBench, ShuttingDown, EditError, EditReading, EditFilling,
    EditInternal, EditParseError, HadErrorContinue, HadErrorSkip,
    EffectsInited, EditFirst, CloseWorkbench, EditFailing, InitedLib,
    FullyRunning, HadParmError, UseIP, SerialShared, Serial7Wire, Bright, Dim;

bool Serial, SerialReady, SerialOpen;
HistoryType_t HistoryType;
ScrollType_t ScrollType;

[NAME_BUFFER_LEN] char SerialDevice;
[NAME_BUFFER_LEN] char PortName;
[NAME_BUFFER_LEN] char IPHost;

[DIAL_STRING_LEN] char DialString;
[PLAYER_NAME_LEN] char CharacterName;
[PASSWORD_LEN] char Password;

proc FD_ZERO(*FDSet_t fds)void:
    register uint i;
    register *ulong p;

    p := &fds*.fds_bits[0];
    for i from NFDBITS - 1 downto 0 do
	p* := 0x0;
	p := p + sizeof(ulong);
    od;
corp;

proc FD_SET(*FDSet_t fds; uint n)void:
    *ulong p;

    p := &fds*.fds_bits[n / NFDBITS];
    p* := p* | (1 << (n % NFDBITS));
corp;

/*
proc FD_CLR(*FDSet_t fds; uint n)void:
    *ulong p;

    p := &fds*.fds_bits[n / NFDBITS];
    p* := p* & ~(1 << (n % NFDBITS));
corp;
*/

proc FD_ISSET(*FDSet_t fds; uint n)bool:

    (fds*.fds_bits[n / NFDBITS] & (1 << (n % NFDBITS))) ~= 0x0
corp;

/*
 * errorChar - routine to write a single character of an error/debug message.
 */

proc errorChar(char ch)void:
    ulong MIN_WIDTH = 410;
    ulong len;
    IntuiText_t bodyText, cancelText;

    if ch = '\n' then
	if FORCE_WB or FromWorkBench or not FORCE_CLI and TextWindow ~= nil
	then
	    ErrorBuffer[ErrorPos] := '\e';
	    bodyText.it_FrontPen := 0;
	    bodyText.it_BackPen := 1;
	    bodyText.it_DrawMode := JAM1;
	    bodyText.it_LeftEdge := 8;
	    bodyText.it_TopEdge := 8;
	    bodyText.it_ITextFont := nil;
	    bodyText.it_IText := &ErrorBuffer[0];
	    bodyText.it_NextText := nil;
	    cancelText.it_FrontPen := 0;
	    cancelText.it_BackPen := 1;
	    cancelText.it_DrawMode := JAM1;
	    cancelText.it_LeftEdge := 6;
	    cancelText.it_TopEdge := 4;
	    cancelText.it_ITextFont := nil;
	    cancelText.it_IText := "ABORT";
	    cancelText.it_NextText := nil;
	    len := (ErrorPos + 4) * 8 + 4;
	    if len < MIN_WIDTH then
		len := MIN_WIDTH;
	    fi;
	    ignore AutoRequest(if FORCE_WB then nil else TextWindow fi,
			       &bodyText, nil, &cancelText, 0x0, 0x0, len, 52);
	else
	    ErrorBuffer[ErrorPos] := '\n';
	    ErrorPos := ErrorPos + 1;
	    ignore Write(StdOut, &ErrorBuffer[0], ErrorPos);
	fi;
	ErrorPos := 0;
    else
	if ErrorPos = ERROR_LEN - 2 then
	    ErrorBuffer[ERROR_LEN - 2] := '*';
	else
	    ErrorBuffer[ErrorPos] := ch;
	    ErrorPos := ErrorPos + 1;
	fi;
    fi;
corp;

/*
 * errorString - send a whole string of debug/error output.
 */

proc errorString(register *char st)void:

    while st* ~= '\e' do
	errorChar(st*);
	st := st + sizeof(char);
    od;
corp;

/*
 * errorNum - send an unsigned number to the output.
 */

proc errorNum(long n)void:

    if n < 0 then
	if n = 0x80000000 then
	    n := n + 1;
	fi;
	errorChar('-');
	errorNum(-n);
    else
	if n >= 10 then
	    errorNum(n / 10);
	fi;
	errorChar(n % 10 + '0');
    fi;
corp;

/*
 * errorHex - send a hex number to the output.
 */

proc errorHex(ulong n)void:
    register *char p;
    register uint i, nibble;
    [9] char buff;

    p := &buff[8];
    p* := '\e';
    for i from 8 - 1 downto 0 do
	p := p - sizeof(char);
	nibble := n & 0xf;
	p* := if nibble >= 10 then nibble - 10 + 'a' else nibble + '0' fi;
	n := n >> 4;
    od;
    errorString(&buff[0]);
corp;

/*
 * abort - something has gone wrong in our communications with the
 *	MUD server. We shut down. This routine must not return.
 */

proc abort()void:
    register *Request_t rq;

    if UseIP then
	ignore Shutdown(IPSocket, 2);
	ignore CloseSocket(IPSocket);
	CloseBsdsocketLibrary();
    else
	if not Serial then
	    DeletePort(MyPort);
	fi;
	if SerialOpen then
	    closeSerialHandler(&SerialDevice[0], SerialUnit);
	fi;
    fi;
    FreeSignal(ChildSignal);
    MyProcess*.pr_WindowPtr := SaveWindowPtr;
    /* may have done this early in shutDownAndAbort */
    if TextWindow ~= nil then
	intuitionTerm();
    fi;
    CloseMudLibrary();
    if InitialRequest ~= nil then
	FreeMem(InitialRequest, sizeof(Request_t) - TEXT_LENGTH);
    fi;
    CloseIconLibrary();
    CloseIntuitionLibrary();
    CloseGraphicsLibrary();
    if FromWorkBench then
	Forbid();
	ReplyMsg(&StartupMessage*.sm_Message);
    fi;
    CloseExecLibrary();
    Exit(RETURN_FAIL);
corp;

/*
 * doExit - allow others to request an exit here.
 */

proc doExit()void:

    if InitedLib then
	MUDAbort(&MS);
    fi;
    abort();
corp;

/*
 * handleControl_C - try to gracefully shutdown. On subsequent hit,
 *	be less graceful about it.
 */

proc handleControl_C()void:

    ignore SetSignal(0x0, SIGBREAKF_CTRL_C);
    if ShuttingDown then
	errorString("CNTL-C - aborting\n");
	doExit();
    else
/*
	removeMenus();
*/
	ShuttingDown := true;
	if FullyRunning then
	    MUDShutDown(&MS);
	fi;
    fi;
corp;

/*
 * doMUDSource - allow intuition code to try library sourcing.
 */

proc doMUDSource(*char fileName)bool:

    if FullyRunning then
	MUDSource(&MS, fileName)
    else
	false
    fi
corp;

/*
 * mudAlloc - try real hard to allocate some memory.
 */

proc mudAlloc(ulong length, flags)arbptr:
    arbptr p;

    if EffectsInited then
	while
	    p := AllocMem(length, flags);
	    p = nil and freeOneCachedFile()
	do
	od;
    else
	p := AllocMem(length, flags);
    fi;
    p
corp;

/*
 * complain1 - a single-string complaing about something.
 */

proc complain1(*char m)void:
    ushort savedPen;

    savedPen := GraphicsRastPort*.rp_FgPen;
    SetAPen(GraphicsRastPort, TextPen);
    putString("::");
    putString(m);
    putString("::\n");
    SetAPen(GraphicsRastPort, savedPen);
corp;

/*
 * complain3 - a three-string complaint about something.
 */

proc complain3(*char m1, m2, m3)void:
    ushort savedPen;

    savedPen := GraphicsRastPort*.rp_FgPen;
    SetAPen(GraphicsRastPort, TextPen);
    putString("::");
    putString(m1);
    putString(m2);
    if m3 ~= nil then
	putString(m3);
    fi;
    putString("::\n");
    SetAPen(GraphicsRastPort, savedPen);
corp;

/*
 * editFileComplain - complain about the edit temporary file.
 */

proc editFileComplain(*char m1, m3)void:

    if not EditFailing then
	complain3(m1, EDIT_FILE_NAME, m3);
	EditFailing := true;
    fi;
corp;

/*
 * noMemory - complain about lack of memory for something.
 */

proc noMemory(*char what)void:

    complain3("No memory for ", what, nil);
corp;

/*
 * editComplain - a one-string complaint about editing.
 */

proc editComplain(*char m)void:

    if not EditFailing then
	putString("::");
	putString(m);
	putString("::\n");
	EditFailing := true;
    fi;
corp;

/*
 * myAllocRequest - interface to allow the serial routines to allocate a
 *	Request_t for an incoming transaction.
 */

proc myAllocRequest(uint len)*Request_t:

    allocRequestE(&MS, len)
corp;

/*
 * myFreeRequest - similarly, allow the serial routines to free a request
 *	that has been transmitted.
 */

proc myFreeRequest(*Request_t rq)void:

    freeRequestE(&MS, rq);
corp;

/*
 * getVar - get the value of a DOS variable.
 */

proc getVar(*char varName, buffer; ulong buffLen)bool:
    [50] char nameBuffer;
    extern GetVar(*char name, buffer; ulong size, flags)long;
    long len;
    *Window_t saveWindow;
    Handle_t fh;

    if OS2 then
	GetVar(varName, buffer, buffLen, 0x0) > 0
    else
	CharsCopy(&nameBuffer[0], "ENV:");
	CharsCopy(&nameBuffer[4], varName);
	saveWindow := MyProcess*.pr_WindowPtr;
	MyProcess*.pr_WindowPtr := pretend(-1, *Window_t);
	fh := Open(&nameBuffer[0], MODE_OLDFILE);
	MyProcess*.pr_WindowPtr := saveWindow;
	if fh ~= 0 then
	    len := Read(fh, buffer, buffLen - 1);
	    Close(fh);
	    if len > 0 then
		(buffer + len)* := '\e';
		true
	    else
		false
	    fi
	else
	    false
	fi
    fi
corp;

/*
 * localGetMessage - parameterless routine to get a server message.
 */

proc localGetMessage()*Request_t:

    pretend(GetMsg(MyPort), *Request_t)
corp;

/*
 * waitInput - simple wait for Intuition event to exit on.
 */

proc waitInput()void:
    register *Message_t m;
    register ulong intuitionBit, gotBits;
    bool hadKey;

    setPrompt("Type any key to exit: ");
    intuitionBit := 1 << TextWindow*.w_UserPort*.mp_SigBit;
    hadKey := false;
    while not hadKey do
	gotBits := Wait(intuitionBit | SIGBREAKF_CTRL_C);
	if gotBits & intuitionBit ~= 0 then
	    while
		m := GetMsg(TextWindow*.w_UserPort);
		m ~= nil
	    do
		if pretend(m, *IntuiMessage_t)*.im_Class = RAWKEY then
		    hadKey := true;
		fi;
		ReplyMsg(m);
	    od;
	fi;
    od;
corp;

/*
 * doRecv - do a Recv from our socket.
 */

proc doRecv(arbptr buf; ulong len)long:
    register long res;

    res := Recv(IPSocket, buf, len, 0);
    if res < 0 then
	errorString("Error on Recv - aborting\n");
	doExit();
    elif res = 0 then
	errorString("EOF on socket - aborting\n");
	ShuttingDown := true;
	MUDShutDown(&MS);
    fi;
    res
corp;

/*
 * copyRequest - copy data from the static request buffer into the passed one.
 */

proc copyRequest(register *Request_t rq)void:

    rq*.rq_key := RqBuf.rq_key;
    rq*.rq_clientId := RqBuf.rq_clientId;
    rq*.rq_usedLen := RqBuf.rq_usedLen;
    rq*.rq_type := RqBuf.rq_type;
    rq*.rq_flag := RqBuf.rq_flag;
corp;

/*
 * checkSocket - check for activity, including on the socket. Return
 *	the indicated signal activity bits, and any received request.
 */

proc checkSocket(*ulong mask)*Request_t:
    register *Request_t rq;
    register long res;
    FDSet_t readFds, exceptFds;

    FD_ZERO(&readFds);
    FD_SET(&readFds, IPSocket);
    FD_ZERO(&exceptFds);
    FD_SET(&exceptFds, IPSocket);
    res := WaitSelect(FD_SETSIZE, &readFds, nil, &exceptFds, nil, mask);
    if res < 0 then
	errorString("Error on WaitSelect, errno ");
	errorNum(Errno());
	errorChar('\n');
	doExit();
    fi;
    if FD_ISSET(&exceptFds, IPSocket) then
	errorString("Exception on socket\n");
	doExit();
    fi;
    rq := nil;
    if FD_ISSET(&readFds, IPSocket) then
	if MoreHeader then
	    /* Need more of the current header. */
	    res := doRecv(&RqBuf.rq_key + (HeaderLen - Expecting), Expecting);
	    Expecting := Expecting - res;
	    if Expecting = 0 then
		/* Now got the full header. */
		MoreHeader := false;
		rq := allocRequestE(&MS, RqBuf.rq_usedLen);
		copyRequest(rq);
		Expecting := rq*.rq_usedLen;
		if Expecting ~= 0 then
		    /* Some tail stuff coming. */
		    CurrentRq := rq;
		    rq := nil;
		fi;
	    fi;
	elif Expecting ~= 0 then
	    /* Got some extra tail stuff. */
	    rq := CurrentRq;
	    res :=
		doRecv(&rq*.rq_u.ru_bytes[rq*.rq_usedLen-Expecting],Expecting);
	    Expecting := Expecting - res;
	    if Expecting ~= 0 then
		/* Still more to come. */
		rq := nil;
	    fi;
	else
	    /* Not expecting any tail data - read just a header. */
	    res := doRecv(&RqBuf.rq_key, HeaderLen);
	    Expecting := HeaderLen - res;
	    if Expecting = 0 then
		/* Got a full header. Anything with it? */
		rq := allocRequestE(&MS, RqBuf.rq_usedLen);
		copyRequest(rq);
		Expecting := rq*.rq_usedLen;
		if Expecting ~= 0 then
		    /* Got more coming - wait for it. */
		    CurrentRq := rq;
		    rq := nil;
		fi;
	    else
		MoreHeader := true;
	    fi
	fi;
    fi;
    rq
corp;

/*
 * handleIntuition - handle a bunch of message from Intuition IDCMP.
 */

proc handleIntuition(bool wantInput)void:
    register *Message_t m;
    register *IntuiMessage_t im @ m;

    while
	m := GetMsg(TextWindow*.w_UserPort);
	m ~= nil
    do
	if not handleMessage(im, wantInput) then
	    /* removeMenus already called */
	    ShuttingDown := true;
	    MUDShutDown(&MS);
	fi;
    od;
corp;

/*
 * editErrorStart - the start of an error while parsing after editing.
 */

proc editErrorStart(uint line, column)void:

    if not HadErrorSkip then
	EditParseError := true;
	if EditInternal then
	    ed_errorStart(column);
	else
	    putString("*** (");
	    putUnsigned(line);
	    putChar(',');
	    putUnsigned(column);
	    putString(") ");
	fi;
    fi;
corp;

/*
 * editErrorString - the string of an error when parsing after editing.
 */

proc editErrorString(*char m)void:

    if not HadErrorSkip then
	if EditInternal then
	    ed_errorString(m);
	else
	    putString(m);
	fi;
    fi;
corp;

/*
 * editErrorEnd - the end of an edit-time error.
 */

proc editErrorEnd()void:
    register ulong breakBit, intuitionBit, waitBits, gotBits;
    register *Request_t rq;
    register proc()*Request_t getMessage;
    register *Message_t m @ rq;
    ulong mask;

    if EditInternal and not HadErrorSkip then
	HadErrorContinue := false;
	if Serial then
	    getMessage := serialGetMessage;
	else
	    getMessage := localGetMessage;
	fi;
	breakBit := SIGBREAKF_CTRL_C;
	intuitionBit := 1 << TextWindow*.w_UserPort*.mp_SigBit;
	waitBits :=
	    intuitionBit | ServerBit | breakBit | SerialBits | EffectsBits |
		SIGBREAKF_CTRL_F;
	while not HadErrorContinue and not ShuttingDown do
	    if UseIP then
		mask := waitBits;
		rq := checkSocket(&mask);
		gotBits := mask;
		if rq ~= nil then
		    serverMessage(&MS, rq);
		fi;
	    else
		while
		    if MyTask*.tc_SigRecvd & breakBit ~= 0 then
			handleControl_C();
		    fi;
		    rq := getMessage();
		    rq ~= nil
		do
		    serverMessage(&MS, rq);
		od;
		gotBits := Wait(waitBits);
	    fi;
	    if gotBits & breakBit ~= 0 then
		handleControl_C();
	    fi;
	    if gotBits & SIGBREAKF_CTRL_F ~= 0 then
		screenToFront();
	    fi;
	    if gotBits & intuitionBit ~= 0 then
		handleIntuition(false);
	    fi;
	    if gotBits & SerialBits ~= 0x0 then
		handleSerialEvents(gotBits);
	    fi;
	    if gotBits & EffectsBits ~= 0 then
		handleEffectSignal(gotBits);
	    fi;
	    checkRefresh();
	od;
    fi;
corp;

/*
 * editNextError - go to next error, or skip all the rest.
 */

proc editNextError(bool skipRest)void:

    HadErrorContinue := true;
    HadErrorSkip := skipRest;
corp;

/*
 * editCleanup - call editProcCancel and clear some flags.
 */

proc editCleanup()void:

    if EditIsProc then
	editProcCancel(&MS);
    else
	editStringDone(&MS, "", 0, BOOLEAN_FALSE);
    fi;
    EditFailing := false;
    EditFirst := true;
corp;

/*
 * editEnd - end of internal edit.
 */

proc editEnd(bool aborting)void:
    *char buff;
    uint len, allocLen;

    if aborting then
	HadErrorContinue := true;
	if EditIsProc then
	    editCleanup();
	else
	    editStringDone(&MS, "", 0, BOOLEAN_FALSE);
	fi;
    else
	if EditIsProc then
	    EditReading := true;
	    EditInternal := true;
	    EditParseError := false;
	    HadErrorSkip := false;
	    editProcStart(&MS);
	    EditInternal := false;
	    EditReading := false;
	else
	    ed_getString(&buff, &len, &allocLen);
	    if buff = nil then
		editStringDone(&MS, "", 0, BOOLEAN_TRUE);
	    else
		editStringDone(&MS, buff, len, BOOLEAN_TRUE);
		FreeMem(buff, allocLen * sizeof(char));
	    fi;
	fi;
    fi;
corp;

/*
 * editWrite - try to write something to the edit file.
 */

proc editWrite(*char buf; uint len)bool:

    if not EditError then
	if Write(EditFH, buf, len) ~= len then
	    EditError := true;
	fi;
    fi;
    true
corp;

/*
 * editCook - cook a string being edited through the passed handler.
 */

proc editCook(register *char buf; register uint bLen, maxColumn;
	      register proc(*char buf; uint len)bool cooker)bool:
    register *char backPos;
    register uint column, len;
    register char ch @ len;
    uint shrunkMax;
    bool needSpace;

    bLen := bLen - 1;		/* ignore the \e at the end */
    column := 0;
    needSpace := false;
    while bLen ~= 0 do
	ch := buf*;
	if ch = '\n' or ch = '\t' or ch = '\\' then
	    /* an actual one in the string - expand it */
	    if needSpace then
		needSpace := false;
		if not cooker(" ", 1) then
		    return(false);
		fi;
		column := column + 1;
	    fi;
	    if column > maxColumn - 2 then
		if not cooker("\n", 1) then
		    return(false);
		fi;
		column := 0;
	    fi;
	    if not cooker(
		if ch = '\n' then "\\n"
		elif ch = '\t' then "\\t"
		else "\\\\" fi,
		2)
	    then
		return(false);
	    fi;
	    buf := buf + sizeof(char);
	    bLen := bLen - 1;
	    column := column + 2;
	elif ch = ' ' then
	    backPos := buf;
	    len := 0;
	    while bLen ~= 0 and buf* = ' ' do
		len := len + 1;
		buf := buf + sizeof(char);
		bLen := bLen - 1;
	    od;
	    if column ~= 0 and
		(column + len > maxColumn or
		 column + len = maxColumn and bLen ~= 0)
	    then
		if not cooker("\n", 1) then
		    return(false);
		fi;
		column := 0;
		if len = 1 then
		    /* chuck a single separating space */
		    len := 0;
		fi;
	    fi;
	    if len ~= 0 and len <= maxColumn then
		if len = 1 then
		    needSpace := true;
		else
		    if not cooker(backPos, len) then
			return(false);
		    fi;
		    column := column + len;
		fi;
	    fi;
	else
	    /* something that is part of a 'word' */
	    shrunkMax := maxColumn;
	    if needSpace then
		shrunkMax := shrunkMax - 1;
	    fi;
	    backPos := buf;
	    len := 0;
	    while
		if column + len > shrunkMax and column ~= 0 then
		    shrunkMax := maxColumn;
		    needSpace := false;
		    if not cooker("\n", 1) then
			return(false);
		    fi;
		    column := 0;
		fi;
		if len > shrunkMax then
		    shrunkMax := maxColumn;
		    needSpace := false;
		    if not cooker(backPos, len - 1) or not cooker("\n", 1) then
			return(false);
		    fi;
		    column := 0;
		    backPos := buf - sizeof(char);
		    len := 1;
		fi;
		bLen ~= 0 and buf* ~= ' ' and buf* ~= '\n' and buf* ~= '\t' and
		    buf* ~= '\\'
	    do
		len := len + 1;
		buf := buf + sizeof(char);
		bLen := bLen - 1;
	    od;
	    if len ~= 0 then
		if needSpace then
		    needSpace := false;
		    if not cooker(" ", 1) then
			return(false);
		    fi;
		    column := column + 1;
		fi;
		if not cooker(backPos, len) then
		    return(false);
		fi;
		column := column + len;
	    fi;
	fi;
    od;
    cooker("\n", 1)
corp;

/*
 * childCommand - stub that is the body of the child editing task.
 */

proc childCommand()void:

    ignore Execute(&CmdBuffer[0], 0, 0);
    Signal(MyTask, ChildSignalBit);
corp;

/*
 * doEditTail - second half of editing.
 */

proc doEditTail()void:
    register *char buf, backPos;
    register uint bLen;
    *FileInfoBlock_t fib;
    *char allocBuf;
    ulong longLen;
    uint allocLen, extraLen, column, len;
    Lock_t lock;

    if EditIsProc then
	EditFH := Open(EDIT_FILE_NAME, MODE_OLDFILE);
	if EditFH = 0 then
	    /* eek! the file has gone away */
	    editFileComplain("Can't reopen ", " after edit");
	    editCleanup();
	    return;
	fi;
	EditReading := true;
	EditMax := 0;
	EditPos := 0;
	editProcStart(&MS);
	editCleanup();
    else
	lock := Lock(EDIT_FILE_NAME, ACCESS_READ);
	if lock = 0 then
	    editFileComplain("Can't lock ", " after edit");
	    editCleanup();
	    return;
	fi;
	fib := mudAlloc(sizeof(FileInfoBlock_t), 0x0);
	if fib = nil then
	    editComplain("Can't allocate for size after edit");
	    DeleteFile(EDIT_FILE_NAME);
	    editCleanup();
	    return;
	fi;
	if not Examine(lock, fib) then
	    editFileComplain("Can't examine ", " after edit");
	    FreeMem(fib, sizeof(FileInfoBlock_t));
	    DeleteFile(EDIT_FILE_NAME);
	    editCleanup();
	    return;
	fi;
	longLen := fib*.fib_Size;
	UnLock(lock);
	FreeMem(fib, sizeof(FileInfoBlock_t));
	if longLen > 3700 then
	    longLen := 3700;
	fi;
	bLen := longLen;
	extraLen := bLen / 10;
	if extraLen < 50 then
	    extraLen := 50;
	fi;
	allocLen := bLen + extraLen;
	allocBuf := mudAlloc(allocLen * sizeof(char), 0x0);
	if allocBuf = nil then
	    editComplain("Can't allocate string buffer after edit");
	    DeleteFile(EDIT_FILE_NAME);
	    editCleanup();
	    return;
	fi;
	EditFH := Open(EDIT_FILE_NAME, MODE_OLDFILE);
	if EditFH = 0 then
	    /* eek! the file has gone away */
	    editFileComplain("Can't reopen ", " after edit");
	    FreeMem(allocBuf, allocLen * sizeof(char));
	    return;
	fi;
	/* Note what we are doing here: we read the raw data into the latter
	   part of the allocated buffer, then we let 'ed_normalize' read from
	   it and write into the beginning of the same buffer. This works
	   because 'ed_normalize' works from left to right. */
	buf := allocBuf + extraLen;
	if Read(EditFH, buf, bLen) ~= bLen then
	    editFileComplain("Can't read ", " after edit");
	    FreeMem(allocBuf, allocLen * sizeof(char));
	    Close(EditFH);
	    DeleteFile(EDIT_FILE_NAME);
	    editCleanup();
	    return;
	fi;
	Close(EditFH);
	DeleteFile(EDIT_FILE_NAME);
	column := 0;
	len := ed_normalize(buf, bLen, true, allocBuf, allocLen,
			    &extraLen, &column);
	/* Note that this is a call to the mud.library from inside a callout
	   from that library! */
	editStringDone(&MS, allocBuf, len, BOOLEAN_TRUE);
	FreeMem(allocBuf, allocLen * sizeof(char));
    fi;
corp;

/*
 * doEdit - allow player to edit a proc or string.
 */

proc doEdit(register *char buf; register ulong bLen; uint mode;
	    bool isProc, isLast, deleteFirst;
	    *char prompt)bool:
    extern child(proc()void addr)bool;
    ulong MAX_COLUMN = 77;
    register *char backPos;
    register long len;
    register ulong column;
    bool wasFirst;

    if EditReading then
	return(false);
    fi;
    if prompt = nil then
	prompt := "Press HELP key for help";
    fi;
    wasFirst := EditFirst;
    if wasFirst then
	EditFirst := false;
	EditFailing := false;
    fi;

    EditIsProc := isProc;
    ed_setIsRaw(mode ~= EDIT_COOKED);
    if mode = EDIT_CODE then
	/* Be careful of the double meaning here. Initially, 'isProc' says
	   we are actually editing a proc directly. After this assignment,
	   it just says that we should do output as if we were editing a
	   proc. */
	isProc := true;
    fi;

    if EditorChoice = ed_external or EditorChoice = ed_selected and isProc then
	if bLen ~= 0 and not EditFailing then
	    ignore IoErr();
	    if deleteFirst and wasFirst then
		DeleteFile(EDIT_FILE_NAME);
		EditFH := 0;
	    else
		EditFH := Open(EDIT_FILE_NAME, MODE_READWRITE);
	    fi;
	    if EditFH = 0 then
		if IoErr() = ERROR_DEVICE_NOT_MOUNTED then
		    editFileComplain("Can't open ", nil);
		    return(false);
		fi;
		EditFH := Open(EDIT_FILE_NAME, MODE_NEWFILE);
		if EditFH = 0 then
		    /* oh-oh - can't write */
		    editFileComplain("Can't create ", nil);
		    return(false);
		fi;
	    fi;

	    EditError := false;
	    if isProc then
		ignore Seek(EditFH, 0, OFFSET_END);
		ignore editWrite(buf, bLen);
	    else
		ignore editCook(buf, bLen, MAX_COLUMN, editWrite);
	    fi;

	    Close(EditFH);
	    if EditError then
		editFileComplain("Error writing ", nil);
		DeleteFile(EDIT_FILE_NAME);
		return(false);
	    fi;
	fi;
	if EditFailing then
	    if isLast then
		EditFirst := true;
		EditFailing := false;
	    fi;
	    return(false);
	fi;
	if not isLast then
	    return(true);
	fi;

	if not getVar("MUDEditor", &CmdBuffer[0], CMD_BUF_LENGTH) and
	    not getVar("EDITOR", &CmdBuffer[0], CMD_BUF_LENGTH)
	then
	    CharsCopy(&CmdBuffer[0], "C:Ed");
	fi;
	len := CharsLen(&CmdBuffer[0]);
	if len >= CMD_BUF_LENGTH - EDIT_FILE_NAME_LEN - EDIT_OVERHEAD then
	    /* command would be too long! */
	    editComplain("Edit command too long");
	    DeleteFile(EDIT_FILE_NAME);
	    return(false);
	fi;
	CharsCopy(&CmdBuffer[len], " >NIL: <NIL: ");
	CharsCopy(&CmdBuffer[len + EDIT_OVERHEAD - 1], EDIT_FILE_NAME);
	if CreateProc("MUD child", 0, (pretend(child, ulong) + 4) >> 2, 9996) =
	    nil
	then
	    editComplain("Can't create edit process");
	    DeleteFile(EDIT_FILE_NAME);
	    return(false);
	else
	    screenToBack();
	fi;
    else
	if isProc then
	    if not deleteFirst then
		editComplain(
		    "Can only do anonymous edit with external editor");
		EditFailing := false;
		EditFirst := true;
		return(false);
	    fi;
	    if not EditFilling then
		EditFilling := true;
		ed_start();
	    fi;
	    if not ed_text(buf, bLen) then
		EditFilling := false;
		editComplain("Out of memory on edit");
		EditFailing := false;
		EditFirst := true;
		return(false);
	    fi;
	    if isLast then
		EditFilling := false;
		HadErrorSkip := false;
		ed_go(prompt);
	    fi;
	else
	    ed_start();
	    if not ed_stringText(buf, bLen) then
		editComplain("Out of memory on edit");
		EditFailing := false;
		EditFirst := true;
		return(false);
	    fi;
	    ed_go(prompt);
	fi;
    fi;
    true
corp;

/*
 * serverSend - library wants to send a message to the server.
 */

proc serverSend(*Request_t rq)void:
    register ulong len;

    if UseIP then
	len := HeaderLen + rq*.rq_usedLen;
	if Send(IPSocket, &rq*.rq_key, len, 0) ~= len then
	    freeRequestE(&MS, rq);
	    errorString("Error on Send, errno ");
	    errorNum(Errno());
	    errorChar('\n');
	    doExit();
	fi;
	freeRequestE(&MS, rq);
    elif Serial then
	serialSendMessage(rq);
    else
	/* This test is based on the
	   fact that with PARANOID enabled (which it is) in the ExecSupport
	   code (DeletePort in particular), these fields are zapped when the
	   server port goes away. A check for ln_Type = 0xff did NOT go off,
	   but this one does - I guess it relates to FreeMem. The whole point
	   being that we have no way of knowing if the server has aborted,
	   and thus deleted and freed its port. */
	Forbid();
	if ServerPort*.mp_Node.ln_Type ~= NT_MSGPORT or
	    ServerPort*.mp_MsgList.lh_Head = pretend(-1, *Node_t)
	then
	    Permit();
	    errorString("Server gone - aborting!\n");
	    doExit();
	fi;
	PutMsg(ServerPort, &rq*.rq_message);
	Permit();
    fi;
corp;

/*
 * serverRequest - routine to allow library to send a request to the MUD
 *	server, and wait for a reply.
 */

proc serverRequest(register *Request_t rq)*Request_t:
    register proc()*Request_t getMessage;
    register *Message_t m;
    register ulong waitBits, breakBit, gotBits;
    register RequestType_t rt;
    ulong intuitionBit, mask;

    /* This routine used to ignore Intuition messages. Unfortunately, that
       is how the user would abort out of a 'stuck' situation, like if the
       server has gone away. Sigh. */

    rt := rq*.rq_type;
    getMessage := if Serial then serialGetMessage else localGetMessage fi;
    serverSend(rq);

    breakBit := SIGBREAKF_CTRL_C;
    intuitionBit := 1 << TextWindow*.w_UserPort*.mp_SigBit;
    waitBits :=
	intuitionBit | ServerBit | breakBit | SerialBits | EffectsBits |
	    SIGBREAKF_CTRL_F;
    while
	if UseIP then
	    mask := waitBits;
	    rq := checkSocket(&mask);
	    if ShuttingDown then
		/* mud.library *must* have a reply! */
		doExit();
	    fi;
	    gotBits := mask;
	    if rq ~= nil and rq*.rq_type ~= rt then
		if rt = rt_beginClient and rq*.rq_type = rt_stopClient then
		    freeRequestE(&MS, rq);
		    waitInput();
		    /* In this special case, library allows a nil result. */
		    return(nil);
		fi;
		serverMessage(&MS, rq);
		rq := nil;
	    fi;
	else
	    while
		if MyTask*.tc_SigRecvd & breakBit ~= 0 then
		    handleControl_C();
		fi;
		rq := getMessage();
		rq ~= nil and rq*.rq_type ~= rt
	    do
		serverMessage(&MS, rq);
	    od;
	    if rq = nil then
		gotBits := Wait(waitBits);
	    fi;
	fi;
	rq = nil
    do
	if gotBits & breakBit ~= 0 then
	    handleControl_C();
	fi;
	if gotBits & SIGBREAKF_CTRL_F ~= 0 then
	    screenToFront();
	fi;
	if gotBits & intuitionBit ~= 0 then
	    handleIntuition(false);
	fi;
	if gotBits & SerialBits ~= 0x0 then
	    handleSerialEvents(gotBits);
	fi;
	if gotBits & EffectsBits ~= 0 then
	    handleEffectSignal(gotBits);
	fi;
	checkRefresh();
    od;
    rq
corp;

/*
 * sendLine - send an input line to the server.
 */

proc sendLine(*char p; uint len)void:

    if FullyRunning then
	inputLine(&MS, p, len);
    fi;
corp;

/*
 * setSync - send request to control replies to input lines.
 */

proc setSync(bool flag)void:

    if FullyRunning then
	syncControl(&MS, if flag then 1 else 0 fi);
    fi;
corp;

/*
 * sendKey - send a raw key event to the server.
 */

proc sendKey(uint key)void:

    if FullyRunning then
	rawKeyEvent(&MS, key);
    fi;
corp;

/*
 * sendRegion - send a mouse hit event to the server.
 */

proc sendRegion(uint n, mouseX, mouseY)void:

    mouseUpEvent(&MS, n, make(mouseX, ulong) << 16 | mouseY);
corp;

/*
 * sendButton - send a button hit event to the server.
 */

proc sendButton(uint whichButton)void:

    buttonEvent(&MS, whichButton);
corp;

/*
 * sendReleaseEffect - send a request to release an effect.
 */

proc sendReleaseEffect(ulong key)void:

    releaseEffect(&MS, key);
corp;

/*
 * sendResize - send information about the text window being resized.
 */

proc sendResize(uint newRows, newCols)void:

    textResize(&MS, newRows, newCols);
corp;

/*
 * handleAll - handle all of the signals/ports that we care about.
 *	This is only called when we know there is something to do, or we are
 *	happy to Wait, because there is nothing else to do.
 */

proc handleAll(bool wantInput)void:
    uint LINE_LENGTH = 256;
    register *Request_t rq;
    register *Message_t m @ rq;
    register proc()*Request_t getMessage;
    register ulong intuitionBit, breakBit, allBits, gotBits, len @ allBits;
    [LINE_LENGTH + 1] char line;
    register uint pos @ gotBits;
    ulong mask;
    uint lineLen;
    char ch;
    bool hadServer, wantRefresh;

    FullyRunning := true;
    breakBit := SIGBREAKF_CTRL_C;
    intuitionBit := 1 << TextWindow*.w_UserPort*.mp_SigBit;
    allBits :=
	intuitionBit | ServerBit | breakBit | SerialBits | ChildSignalBit |
	    EffectsBits | SIGBREAKF_CTRL_F;

    /* it SHOULD be correct to just do the Wait - I have no idea as to
       why that does not work occasionally */
    gotBits := 0x0;
    if TextWindow*.w_UserPort*.mp_MsgList.lh_Head*.ln_Succ ~= nil then
	gotBits := gotBits | intuitionBit;
    fi;
    hadServer := false;
    if UseIP then
	if gotBits = 0x0 and not EditReading then
	    mask := allBits;
	    rq := checkSocket(&mask);
	    gotBits := mask;
	    if gotBits & EffectsBits ~= 0 then
		handleEffectSignal(gotBits);
	    fi;
	    if rq ~= nil then
		serverMessage(&MS, rq);
		hadServer := true;
	    fi;
	fi;
    else
	if not Serial then
	    if MyPort*.mp_MsgList.lh_Head*.ln_Succ ~= nil then
		gotBits := gotBits | ServerBit;
	    fi;
	else
	    gotBits := gotBits | MyTask*.tc_SigRecvd & SerialBits;
	fi;

	/* do not want to wait for other input if are currently reading
	   from the edit file! */
	if gotBits = 0x0 and not EditReading then
	    gotBits := Wait(allBits);
	fi;

	if gotBits & SerialBits ~= 0x0 then
	    handleSerialEvents(gotBits);
	fi;
	if gotBits & EffectsBits ~= 0 then
	    handleEffectSignal(gotBits);
	fi;
	getMessage := if Serial then serialGetMessage else localGetMessage fi;
	while
	    rq := getMessage();
	    rq ~= nil
	do
	    serverMessage(&MS, rq);
	    hadServer := true;
	od;
    fi;
    if not hadServer then
	wantInput := true;
    fi;

    if gotBits & ChildSignalBit ~= 0x0 then
	screenToFront();
	doEditTail();
    fi;
    if gotBits & SIGBREAKF_CTRL_F ~= 0 then
	screenToFront();
    fi;

    handleIntuition(wantInput);

    if gotBits & breakBit ~= 0x0 then
	/* a BREAK signal from another CLI - go away */
	handleControl_C();
    fi;
    checkRefresh();

    if EditReading then
	if EditInternal then
	    if ed_getLine(&line[0], INPUT_BUFFER_SIZE, &lineLen) then
		inputLine(&MS, &line[0], lineLen);
	    else
		editProcDone(&MS);
	    fi;
	else
	    pos := 0;
	    while
		if EditPos = EditMax then
		    len := Read(EditFH, &EditBuffer[0], INPUT_BUFFER_SIZE);
		    if len <= 0 then
			if pos ~= 0 then
			    if line[pos - 1] = '\n' then
				pos := pos - 1;
			    fi;
			    inputLine(&MS, &line[0], pos);
			fi;
			EditReading := false;
			Close(EditFH);
			editProcDone(&MS);
			EditFirst := true;
			EditFailing := false;
			false
		    else
			EditMax := len;
			EditPos := 0;
			true
		    fi
		else
		    ch := EditBuffer[EditPos];
		    EditPos := EditPos + 1;
		    if ch = '\n' then
			inputLine(&MS, &line[0], pos);
			false
		    else
			if pos < LINE_LENGTH then
			    line[pos] := ch;
			    pos := pos + 1;
			fi;
			true
		    fi
		fi
	    do
		checkRefresh();
	    od;
	fi;
    fi;
corp;

/*
 * checkMessages - MUD library is busy, but taking a while. This is a
 *	kind of poll for non-server stuff that needs doing.
 */

proc checkMessages()void:
    bool wantCheck;
    long res;
    ulong mask;
    FDSet_t readFds;

    wantCheck := false;
    if TextWindow*.w_UserPort*.mp_MsgList.lh_Head*.ln_Succ ~= nil then
	wantCheck := true;
    fi;
    if UseIP then
	FD_ZERO(&readFds);
	FD_SET(&readFds, IPSocket);
	mask := 0x0;
	res := WaitSelect(FD_SETSIZE, &readFds, nil, nil, nil, &mask);
	if res < 0 then
	    errorString("Error on WaitSelect, errno ");
	    errorNum(Errno());
	    errorChar('\n');
	    doExit();
	fi;
	if res > 0 then
	    wantCheck := true;
	fi;
    elif Serial then
	if MyTask*.tc_SigRecvd & SerialBits ~= 0x0 then
	    wantCheck := true;
	fi;
    else
	if MyPort*.mp_MsgList.lh_Head*.ln_Succ ~= nil then
	    wantCheck := true;
	fi;
    fi;
    if wantCheck then
	handleAll(false);
    fi;
corp;

/*
 * interfaces for flipping effects on and off
 */

proc gFlip(bool onOff)void:

    if FullyRunning then
	graphicsFlip(&MS, if onOff then 1 else 0 fi);
    fi;
corp;

proc vFlip(bool onOff)void:

    if FullyRunning then
	voiceFlip(&MS, if onOff then 1 else 0 fi);
    fi;
corp;

proc sFlip(bool onOff)void:

    if FullyRunning then
	soundFlip(&MS, if onOff then 1 else 0 fi);
    fi;
corp;

proc mFlip(bool onOff)void:

    if FullyRunning then
	musicFlip(&MS, if onOff then 1 else 0 fi);
    fi;
corp;

proc eDone(ulong kind, id)void:

    if id ~= 0 then
	effectDone(&MS, kind, id);
    fi;
corp;

/*
 * shutDownAndAbort - an abort call from the library - shutdown cleanly.
 */

proc shutDownAndAbort(bool isServerNuke)void:
    register *Request_t rq;

    errorString("Abort request from mud.library - aborting!\n");
    /* try to free up memory BEFORE sending the die message, so that
       the server has a hope of replying to us, since we cannot
       otherwise free up memory until we get the reply from the
       server. */
    if TextWindow ~= nil then
	intuitionTerm();
    fi;
    if not UseIP and not Serial and not isServerNuke then
	rq := InitialRequest;
	MUDDisconnectMessage(&MS, rq);
	PutMsg(ServerPort, &rq*.rq_message);
	ignore WaitPort(MyPort);
	ignore GetMsg(MyPort);
    fi;
    abort();
corp;

/*
 * playGame - go in and play the game.
 */

proc playGame()void:
    register *Request_t rq;

    ShuttingDown := false;

    MS.ms_AllocMem := mudAlloc;
    MS.ms_FreeMem := FreeMem;
    MS.ms_ReclaimMem := nil;
    MS.ms_RetryMem := nil;
    MS.ms_Open := pretend(Open, proc(*char name; ulong mode)long);
    MS.ms_Close := pretend(Close, proc(ulong fd)void);
    MS.ms_Read := pretend(Read, proc(ulong fd; arbptr buffer; ulong len)long);

    MS.ms_abort := shutDownAndAbort;
    MS.ms_errorChar := errorChar;
    MS.ms_outputChar := putChar;

    MS.ms_incRef := nil;
    MS.ms_decRef := nil;
    MS.ms_lookup := nil;
    MS.ms_findAnyName := nil;
    MS.ms_findProc := nil;
    MS.ms_writeProcNew := nil;
    MS.ms_findBuiltin := nil;
    MS.ms_getIntConstValue := nil;

    MS.ms_putText := putText;
    MS.ms_setPrompt := setPrompt;
    MS.ms_setEcho := setEcho;
    MS.ms_serverRequest := serverRequest;
    MS.ms_serverSend := serverSend;
    MS.ms_handleAll := handleAll;
    MS.ms_checkMessages := checkMessages;
    MS.ms_getEffects := getEffects;
    MS.ms_doEffects := doEffects;
    MS.ms_doEdit := doEdit;
    MS.ms_editErrorStart := editErrorStart;
    MS.ms_editErrorString := editErrorString;
    MS.ms_editErrorEnd := editErrorEnd;
    MS.ms_getString := getString;
    MS.ms_defineEffect := defineEffect;
    MS.ms_flushEffect := flushEffect;

    if MUDInitialize(&MS, MyPort, MUDFlags, 10) then
	InitedLib := true;
	if UseIP or Serial then
	    MUDMain(&MS, ForceCharacterName, ForcePassword);
	else
	    rq := AllocMem(sizeof(Request_t) - TEXT_LENGTH,
			   MEMF_PUBLIC | MEMF_CLEAR);
	    if rq ~= nil then
		putString("Attempting to start a MUD session with a local "
			  "MUDServ on this machine...\n\n");
		InitialRequest := rq;
		MUDConnectMessage(&MS, rq);
		Forbid();
		ServerPort := FindPort(&PortName[0]);
		if ServerPort ~= nil then
		    PutMsg(ServerPort, &rq*.rq_message);
		    Permit();
		    ignore WaitPort(MyPort);
		    if GetMsg(MyPort) ~= &rq*.rq_message then
			errorString("Bad reply on server connect.\n");
		    else
			if MUDConnected(&MS, rq) then
			    putString("...connected.\n\n");
			    MUDMain(&MS, ForceCharacterName, ForcePassword);
			    MUDDisconnectMessage(&MS, rq);
			    PutMsg(ServerPort, &rq*.rq_message);
			    ignore WaitPort(MyPort);
			    if GetMsg(MyPort) ~= &rq*.rq_message then
				errorString(
				    "Bad reply on server disconnect.\n");
			    fi;
			fi;
		    fi;
		else
		    Permit();
		    putString("...MUD could not find the AmigaOS public "
			      "message port ('");
		    putString(&PortName[0]);
		    putString("') of a MUDServ running on this machine. If "
			      "you wish to use a local MUDServ, you must "
			      "start it up first.\n"
			      "   If you wish to use a MUDServ running on "
			      "another machine, via a serial port connection, "
			      "do not specify the -L CLI flag or the LOCAL "
			      "Workbench tooltype.\n"
			      "   If you wish to connect via TCP/IP, then "
			      "specify the hostname [and port] on the CLI "
			      "command line, or use the IPHOST and IPPORT "
			      "tooltypes.\n\n");
		    waitInput();
		fi;
		FreeMem(rq, sizeof(Request_t) - TEXT_LENGTH);
	    else
		errorString("Can't allocate initial server request.\n");
	    fi;
	fi;
	MUDTerminate(&MS);
    fi;
corp;

/*
 * runMUD - start up the MUD-specific stuff.
 */

proc runMUD()void:
    register *HostEnt_t hp;
    SockAddrIn_t addr;

    if UseIP and IPHost[0] = '\e' then
	errorString("Hostname needed for TCP/IP operation.\n");
	return;
    fi;
    tx_setHistorySizes(OutputHistorySize, InputHistorySize);
    if intuitionInit(not UseIP and Serial, CloseWorkbench, Bright, Dim) then
	SaveWindowPtr := MyProcess*.pr_WindowPtr;
	MyProcess*.pr_WindowPtr := TextWindow;
	EffectsInited := true;
	ChildSignal := AllocSignal(-1);
	if ChildSignal ~= -1 then
	    ChildSignalBit := 1 << ChildSignal;
	    if UseIP then
		MUDFlags := MUDFlags | FLAG_IS_REMOTE;
		putString("Attempting to start a MUD session using TCP/IP"
			  " with a server on host '");
		putString(&IPHost[0]);
		putString("' with port ");
		putUnsigned(IPPort);
		putString("...\n\n");
		if OpenBsdsocketLibrary(0) ~= nil then
		    hp := GetHostByName(&IPHost[0]);
		    if hp ~= nil then
			IPSocket := Socket(AF_INET, SOCK_STREAM, 0);
			if IPSocket >= 0 then
			    BlockFill(&addr, sizeof(SockAddrIn_t), 0);
			    addr.sin_len := sizeof(SockAddrIn_t);
			    addr.sin_family := AF_INET;
			    addr.sin_port := IPPort;
			    addr.sin_addr.s_addr :=
				pretend(hp*.h_addrlist*, *ulong)*;
			    if Connect(IPSocket, &addr,sizeof(SockAddrIn_t))>=0
			    then
				putString("...connected.\n\n");
				playGame();
			    else
				putString("...can't connect to server at '");
				putString(&IPHost[0]);
				putString("', port ");
				putUnsigned(IPPort);
				putString(", errno ");
				putUnsigned(Errno());
				putString("\n\n");
				waitInput();
			    fi;
			    ignore Shutdown(IPSocket, 2);
			    ignore CloseSocket(IPSocket);
			else
			    errorString("...can't create socket, errno ");
			    errorNum(Errno());
			    errorChar('\n');
			fi;
		    else
			putString("...can't find host '");
			putString(&IPHost[0]);
			putString("', errno ");
			putUnsigned(Errno());
			putString("\n\n");
			waitInput();
		    fi;
		    CloseBsdsocketLibrary();
		else
		    putString("...can't open bsdsocket.library. "
			"Is your TCP/IP stack running?\n\n");
		    waitInput();
		fi;
	    elif Serial then
		putString("Attempting to start a MUD session with a remote "
			  "MUDServ via the serial port. If you wish to use "
			  "a MUDServ running on the local machine, specify "
			  "the -L CLI flag or the LOCAL Workbench tooltype. "
			  "If you wish to connect via TCP/IP specify the "
			  "hostname [and port] on the CLI command line, or "
			  "use the IPHOST and IPPORT tooltypes. "
			  "You are now in a simple terminal emulator.\n\n");
		MUDFlags := MUDFlags | FLAG_IS_REMOTE;
		SerialBits :=
		    openSerialHandler(&SerialDevice[0], SerialUnit, RetryLimit,
				      SerialReady, SerialShared, Serial7Wire);
		if SerialBits ~= 0x00000000 then
		    SerialOpen := true;
		    while serialConversation(TextWindow*.w_UserPort,
					     &DialString[0])
		    do
			playGame();
			DialString[0] := '\e';
			effectsTerm(true);
		    od;
		    closeSerialHandler(&SerialDevice[0], SerialUnit);
		    SerialOpen := false;
		else
		    errorString("Can't open ");
		    errorString(&SerialDevice[0]);
		    errorString(" unit ");
		    errorNum(SerialUnit);
		    errorChar('\n');
		fi;
	    else
		MyPort := CreatePort(nil, 0);
		if MyPort ~= nil then
		    ServerBit := 1 << MyPort*.mp_SigBit;
		    playGame();
		    DeletePort(MyPort);
		else
		    errorString("Can't create client reply port!\n");
		    waitInput();
		fi;
	    fi;
	    FreeSignal(ChildSignal);
	fi;
	MyProcess*.pr_WindowPtr := SaveWindowPtr;
	intuitionTerm();
    fi;
corp;

/*
 * addEffectsBit - the effects code is adding a signal bit for an effect event.
 */

proc addEffectsBit(ulong bit)void:

    EffectsBits := EffectsBits | bit;
corp;

/*
 * getNumber - parse a number from the passed string. No error checking.
 */

proc getNumber(*char what; register *char st)ulong:
    register ulong n;

    n := 0;
    while st* >= '0' and st* <= '9' do
	n := n * 10 + (st* - '0');
	st := st + sizeof(char);
    od;
    if st* ~= '\e' then
	errorString("Bad number for ");
	errorString(what);
	errorChar('\n');
	HadParmError := true;
    fi;
    n
corp;

/*
 * getBool - interpret a YES/NO or ON/OFF from a string.
 */

proc getBool(*char what; register *char st)bool:
    register proc(*char a, b)bool r_CharsEqual;

    r_CharsEqual := CharsEqual;
    if st* = '\e' or r_CharsEqual(st, "YES") or r_CharsEqual(st, "yes") or
	r_CharsEqual(st, "ON") or r_CharsEqual(st, "TRUE") or
	r_CharsEqual(st, "on") or r_CharsEqual(st, "true") or
	r_CharsEqual(st, "Y") or r_CharsEqual(st, "y") or
	r_CharsEqual(st, "T") or r_CharsEqual(st, "t")
    then
	true
    elif r_CharsEqual(st, "NO") or r_CharsEqual(st, "no") or
	r_CharsEqual(st, "OFF") or r_CharsEqual(st, "false") or
	r_CharsEqual(st, "off") or r_CharsEqual(st, "false") or
	r_CharsEqual(st, "N") or r_CharsEqual(st, "n") or
	r_CharsEqual(st, "F") or r_CharsEqual(st, "f")
    then
	false
    else
	errorString("Bad flag for ");
	errorString(what);
	errorChar('\n');
	HadParmError := true;
	false
    fi
corp;

/*
 * getMode - check for a valid display style.
 */

proc getMode(register *char m)void:
    register proc(*char a, b)bool r_CharsEqual;

    r_CharsEqual := CharsEqual;
    if r_CharsEqual(m, "ONEWINDOW") or r_CharsEqual(m, "onewindow") or
	r_CharsEqual(m, "ONE_WINDOW") or r_CharsEqual(m, "one_window")
    then
	Mode := md_oneWindow;
    elif r_CharsEqual(m, "TWOWINDOW") or r_CharsEqual(m, "twowindow") or
	r_CharsEqual(m, "TWO_WINDOW") or r_CharsEqual(m, "two_window")
    then
	Mode := md_twoWindow;
    elif r_CharsEqual(m, "ONESCREEN") or r_CharsEqual(m, "onescreen") or
	r_CharsEqual(m, "ONE_SCREEN") or r_CharsEqual(m, "one_screen")
    then
	Mode := md_oneScreen;
    elif r_CharsEqual(m, "TWOSCREEN") or r_CharsEqual(m, "twoscreen") or
	r_CharsEqual(m, "TWO_SCREEN") or r_CharsEqual(m, "two_screen")
    then
	Mode := md_twoScreen;
    else
	errorString("Bad value ('");
	errorString(m);
	errorString("') for MODE.\n");
	HadParmError := true;
    fi;
corp;

/*
 * scanToolTypes - scan a tooltype array for the ones we handle.
 */

proc scanToolTypes(register **char toolTypeArray)bool:
    register *char value;
    register proc(**char toolTypeArray; *char name)*char r_FindToolType;
    register bool known;

    r_FindToolType := FindToolType;
    known := false;
    value := r_FindToolType(toolTypeArray, "SERIAL");
    if value ~= nil then
	known := true;
	Serial := getBool("SERIAL", value);
    fi;
    value := r_FindToolType(toolTypeArray, "CLOSEWORKBENCH");
    if value ~= nil then
	known := true;
	CloseWorkbench := getBool("CLOSEWORKBENCH", value);
    fi;
    value := r_FindToolType(toolTypeArray, "LOCAL");
    if value ~= nil then
	known := true;
	Serial := not getBool("LOCAL", value);
    fi;
    value := r_FindToolType(toolTypeArray, "TEST");
    if value ~= nil then
	known := true;
	Serial := false;
	CharsCopy(&PortName[0], TEST_PORT_NAME);
    fi;
    value := r_FindToolType(toolTypeArray, "BRIGHT");
    if value ~= nil then
	known := true;
	Bright := getBool("BRIGHT", value);
    fi;
    value := r_FindToolType(toolTypeArray, "DIM");
    if value ~= nil then
	known := true;
	Dim := getBool("DIM", value);
    fi;
    value := r_FindToolType(toolTypeArray, "FLAGS");
    if value ~= nil then
	known := true;
	MUDFlags := getNumber("FLAGS", value);
    fi;
    value := r_FindToolType(toolTypeArray, "PORTNAME");
    if value ~= nil then
	known := true;
	Serial := false;
	CharsCopyN(&PortName[0], value, NAME_BUFFER_LEN);
    fi;
    value := r_FindToolType(toolTypeArray, "DEVICE");
    if value ~= nil then
	known := true;
	Serial := true;
	CharsCopyN(&SerialDevice[0], value, NAME_BUFFER_LEN);
    fi;
    value := r_FindToolType(toolTypeArray, "UNIT");
    if value ~= nil then
	known := true;
	Serial := true;
	SerialUnit := getNumber("UNIT", value);
    fi;
    value := r_FindToolType(toolTypeArray, "BAUD");
    if value ~= nil then
	known := true;
	Serial := true;
	SerialBaud := getNumber("BAUD", value);
    fi;
    value := r_FindToolType(toolTypeArray, "IGNORECD");
    if value ~= nil then
	known := true;
	Serial := true;
	SerialIgnoreCD := getBool("IGNORECD", value);
    fi;
    value := r_FindToolType(toolTypeArray, "7WIRE");
    if value ~= nil then
	known := true;
	Serial := true;
	Serial7Wire := getBool("7WIRE", value);
    fi;
    value := r_FindToolType(toolTypeArray, "SHARED");
    if value ~= nil then
	known := true;
	Serial := true;
	SerialShared := getBool("SHARED", value);
    fi;
    value := r_FindToolType(toolTypeArray, "MODE");
    if value ~= nil then
	known := true;
	getMode(value);
    fi;
    value := r_FindToolType(toolTypeArray, "HIRES");
    if value ~= nil then
	known := true;
	HiRes := getBool("HIRES", value);
    fi;
    value := r_FindToolType(toolTypeArray, "OLDSTYLE");
    if value ~= nil then
	known := true;
	OldStyle := getBool("OLDSTYLE", value);
    fi;
    value := r_FindToolType(toolTypeArray, "MAPCOLOURS");
    if value ~= nil then
	known := true;
	MapColours := getBool("MAPCOLOURS", value);
    fi;
    value := r_FindToolType(toolTypeArray, "EFFECTSQUIET");
    if value ~= nil then
	known := true;
	EffectsQuiet := getBool("EFFECTSQUIET", value);
    fi;
    value := r_FindToolType(toolTypeArray, "FULLREDRAW");
    if value ~= nil then
	known := true;
	FullRedraw := getBool("FULLREDRAW", value);
    fi;
    value := r_FindToolType(toolTypeArray, "RETRIES");
    if value ~= nil then
	known := true;
	RetryLimit := getNumber("RETRIES", value);
    fi;
    value := r_FindToolType(toolTypeArray, "DIAL");
    if value ~= nil then
	known := true;
	Serial := true;
	CharsCopyN(&DialString[0], value, DIAL_STRING_LEN);
    fi;
    value:= r_FindToolType(toolTypeArray, "NAME");
    if value ~= nil then
	known := true;
	CharsCopyN(&CharacterName[0], value, PLAYER_NAME_LEN);
	ForceCharacterName := &CharacterName[0];
    fi;
    value := r_FindToolType(toolTypeArray, "PASSWORD");
    if value ~= nil then
	known := true;
	CharsCopyN(&Password[0], value, PASSWORD_LEN);
	ForcePassword := &Password[0];
    fi;
    value := r_FindToolType(toolTypeArray, "EDITOR");
    if value ~= nil then
	known := true;
	if CharsEqual(value, "INTERNAL") or CharsEqual(value, "internal") then
	    EditorChoice := ed_internal;
	elif CharsEqual(value, "SELECTED") or CharsEqual(value,"selected") then
	    EditorChoice := ed_selected;
	elif CharsEqual(value, "EXTERNAL") or CharsEqual(value,"external") then
	    EditorChoice := ed_external;
	else
	    EditorChoice := ed_internal;
	fi;
    fi;
    value := r_FindToolType(toolTypeArray, "OHISTORY");
    if value ~= nil then
	known := true;
	OutputHistorySize := getNumber("OHISTORY", value);
	if OutputHistorySize < 24 * 80 then
	    OutputHistorySize := 24 * 80;
	fi;
    fi;
    value := r_FindToolType(toolTypeArray, "IHISTORY");
    if value ~= nil then
	known := true;
	InputHistorySize := getNumber("IHISTORY", value);
	if InputHistorySize < 2 * 80 then
	    InputHistorySize := 2 * 80;
	fi;
    fi;
    value := r_FindToolType(toolTypeArray, "EDITORSTACK");
    if value ~= nil then
	known := true;
	EditorStackSize := getNumber("EDITORSTACK", value);
	if EditorStackSize < 4096 then
	    EditorStackSize := 4096;
	fi;
    fi;
    value := r_FindToolType(toolTypeArray, "IPHOST");
    if value ~= nil then
	known := true;
	UseIP := true;
	CharsCopyN(&IPHost[0], value, NAME_BUFFER_LEN);
    fi;
    value := r_FindToolType(toolTypeArray, "IPPORT");
    if value ~= nil then
	known := true;
	UseIP := true;
	IPPort := getNumber("IPPORT", value);
    fi;
    known
corp;

/*
 * stackComplain - complain about a too small stack.
 */

proc stackComplain(ulong stackSize)void:

    errorString("Stack size (");
    errorNum(stackSize);
    errorString(") too small - use at least ");
    errorNum(STACK_NEEDED);
    errorString(".\n");
corp;

/*
 * doRun - common code to open library and start up.
 */

proc doRun()void:

    if OldStyle or not OS2 then
	Mode := md_twoScreen;
	OldStyle := true;
    fi;
    if Mode = md_oneScreen or Mode = md_twoScreen then
	MapColours := false;
    else
	CloseWorkbench := false;
    fi;
    if OpenMudLibrary(MUD_VERSION) ~= nil then
	runMUD();
	CloseMudLibrary();
    else
	errorString("Cannot open mud.library. "
		    "Is it in your LIBS: directory?\n");
    fi;
corp;

/*
 * WBProcess - process a WorkBench object.
 */

proc WBProcess(register *WBArg_t wa; *ulong pStackSize)void:
    register *char name @ wa;
    register *DiskObject_t dob @ wa;
    register Lock_t oldDir;

    oldDir := wa*.wa_Lock;
    if oldDir ~= 0 then
	oldDir := CurrentDir(oldDir);
	name := wa*.wa_Name;
	if name ~= nil and name* ~= '\e' and name* ~= ' ' then
	    dob := GetDiskObject(name);
	    if dob ~= nil then
		if dob*.do_StackSize > pStackSize* then
		    pStackSize* := dob*.do_StackSize;
		fi;
		ignore scanToolTypes(dob*.do_ToolTypes);
		FreeDiskObject(dob);
	    fi;
	fi;
	ignore CurrentDir(oldDir);
    fi;
corp;

/*
 * WBRun - WorkBench half of main.
 */

proc WBRun()void:
    register *WBArg_t wa;
    ulong stackSize;

    stackSize := 4000;
    wa := StartupMessage*.sm_ArgList;
    if wa ~= nil then
	if StartupMessage*.sm_NumArgs <= 1 then
	    WBProcess(wa, &stackSize);
	else
	    wa := wa + sizeof(WBArg_t);
	    WBProcess(wa, &stackSize);
	fi;
    fi;
    if stackSize < STACK_NEEDED then
	stackComplain(stackSize);
    elif not HadParmError then
	doRun();
    fi;
corp;

/*
 * showUse - show help info.
 */

proc showUse()void:

    errorString(
"-? - show this info\n"
"-w - open in Workbench window\n"
"-1 - onescreen/onewindow display\n"
"-h - hires display\n"
"-O - oldstyle display\n"
"-m - do not map colours (Workbench only)\n"
"-f - fullredraw\n"
"-q - effects quiet\n"
"-L - force connection to local server\n"
"-T - force local connection on test port\n"
"-D - dim text (old-style only)\n"
"-B - bright text (old-style only)\n"
"-R - force serial port connection\n"
"-S - force shared serial port connection\n"
"-c - close Workbench\n"
"-i - force serial port, with CD ignored\n"
"-7 - force serial port, with 7-wire hardware handshake\n"
"-E - disable editing ability\n"
"-F# - specify MUD flags\n"
"-P<port> - force local connection using named port\n"
"-d<device> - force serial connection on named device\n"
"-u<unit> - force serial connection on given unit\n"
"-b# - force serial connection at given bps speed\n"
"-r# - set given retry limit (only useful when serial)\n"
"-Nname - use 'name' as character name\n"
"-Wword - use 'word' as password on first attempt\n"
"-s# - set size of stack for external editor\n"
"[host [port]] - use TCP/IP stack for connection to server\n"
"    default port is "
    );
    errorNum(DEFAULT_IP_PORT);
    errorChar('\n');
corp;

/*
 * CLIRun - CLI half of main.
 */

proc CLIRun(ulong stackSize)void:
    extern _d_pars_initialize()void;
    register *char par;
    register proc(*char a, b)bool r_CharsEqual;
    register char ch;
    [2] *char ttArray;
    register bool hadError;
    bool hadPort;

    _d_pars_initialize();
    hadError := false;
    ttArray[1] := nil;
    r_CharsEqual := CharsEqual;
    hadPort := false;
    while
	par := GetPar();
	par ~= nil
    do
	if par* = '-' then
	    par := par + sizeof(char);
	    if r_CharsEqual(par, "Getty") then
		SerialReady := true;
		SerialShared := true;
	    elif r_CharsEqual(par, "DEVICE") then
		par := GetPar();
		if par ~= nil then
		    CharsCopy(&SerialDevice[0], par);
		fi;
	    elif r_CharsEqual(par, "UNIT") then
		par := GetPar();
		if par ~= nil then
		    SerialUnit := getNumber("UNIT", par);
		fi;
	    elif r_CharsEqual(par, "HLINK") then
		SerialIgnoreCD := true;
	    elif r_CharsEqual(par, "BAUD") then
		par := GetPar();
		if par ~= nil then
		    SerialBaud := getNumber("BAUD", par);
		fi;
	    elif r_CharsEqual(par, "USER") then
		par := GetPar();
	    elif r_CharsEqual(par, "HELP") or r_CharsEqual(par, "help") then
		showUse();
		hadError := true;
	    else
		while
		    ch := par*;
		    par := par + sizeof(char);
		    ch ~= '\e'
		do
		    case ch
		    incase '?':
			showUse();
			hadError := true;
		    incase 'h':
			HiRes := true;
		    incase '1':
			if Mode = md_twoScreen then
			    Mode := md_oneScreen;
			elif Mode = md_twoWindow then
			    Mode := md_oneWindow;
			fi;
		    incase 'O':
			OldStyle := true;
		    incase 'w':
			if Mode = md_oneScreen then
			    Mode := md_oneWindow;
			elif Mode = md_twoScreen then
			    Mode := md_twoWindow;
			fi;
		    incase 'm':
			MapColours := false;
		    incase 'f':
			FullRedraw := true;
		    incase 'q':
			EffectsQuiet := true;
		    incase 'L':
			Serial := false;
		    incase 'R':
			Serial := true;
		    incase 'T':
			Serial := false;
			CharsCopy(&PortName[0], TEST_PORT_NAME);
		    incase 'B':
			Bright := true;
		    incase 'D':
			Dim := true;
		    incase 'F':
			MUDFlags := getNumber("-F", par);
			par := "";
		    incase 'P':
			Serial := false;
			CharsCopy(&PortName[0], par);
			par := "";
		    incase 'S':
			Serial := true;
			SerialShared := true;
		    incase 'c':
			CloseWorkbench := true;
		    incase 'd':
			Serial := true;
			CharsCopy(&SerialDevice[0], par);
			par := "";
		    incase 'u':
			Serial := true;
			SerialUnit := getNumber("-u", par);
			par := "";
		    incase 'b':
			Serial := true;
			SerialBaud := getNumber("-b", par);
			par := "";
		    incase 'i':
			Serial := true;
			SerialIgnoreCD := true;
		    incase '7':
			Serial := true;
			Serial7Wire := true;
		    incase 'r':
			RetryLimit := getNumber("-r", par);
			par := "";
		    incase 'E':
			DisableEdit := true;
		    incase 'N':
			ForceCharacterName := par;
			par := "";
		    incase 'W':
			ForcePassword := par;
			par := "";
		    incase 's':
			EditorStackSize := getNumber("-s", par);
			if EditorStackSize < 4096 then
			    EditorStackSize := 4096;
			fi;
			par := "";
		    default:
			errorString("Unknown option: ");
			errorChar(ch);
			errorChar('\n');
			hadError := true;
		    esac;
		od;
	    fi;
	else
	    ttArray[0] := par;
	    if not scanToolTypes(&ttArray[0]) then
		if IPHost[0] = '\e' then
		    CharsCopyN(&IPHost[0], par, NAME_BUFFER_LEN);
		    UseIP := true;
		elif hadPort then
		    hadError := true;
		else
		    IPPort := getNumber("IP port", par);
		    hadPort := true;
		fi;
	    fi;
	fi;
    od;
    if hadError then
	errorString("Use is: MUD [-?h1OwmfqLTDBSci7E -F\# -P<port> -d<device> "
		    "-u<unit> -b\# -r\# -Nname -Wword -s\#] [host [port]]\n");
    fi;
    if stackSize < STACK_NEEDED then
	stackComplain(stackSize);
	hadError := true;
    fi;
    if not hadError and not HadParmError then
	doRun();
    fi;
corp;

/*
 * checkEnvironment - check the environment variables we care about.
 */

proc checkEnvironment()void:

    if getVar("MUDLocal", &CmdBuffer[0], CMD_BUF_LENGTH) then
	Serial := not getBool("MUDLocal", &CmdBuffer[0]);
    fi;
    if getVar("MUDMode", &CmdBuffer[0], CMD_BUF_LENGTH) then
	getMode(&CmdBuffer[0]);
    fi;
    if getVar("MUDHiRes", &CmdBuffer[0], CMD_BUF_LENGTH) then
	HiRes := getBool("MUDHiRes", &CmdBuffer[0]);
    fi;
    if getVar("MUDMapColours", &CmdBuffer[0], CMD_BUF_LENGTH) then
	MapColours := getBool("MUDMapColours", &CmdBuffer[0]);
    fi;
    if getVar("MUDFullRedraw", &CmdBuffer[0], CMD_BUF_LENGTH) then
	FullRedraw := getBool("MUDFullRedraw", &CmdBuffer[0]);
    fi;
    if getVar("MUDEffectsQuiet", &CmdBuffer[0], CMD_BUF_LENGTH) then
	EffectsQuiet := getBool("MUDEffectsQuiet", &CmdBuffer[0]);
    fi;
corp;

/*
 * main - start everything up.
 */

proc main()void:
    *DosLibrary_t dosLib;
    *IntuitionBase_t intuiBase;

    if OpenExecLibrary(0) ~= nil then
	dosLib := OpenDosLibrary(0);
	if dosLib ~= nil then
	    GfxBase := OpenGraphicsLibrary(0);
	    if GfxBase ~= nil then
		intuiBase := OpenIntuitionLibrary(0);
		if intuiBase ~= nil then
		    OS2 := dosLib*.dl_lib.lib_Version >= 36 and
			GfxBase*.gb_LibNode.lib_Version >= 36 and
			intuiBase*.ib_LibNode.lib_Version >= 36;
		    OS3 := dosLib*.dl_lib.lib_Version >= 39 and
			GfxBase*.gb_LibNode.lib_Version >= 39 and
			intuiBase*.ib_LibNode.lib_Version >= 39;
		    if OpenIconLibrary(0) ~= nil then
			/* Do this so that errorString will not use StdOut
			   until we know it is OK to do so. */
			FromWorkBench := true;
			TextWindow := nil;
			ErrorPos := 0;
			CharsCopy(&SerialDevice[0], "serial.device");
			CharsCopy(&PortName[0], MUD_PORT_NAME);
			DialString[0] := '\e';
			SerialUnit := 0;
			SerialBaud := 19200;
			RetryLimit := DEFAULT_RETRY_LIMIT;
			Serial := false;
			Serial7Wire := false;
			SerialIgnoreCD := false;
			SerialShared := false;
			SerialReady := false;
			SerialOpen := false;
			SerialBits := 0x00000000;
			EffectsBits := 0x00000000;
			ServerBit := 0x00000000;
			EditorStackSize := DEFAULT_EDITOR_STACK;
			Bright := false;
			Dim := false;
			EditReading := false;
			EditInternal := false;
			EditFilling := false;
			EditorChoice := ed_selected;
			EffectsInited := false;
			EditFirst := true;
			EditFailing := false;
			DisableEdit := false;
			CloseWorkbench := false;
			Mode := md_twoScreen;
			HiRes := false;
			MapColours := true;
			OldStyle := false;
			FullRedraw := false;
			EffectsQuiet := false;
			InitedLib := false;
			FullyRunning := false;
			MUDFlags := 0x00;
			OutputHistorySize := DEFAULT_OUTPUT_HISTORY_SIZE;
			InputHistorySize := DEFAULT_INPUT_HISTORY_SIZE;
			InitialRequest := nil;
			ForceCharacterName := nil;
			ForcePassword := nil;
			UseIP := false;
			IPPort := DEFAULT_IP_PORT;
			IPHost[0] := '\e';
			HadParmError := false;
			Expecting := 0;
			MoreHeader := false;
			HeaderLen :=
			    sizeof(Request_t) - sizeof(Message_t) -TEXT_LENGTH;
			MyTask := FindTask(nil);
			MyProcess := pretend(MyTask, *Process_t);
			if MyProcess*.pr_CLI = 0 then
			    /* running from WorkBench */
			    ignore WaitPort(&MyProcess*.pr_MsgPort);
			    StartupMessage :=
				pretend(GetMsg(&MyProcess*.pr_MsgPort),
					*WBStartup_t);
			    /* NOTE: when running under WorkBench, cannot do
			       any DOS calls until after the above WaitPort! */
			    checkEnvironment();
			    WBRun();
			    Forbid();
			    ReplyMsg(&StartupMessage*.sm_Message);
			else
			    /* running from CLI */
			    FromWorkBench := false;
			    checkEnvironment();
			    StdOut := Output();
			    CLIRun(pretend(MyProcess*.pr_ReturnAddr, *ulong)*);
			fi;
			CloseIconLibrary();
		    else
			errorString("Cannot open icon.library. "
				    "Is it in your LIBS: directory?\n");
		    fi;
		    CloseIntuitionLibrary();
		fi;
		CloseGraphicsLibrary();
	    fi;
	    CloseDosLibrary();
	fi;
	CloseExecLibrary();
    fi;
corp;
