#drinc:exec/nodes.g
#drinc:exec/lists.g
#drinc:exec/tasks.g
#drinc:exec/io.g
#drinc:exec/ports.g
#drinc:devices/serial.g
#drinc:devices/timer.g
#drinc:libraries/dos.g
#drinc:hardware/cia.g
#drinc:OwnDevUnit.g

#/Include/MUD.g
#/Include/Request.g
#globals.g
#funcs.g

/*
 * Amiga MUD
 *
 * Copyright (c) 1997 by Chris Gray
 */

/*
 * serial.d - serial port handling code for MUD.
 */

bool BIG_ENDIAN = true ;

bool DEBUG = false;
bool VERBOSE = false;

uint BITS_PER_BYTE = 8;

ulong
    SIMPLE_WRITE_TIMEOUT = 1,
    SIMPLE_READ_TIMEOUT = 5,		/* want it bigger than write timeout */
    NORMAL_WRITE_TIMEOUT = 59,
    ACK_READ_TIMEOUT = 3,		/* enough for ACK/NAK to return */
    MAIN_READ_TIMEOUT = 20,
    READ_HEADER_TIMEOUT = 2,		/* lots of time for header at 300 bps*/
    READ_BODY_TIMEOUT = 59;		/* enough for 1k bytes at 300 bps */

/*

The header does not actually have a structure for it, but it is:

    SOH - common start byte
    STX - type code for a packet

    CRCHi - high byte of packet 16 bit CRC
    CRCLo - low byte of packet 16 bit CRC
    Seq - modulo 256 sequence number
    LenHi - high byte of 16 bit data len
    LenLo - low byte of 16 bit data len
    Key3 - high byte of 32 bit key
    Key2
    Key1
    Key0 - low byte of 32 bit key
    IdHi - high byte of 16 bit client id
    IdLo - low byte of 16 bit client id
    Type - transaction type code (request type, as in Request.g)
    Flag - a boolean used for whatever

A positive acknowledgement packet consists of

    SOH - common start byte
    ACK - type code for an ack

A negative acknowledgement packet consists of

    SOH - common start byte
    NAK - type code for a nak

*/

uint
    HEADER_LEN = 2 + 1 + 2 + 4 + 2 + 1 + 1,
    INPUT_BUFFER_SIZE = 82,		/* must be at least HEADER_LEN */
    OUTPUT_BUFFER_SIZE = HEADER_LEN + 2;

byte
    SOH = 0x01,
    STX = 0x02,
    ENQ = 0x05,
    ACK = 0x06,
    NAK = 0x15,
    SYN = 0x16;

type
    InputState_t = enum {
	is_packet,		/* waiting for header byte */
	is_header,		/* waiting for fixed header */
	is_body 		/* waiting for variable body */
    },

    OutputState_t = enum {
	os_syn, 		/* sending SYNs */
	os_enq, 		/* sending ENQs */
	os_idle,		/* nothing happening now */
	os_waiting,		/* idle - waiting for ACK/NAK from other end */
	os_header,		/* writing the header */
	os_body,		/* writing a variable body */
	os_acknak		/* writing an ACK or NAK */
    };

*MsgPort_t ReadSerPort, WriteSerPort;
*IOExtSer_t ReadSerReq, WriteSerReq;
IOExtSer_t SaveParams;
*MsgPort_t ReadTimerPort, WriteTimerPort;
*timerequest_t ReadTimerReq, WriteTimerReq;
ulong ReadSerBit, ReadTimerBit, WriteSerBit, WriteTimerBit;
*Task_t MyTask;
*Node_t InputQueue, OutputQueue;
*Request_t InputRequest, OutputRequest;
ulong PacketSent, AckSent, NakSent, WaitTimedOut,
    PacketReceived, AckReceived, NakReceived, ReadNoBytes, WriteFailed;
uint RetryLimit, ReadAttemptCount, WriteAttemptCount;
bool ReadSerBusy, ReadTimerBusy, WriteSerBusy, WriteTimerBusy,
    NeedAck, NeedNak, NeedWait, WantForceConnect, ODUActive;
byte TransmitSequence, ReceiveSequence;
InputState_t InputState;
OutputState_t OutputState;
[INPUT_BUFFER_SIZE] byte InputBuffer;
[OUTPUT_BUFFER_SIZE] byte OutputBuffer;

[1 << BITS_PER_BYTE] uint CRCTable := (
    0x0000, 0xc0c1, 0xc181, 0x0140, 0xc301, 0x03c0, 0x0280, 0xc241,
    0xc601, 0x06c0, 0x0780, 0xc741, 0x0500, 0xc5c1, 0xc481, 0x0440,
    0xcc01, 0x0cc0, 0x0d80, 0xcd41, 0x0f00, 0xcfc1, 0xce81, 0x0e40,
    0x0a00, 0xcac1, 0xcb81, 0x0b40, 0xc901, 0x09c0, 0x0880, 0xc841,
    0xd801, 0x18c0, 0x1980, 0xd941, 0x1b00, 0xdbc1, 0xda81, 0x1a40,
    0x1e00, 0xdec1, 0xdf81, 0x1f40, 0xdd01, 0x1dc0, 0x1c80, 0xdc41,
    0x1400, 0xd4c1, 0xd581, 0x1540, 0xd701, 0x17c0, 0x1680, 0xd641,
    0xd201, 0x12c0, 0x1380, 0xd341, 0x1100, 0xd1c1, 0xd081, 0x1040,
    0xf001, 0x30c0, 0x3180, 0xf141, 0x3300, 0xf3c1, 0xf281, 0x3240,
    0x3600, 0xf6c1, 0xf781, 0x3740, 0xf501, 0x35c0, 0x3480, 0xf441,
    0x3c00, 0xfcc1, 0xfd81, 0x3d40, 0xff01, 0x3fc0, 0x3e80, 0xfe41,
    0xfa01, 0x3ac0, 0x3b80, 0xfb41, 0x3900, 0xf9c1, 0xf881, 0x3840,
    0x2800, 0xe8c1, 0xe981, 0x2940, 0xeb01, 0x2bc0, 0x2a80, 0xea41,
    0xee01, 0x2ec0, 0x2f80, 0xef41, 0x2d00, 0xedc1, 0xec81, 0x2c40,
    0xe401, 0x24c0, 0x2580, 0xe541, 0x2700, 0xe7c1, 0xe681, 0x2640,
    0x2200, 0xe2c1, 0xe381, 0x2340, 0xe101, 0x21c0, 0x2080, 0xe041,
    0xa001, 0x60c0, 0x6180, 0xa141, 0x6300, 0xa3c1, 0xa281, 0x6240,
    0x6600, 0xa6c1, 0xa781, 0x6740, 0xa501, 0x65c0, 0x6480, 0xa441,
    0x6c00, 0xacc1, 0xad81, 0x6d40, 0xaf01, 0x6fc0, 0x6e80, 0xae41,
    0xaa01, 0x6ac0, 0x6b80, 0xab41, 0x6900, 0xa9c1, 0xa881, 0x6840,
    0x7800, 0xb8c1, 0xb981, 0x7940, 0xbb01, 0x7bc0, 0x7a80, 0xba41,
    0xbe01, 0x7ec0, 0x7f80, 0xbf41, 0x7d00, 0xbdc1, 0xbc81, 0x7c40,
    0xb401, 0x74c0, 0x7580, 0xb541, 0x7700, 0xb7c1, 0xb681, 0x7640,
    0x7200, 0xb2c1, 0xb381, 0x7340, 0xb101, 0x71c0, 0x7080, 0xb041,
    0x5000, 0x90c1, 0x9181, 0x5140, 0x9301, 0x53c0, 0x5280, 0x9241,
    0x9601, 0x56c0, 0x5780, 0x9741, 0x5500, 0x95c1, 0x9481, 0x5440,
    0x9c01, 0x5cc0, 0x5d80, 0x9d41, 0x5f00, 0x9fc1, 0x9e81, 0x5e40,
    0x5a00, 0x9ac1, 0x9b81, 0x5b40, 0x9901, 0x59c0, 0x5880, 0x9841,
    0x8801, 0x48c0, 0x4980, 0x8941, 0x4b00, 0x8bc1, 0x8a81, 0x4a40,
    0x4e00, 0x8ec1, 0x8f81, 0x4f40, 0x8d01, 0x4dc0, 0x4c80, 0x8c41,
    0x4400, 0x84c1, 0x8581, 0x4540, 0x8701, 0x47c0, 0x4680, 0x8641,
    0x8201, 0x42c0, 0x4380, 0x8341, 0x4100, 0x81c1, 0x8081, 0x4040
);

/*
proc say(*char s)void:
    *char p;

    p := s;
    while p* ~= '\e' do
	p := p + 1;
    od;
    ignore Write(Output(), s, p - s);
corp;

proc sayNum(ulong n)void:
    [2] char buf;

    if n >= 10 then
	sayNum(n / 10);
    fi;
    buf[0] := n % 10 + '0';
    buf[1] := '\e';
    say(&buf[0]);
corp;

proc sayHex(ulong n)void:
    register *char p;
    register uint i, nibble;
    [9] char buff;

    p := &buff[8];
    p* := '\e';
    for i from 8 - 1 downto 0 do
	p := p - sizeof(char);
	nibble := n & 0xf;
	p* := if nibble >= 10 then nibble - 10 + 'a' else nibble + '0' fi;
	n := n >> 4;
    od;
    say(&buff[0]);
corp;
*/

/*
 * updateCRC - calculate additional CRC for a bufferfull of stuff.
 */

proc updateCRC(register uint crc; register arbptr ip; register uint len)uint:
    register *[1 << BITS_PER_BYTE] uint pTable;
    register *byte p @ ip;

    pTable := &CRCTable;
    while len ~= 0 do
	len := len - 1;
	crc := (crc >> BITS_PER_BYTE) ><
	    pTable*[(crc & ((1 << BITS_PER_BYTE) - 1)) >< p*];
	p := p + 1;
    od;
    crc
corp;

/*
 * checkDoIO - do and check a DoIO write call for the serial device.
 *	Note: this is NOT used for any read or write calls.
 */

proc checkDoIO(register *IOExtSer_t req; uint command; *char message)void:
    uint errnum;

    req*.ios_IOSer.io_io.io_Command := command;
    req*.ios_IOSer.io_io.io_Flags := 0;
    /* apparently this is needed by the Supra 'modem0.device' */
    req*.ios_IOSer.io_Length := 0;
    req*.ios_IOSer.io_Actual := 0;
    if VERBOSE then
	errnum := DoIO(&req*.ios_IOSer.io_io);
	if errnum ~= 0 then
	    errorString("MUD: ");
	    errorString(message);
	    errorString(" error on serial port: ");
	    errorNum(errnum);
	    errorString("\n");
	fi;
    else
	ignore DoIO(&req*.ios_IOSer.io_io);
    fi;
corp;

/*
 * setupRead - common utility to set up a read request.
 */

proc setupRead(arbptr buffer; ulong len)void:
    register *IOExtSer_t serReq;

    serReq := ReadSerReq;
    serReq*.ios_IOSer.io_Length := len;
    serReq*.ios_IOSer.io_io.io_Command := CMD_READ;
    serReq*.ios_IOSer.io_Data := buffer;
    serReq*.ios_IOSer.io_io.io_Flags := 0;
    serReq*.ios_IOSer.io_Actual := 0;	/* in case device needs it */
corp;

/*
 * setupWrite - common utility to set up a write request.
 */

proc setupWrite(arbptr buffer; ulong len)void:
    register *IOExtSer_t serReq;

    serReq := WriteSerReq;
    serReq*.ios_IOSer.io_Length := len;
    serReq*.ios_IOSer.io_io.io_Command := CMD_WRITE;
    serReq*.ios_IOSer.io_Data := buffer;
    serReq*.ios_IOSer.io_io.io_Flags := 0;
    serReq*.ios_IOSer.io_Actual := 0;	/* in case device needs it */
corp;

/*
 * startTwoByteRead - utility to start up a two-byte read request with the
 *	timeout as specified;
 */

proc startTwoByteRead(uint timeout)void:
    register *timerequest_t timerReq;

    setupRead(&InputBuffer[0], 2);

    timerReq := ReadTimerReq;
    timerReq*.tr_time.tv_secs := timeout;
    timerReq*.tr_time.tv_micro := 0;
    timerReq*.tr_node.io_Command := TR_ADDREQUEST;

    ReadSerBusy := true;
    ReadTimerBusy := true;
    SendIO(&timerReq*.tr_node);
    SendIO(&ReadSerReq*.ios_IOSer.io_io);
corp;

/*
 * startTwoByteWrite - utility to start up a two-byte write request.
 */

proc startTwoByteWrite(ulong timeout)void:
    register *timerequest_t timerReq;

    setupWrite(&OutputBuffer[0], 2);
    WriteSerBusy := true;

    if timeout ~= 0 then
	timerReq := WriteTimerReq;
	timerReq*.tr_time.tv_secs := timeout;
	timerReq*.tr_time.tv_micro := 0;
	timerReq*.tr_node.io_Command := TR_ADDREQUEST;
	WriteTimerBusy := true;
	SendIO(&timerReq*.tr_node);
    fi;

    SendIO(&WriteSerReq*.ios_IOSer.io_io);
corp;

/*
 * startWriteRequest - start a chunk of a message write.
 */

proc startWriteRequest(*byte buffer; ulong len)void:
    register *timerequest_t timerReq;

    setupWrite(buffer, len);

    timerReq := WriteTimerReq;
    timerReq*.tr_time.tv_secs := NORMAL_WRITE_TIMEOUT;
    timerReq*.tr_time.tv_micro := 0;
    timerReq*.tr_node.io_Command := TR_ADDREQUEST;

    WriteSerBusy := true;
    WriteTimerBusy := true;
    SendIO(&timerReq*.tr_node);
    SendIO(&WriteSerReq*.ios_IOSer.io_io);
corp;

/*
 * writeOneRequest - write to the server the first request on our outgoing
 *	queue.
 */

proc writeOneRequest()void:
    register *Node_t ln;
    register *Request_t rq @ ln;
    register *byte p, q;
    register uint len;
    uint l;

    ln := OutputQueue;
    /* rq := pretend(ln, *Request_t); */
    p := &OutputBuffer[0];
    p* := SOH;
    p := p + 1;
    p* := STX;
    p := p + 3;
    p* := TransmitSequence;
    p := p + 1;

    if BIG_ENDIAN then
	q := pretend(&rq*.rq_usedLen, *byte);
	p* := q*;
	p := p + 1;
	q := q + 1;
	p* := q*;
	p := p + 1;
	q := pretend(&rq*.rq_key, *byte);
	p* := q*;
	p := p + 1;
	q := q + 1;
	p* := q*;
	p := p + 1;
	q := q + 1;
	p* := q*;
	p := p + 1;
	q := q + 1;
	p* := q*;
	p := p + 1;
	q := pretend(&rq*.rq_clientId, *byte);
	p* := q*;
	p := p + 1;
	q := q + 1;
	p* := q*;
	p := p + 1;
    else
	len := rq*.rq_usedLen;
	p* := len >> 8;
	p := p + 1;
	p* := len;
	p := p + 1;
	p* := rq*.rq_key >> 24;
	p := p + 1;
	p* := rq*.rq_key >> 16;
	p := p + 1;
	p* := rq*.rq_key >> 8;
	p := p + 1;
	p* := rq*.rq_key;
	p := p + 1;
	p* := rq*.rq_clientId >> 8;
	p := p + 1;
	p* := rq*.rq_clientId;
	p := p + 1;
    fi;

    p* := rq*.rq_type - rt_first;
    p := p + 1;
    p* := rq*.rq_flag - false;

    l := updateCRC(updateCRC(0, &OutputBuffer[2 + 2], HEADER_LEN - 2),
		   &rq*.rq_u, rq*.rq_usedLen);

    if BIG_ENDIAN then
	p := &OutputBuffer[2];
	q := pretend(&l, *byte);
	p* := q*;
	q := q + 1;
	p := p + 1;
	p* := q*;
    else
	OutputBuffer[2] := l >> 8;
	OutputBuffer[3] := l;
    fi;

    OutputState := os_header;
    OutputRequest := rq;
    startWriteRequest(&OutputBuffer[0], HEADER_LEN + 2);
    PacketSent := PacketSent + 1;
corp;

/*
 * startNormalReadRequest - start up a normal read to get a piece of a
 *	server message.
 */

proc startNormalReadRequest(*byte buffer; ulong len, timeout)void:
    register *timerequest_t timerReq;

    setupRead(buffer, len);

    timerReq := ReadTimerReq;
    timerReq*.tr_time.tv_secs := timeout;
    timerReq*.tr_time.tv_micro := 0;
    timerReq*.tr_node.io_Command := TR_ADDREQUEST;

    ReadSerBusy := true;
    ReadTimerBusy := true;
    SendIO(&timerReq*.tr_node);
    SendIO(&ReadSerReq*.ios_IOSer.io_io);
corp;

/* bunches of little utilities */

proc abortWriteTimer()void:

    WriteTimerBusy := false;
    if CheckIO(&WriteTimerReq*.tr_node) = nil then
	ignore AbortIO(&WriteTimerReq*.tr_node);
    fi;
    ignore WaitIO(&WriteTimerReq*.tr_node);
    ignore SetSignal(0x0, WriteTimerBit);
corp;

proc abortWrite()void:

    WriteSerBusy := false;
    if CheckIO(&WriteSerReq*.ios_IOSer.io_io) = nil then
	ignore AbortIO(&WriteSerReq*.ios_IOSer.io_io);
    fi;
    ignore WaitIO(&WriteSerReq*.ios_IOSer.io_io);
corp;

proc abortReadTimer()void:

    ReadTimerBusy := false;
    if CheckIO(&ReadTimerReq*.tr_node) = nil then
	ignore AbortIO(&ReadTimerReq*.tr_node);
    fi;
    ignore WaitIO(&ReadTimerReq*.tr_node);
    ignore SetSignal(0x0, ReadTimerBit);
corp;

proc abortRead()void:

    ReadSerBusy := false;
    if CheckIO(&ReadSerReq*.ios_IOSer.io_io) = nil then
	ignore AbortIO(&ReadSerReq*.ios_IOSer.io_io);
    fi;
    ignore WaitIO(&ReadSerReq*.ios_IOSer.io_io);
corp;

/*
 * flushInput - flush all the bytes coming in.
 */

proc flushInput()void:
    register *IOExtSer_t req;
    register ulong len;

    if DEBUG then
	errorString("flushInput\n");
    fi;
    req := ReadSerReq;
    while
	Delay(5);
	checkDoIO(req, SDCMD_QUERY, "flush");
	len := req*.ios_IOSer.io_Actual;
	len ~= 0
    do
	if len > INPUT_BUFFER_SIZE then
	    len := INPUT_BUFFER_SIZE;
	fi;
	setupRead(&InputBuffer[0], len);
	ignore DoIO(&req*.ios_IOSer.io_io);
    od;
corp;

/*
 * sendSyn/sendEnq/sendAck/sendNak - handy utilities
 */

proc sendSyn()void:

    if DEBUG then
	errorString("sendSyn\n");
    fi;
    OutputBuffer[0] := SYN;
    OutputBuffer[1] := SYN;
    startTwoByteWrite(SIMPLE_WRITE_TIMEOUT);
corp;

proc sendEnq()void:

    if DEBUG then
	errorString("sendEnq\n");
    fi;
    OutputBuffer[0] := ENQ;
    OutputBuffer[1] := ENQ;
    startTwoByteWrite(SIMPLE_WRITE_TIMEOUT);
corp;

proc sendAck()void:

    if DEBUG then
	errorString("sendAck\n");
    fi;
    OutputState := os_acknak;
    OutputBuffer[0] := SOH;
    OutputBuffer[1] := ACK;
    startTwoByteWrite(0);
    AckSent := AckSent + 1;
corp;

proc sendNak()void:

    if DEBUG then
	errorString("sendNak\n");
    fi;
    OutputState := os_acknak;
    OutputBuffer[0] := SOH;
    OutputBuffer[1] := NAK;
    startTwoByteWrite(0);
    NakSent := NakSent + 1;
corp;

/*
 * checkAckNak - check if ACK or NAK needed. If so - start one.
 *	Return 'true' if we made the output busy.
 */

proc checkAckNak()bool:

    if NeedAck then
	NeedAck := false;
	sendAck();
	true
    elif NeedNak then
	NeedNak := false;
	sendNak();
	true
    else
	false
    fi
corp;

/*
 * killAll - kill off anything that is happening.
 */

proc killAll()void:

    if DEBUG then
	errorString("killAll\n");
    fi;
    if WriteSerBusy then
	abortWrite();
    fi;
    if WriteTimerBusy then
	abortWriteTimer();
    fi;
    if ReadSerBusy then
	abortRead();
    fi;
    if ReadTimerBusy then
	abortReadTimer();
    fi;
corp;

/*
 * writeRetry - we need to retry a write attempt
 */

proc writeRetry()void:

    if DEBUG then
	errorString("writeRetry\n");
    fi;
    WriteAttemptCount := WriteAttemptCount + 1;
    if WriteAttemptCount = RetryLimit then
	errorString("MUD: serial write failed - aborting\n");
	killAll();
	doExit();
    fi;
    writeOneRequest();
corp;

/*
 * readRetry - something wrong with a read - see if we should give up.
 */

proc readRetry()void:

    if DEBUG then
	errorString("readRetry\n");
    fi;
    ReadAttemptCount := ReadAttemptCount + 1;
    if ReadAttemptCount = RetryLimit then
	errorString("MUD: serial read failed - aborting\n");
	killAll();
	doExit();
    fi;
corp;

/*
 * handleNormalReceive - handle the receipt of a normal 2-byte preheader.
 */

proc handleNormalReceive()void:
    register *Node_t ln;
    register *Request_t rq @ ln;

    case InputBuffer[1]
    incase STX:
	/* regular incoming transaction */
	if DEBUG then
	    errorString("handleNormalReceive -> is_header\n");
	fi;
	InputState := is_header;
	ReadAttemptCount := 0;
	startNormalReadRequest(&InputBuffer[0], HEADER_LEN,
			       READ_HEADER_TIMEOUT);
    incase ACK:
	if DEBUG then
	    errorString("handleNormalReceive: ACK\n");
	fi;
	AckReceived := AckReceived + 1;
	if OutputState = os_waiting or NeedWait then
	    /* positive acknowledgement of the last message we sent */
	    if OutputState = os_waiting then
		abortWriteTimer();
		OutputState := os_idle;
	    fi;
	    NeedWait := false;
	    TransmitSequence := TransmitSequence + 1;
	    ln := OutputQueue;
	    OutputQueue := ln*.ln_Succ;
	    /* rq := pretend(ln, *Request_t); */
	    if rq*.rq_type = rt_endClient then
		/* this session is over - return from top level */
		myFreeRequest(rq);
	    else
		myFreeRequest(rq);
		if not WriteSerBusy and OutputQueue ~= nil then
		    WriteAttemptCount := 0;
		    writeOneRequest();
		else
		    OutputState := os_idle;
		fi;
	    fi;
	else
	    flushInput();
	fi;
	startTwoByteRead(MAIN_READ_TIMEOUT);
    incase NAK:
	if DEBUG then
	    errorString("handleNormalReceive: NAK\n");
	fi;
	NakReceived := NakReceived + 1;
	if OutputState = os_waiting or NeedWait then
	    /* negative acknowledgement of the last message we sent - retry */
	    if OutputState = os_waiting then
		abortWriteTimer();
		OutputState := os_idle;
	    fi;
	    NeedWait := false;
	    if not WriteSerBusy then
		writeRetry();
	    fi;
	else
	    flushInput();
	fi;
	startTwoByteRead(MAIN_READ_TIMEOUT);
    default:
	/* anything else is invalid - just go for another preheader */
	if DEBUG then
	    errorString("handleNormalReceive - flushing\n");
	fi;
	flushInput();
	startTwoByteRead(MAIN_READ_TIMEOUT);
    esac;
corp;

/*
 * receivedMessage - all done receiving a message - check and handle it.
 */

proc receivedMessage(register *Request_t rq)void:
    register **Node_t tail;
    register uint crc, gotCRC;

    PacketReceived := PacketReceived + 1;
    InputState := is_packet;
    crc := updateCRC(0, &InputBuffer[2], HEADER_LEN - 2);
    crc := updateCRC(crc, &rq*.rq_u, rq*.rq_usedLen);
    gotCRC := (make(InputBuffer[0], uint) << 8) | InputBuffer[1];
    if gotCRC = crc then
	/* received OK - ACK it */
	if not WriteSerBusy then
	    if OutputState = os_waiting then
		abortWriteTimer();
		NeedWait := true;
	    fi;
	    sendAck();
	else
	    NeedAck := true;
	fi;
	/* We need this sequence number check because:
	   - other end sends message with sequence N
	   - we receive it OK, ACK it and process it
	   - ACK gets garbled or lost
	   - other end times out and resends the message
	   - we do NOT want to process it, but we do ACK it
	 */
	if InputBuffer[2] + 1 = ReceiveSequence then
	    /* it is a new message - put it on the incoming queue */
	    ReceiveSequence := ReceiveSequence + 1;
	    tail := &InputQueue;
	    while tail* ~= nil do
		tail := &tail**.ln_Succ;
	    od;
	    rq*.rq_message.mn_Node.ln_Pred := tail*;
	    tail* := &rq*.rq_message.mn_Node;
	    rq*.rq_message.mn_Node.ln_Succ := nil;
	else
	    readRetry();
	fi;
    else
	/* not received OK - NAK it */
	readRetry();
	if not WriteSerBusy then
	    if OutputState = os_waiting then
		abortWriteTimer();
		NeedWait := true;
	    fi;
	    sendNak();
	else
	    NeedNak := true;
	fi;
    fi;
    startTwoByteRead(MAIN_READ_TIMEOUT);
corp;

/*
 * startAckTimeout - we go to os_waiting, and start a timer. If the timer
 *	goes off before we get an ACK or NAK, we will just resend.
 */

proc startAckTimeout()void:
    register *timerequest_t timerReq;

    OutputState := os_waiting;

    timerReq := WriteTimerReq;
    timerReq*.tr_time.tv_secs := ACK_READ_TIMEOUT;
    timerReq*.tr_time.tv_micro := 0;
    timerReq*.tr_node.io_Command := TR_ADDREQUEST;

    WriteTimerBusy := true;
    SendIO(&timerReq*.tr_node);
corp;

/*
 * handleSerialEvents - called from master Wait - something of interest to
 *	us here has happened.
 */

proc handleSerialEvents(register ulong bits)void:
    register *byte p, q;
    register *Node_t ln;
    register *Request_t rq @ ln;
    register **Node_t tail @ p;
    register ulong ul;
    register uint ui @ ul;
    register OutputState_t os;

    os := OutputState;

    ul := WriteSerBit;
    if bits & ul ~= 0x0 then
	if DEBUG then
	    errorString("write done signal\n");
	fi;
	ignore SetSignal(0x0, ul);
	if GetMsg(WriteSerPort) ~= nil and WriteSerBusy then
	    /* a write request has completed */
	    if DEBUG then
		errorString("write message back\n");
	    fi;
	    WriteSerBusy := false;
	    /* if SYNing or ENQing, then nothing to do - next SYN or ENQ will
	       be sent when the 1 second write timer goes off */
	    if os ~= os_syn and os ~= os_enq then
		if DEBUG then
		    errorString("not syn-ing or enq-ing\n");
		fi;
		if WriteTimerBusy then
		    abortWriteTimer();
		fi;
		case os
		incase os_idle:
		    /* we get set to os_idle when the initial handshake is
		       done, but something is going out */
		    if not checkAckNak() and OutputQueue ~= nil then
			WriteAttemptCount := 0;
			writeOneRequest();
		    fi;
		incase os_waiting:
		    errorString("MUD: serial write done when idle????\n");
		incase os_header:
		    rq := pretend(OutputQueue, *Request_t);
		    if rq*.rq_usedLen = 0 then
			if not checkAckNak() then
			    startAckTimeout();
			else
			    NeedWait := true;
			fi;
		    else
			OutputState := os_body;
			WriteAttemptCount := 0;
			startWriteRequest(&rq*.rq_u.ru_bytes[0],
					  rq*.rq_usedLen);
		    fi;
		incase os_body:
		    if not checkAckNak() then
			startAckTimeout();
		    else
			NeedWait := true;
		    fi;
		incase os_acknak:
		    if not checkAckNak() then
			if NeedWait then
			    NeedWait := false;
			    startAckTimeout();
			elif OutputQueue ~= nil then
			    WriteAttemptCount := 0;
			    writeOneRequest();
			else
			    OutputState := os_idle;
			fi;
		    fi;
		esac;
	    fi;
	fi;
    fi;

    ul := WriteTimerBit;
    if bits & ul ~= 0x0 then
	if DEBUG then
	    errorString("write timer signal\n");
	fi;
	ignore SetSignal(0x0, ul);
	if GetMsg(WriteTimerPort) ~= nil and WriteTimerBusy then
	    /* the write request timer has gone off */
	    if DEBUG then
		errorString("write timer message back\n");
	    fi;
	    WriteTimerBusy := false;
	    if WriteSerBusy then
		/* abort the hung write */
		abortWrite();
	    fi;
	    case os
	    incase os_syn:
		/* just keep pumping out the SYNs */
		sendSyn();
	    incase os_enq:
		/* start/continue pumping out the ENQs */
		sendEnq();
	    incase os_idle:
		errorString(
		    "MUD: serial port write timeout when not writing???\n");
	    incase os_waiting:
		/* timeout waiting for ACK or NAK - retry the message */
		if not SerialIgnoreCD then
		    checkDoIO(WriteSerReq, SDCMD_QUERY, "WTO query");
		    if WriteSerReq*.ios_Status & CIAF_COMCD ~= 0 then
			while
			    ln := OutputQueue;
			    /* rq := pretend(ln, *Request_t); */
			    rq ~= nil and rq*.rq_type = rt_log
			do
			    /* just ignore any lost log requests */
			    OutputQueue := ln*.ln_Succ;
			    myFreeRequest(rq);
			od;
			if rq ~= nil and rq*.rq_type = rt_endClient then
			    /* Just fake a reply to it */
			    tail := &InputQueue;
			    while tail* ~= nil do
				tail := &tail**.ln_Succ;
			    od;
			    rq*.rq_message.mn_Node.ln_Pred := tail*;
			    tail* := &rq*.rq_message.mn_Node;
			    rq*.rq_message.mn_Node.ln_Succ := nil;
			else
			    errorString(
				"MUD: serial carrier lost 1 - aborting\n");
			    killAll();
			    doExit();
			fi;
		    fi;
		fi;
		if OutputQueue ~= nil then
		    WaitTimedOut := WaitTimedOut + 1;
		    if not checkAckNak() then
			writeRetry();
		    fi;
		fi;
	    incase os_header:
	    incase os_body:
		/* something gone wrong in transmit - retry it */
		WriteFailed := WriteFailed + 1;
		if not checkAckNak() then
		    writeRetry();
		fi;
	    incase os_acknak:
		/* there should be no write timer on these! */
		errorString("MUD: serial port write timeout on ACK/NAK???\n");
	    esac;
	fi;
    fi;

    ul := ReadSerBit;
    if bits & ul ~= 0x0 then
	if DEBUG then
	    errorString("read done signal\n");
	fi;
	ignore SetSignal(0x0, ul);
	if GetMsg(ReadSerPort) ~= nil and ReadSerBusy then
	    /* a read request completion has been signalled */
	    if DEBUG then
		errorString("read message back\n");
	    fi;
	    ReadSerBusy := false;
	    if ReadTimerBusy then
		abortReadTimer();
	    fi;
	    if ReadSerReq*.ios_IOSer.io_Actual ~= 0 then
		case InputState
		incase is_packet:
		    case os
		    incase os_syn:
			if DEBUG then
			    errorString("os_syn:\n");
			fi;
			if InputBuffer[0] = SYN or InputBuffer[0] = ENQ then
			    /* other end is there - move up */
			    if DEBUG then
				errorString("os_syn -> os_enq\n");
			    fi;
			    OutputState := os_enq;
			    if not WriteSerBusy then
				abortWriteTimer();
				sendEnq();
			    fi;
			else
			    /* anything else - ignore it */
			    flushInput();
			fi;
			startTwoByteRead(SIMPLE_READ_TIMEOUT);
		    incase os_enq:
			if DEBUG then
			    errorString("os_enq:\n");
			fi;
			if InputBuffer[0] = ENQ and OutputQueue ~= nil then
			    /* Other end moved up - move up here too. Do not
			       want to move up if we have no output message,
			       since if we are ahead of the other end in
			       this, he might not get an ENQ, and will not
			       then move to sending us anything. Experience
			       has shown that the connection WILL fail if we
			       do not check OutputQueue here */
			    if DEBUG then
				errorString("os_enq -> os_idle 1\n");
			    fi;
			    OutputState := os_idle;
			    if not WriteSerBusy then
				abortWriteTimer();
				WriteAttemptCount := 0;
				writeOneRequest();
			    fi;
			    ReadAttemptCount := 0;	/* use to count ENQs */
			    startTwoByteRead(SIMPLE_READ_TIMEOUT);
			elif InputBuffer[0] = SOH then
			    /* other end already sending - accept it */
			    if DEBUG then
				errorString("os_enq -> os_idle 2\n");
			    fi;
			    OutputState := os_idle;
			    abortWriteTimer();
			    if not WriteSerBusy and OutputQueue ~= nil then
				WriteAttemptCount := 0;
				writeOneRequest();
			    fi;
			    handleNormalReceive();
			else
			    /* ignore anything else */
			    flushInput();
			    readRetry();
			    startTwoByteRead(SIMPLE_READ_TIMEOUT);
			fi;
		    incase os_idle:
		    incase os_waiting:
		    incase os_header:
		    incase os_body:
		    incase os_acknak:
			if InputBuffer[0] = SOH then
			    handleNormalReceive();
			elif InputBuffer[0] = ENQ then
			    readRetry();
			    startTwoByteRead(MAIN_READ_TIMEOUT);
			else
			    /* assume it was noise - ignore it */
			    flushInput();
			    readRetry();
			    startTwoByteRead(MAIN_READ_TIMEOUT);
			fi;
		    esac;
		incase is_header:
		    if DEBUG then
			errorString("is_header:\n");
		    fi;
		    p := &InputBuffer[3];
		    ui := make(p*, uint) << 8;
		    p := p + 1;
		    ui := ui | p*;
		    p := p + 1;
		    rq := myAllocRequest(ui);
		    InputRequest := rq;
		    rq*.rq_usedLen := ui;
		    /* big endian hack! */
		    q := pretend(&rq*.rq_key, *byte);
		    q* := p*;
		    q := q + 1;
		    p := p + 1;
		    q* := p*;
		    q := q + 1;
		    p := p + 1;
		    q* := p*;
		    q := q + 1;
		    p := p + 1;
		    q* := p*;
		    p := p + 1;
		    q := pretend(&rq*.rq_clientId, *byte);
		    q* := p*;
		    q := q + 1;
		    p := p + 1;
		    q* := p*;
		    p := p + 1;
/*
		    ul := make(p*, ulong) << 24;
		    p := p + 1;
		    ul := ul | make(p*, ulong) << 16;
		    p := p + 1;
		    ul := ul + make(p*, ulong) << 8;
		    p := p + 1;
		    ul := ul | p*;
		    p := p + 1;
		    rq*.rq_key := ul;
		    ui := make(p*, uint) << 8;
		    p := p + 1;
		    ui := ui | p*;
		    p := p + 1;
		    rq*.rq_clientId := ui;
*/
		    rq*.rq_type := p* + rt_first;
		    p := p + 1;
		    rq*.rq_flag := p* + false;
		    if rq*.rq_usedLen = 0 then
			receivedMessage(rq);
		    else
			InputState := is_body;
			ReadAttemptCount := 0;
			startNormalReadRequest(&rq*.rq_u.ru_bytes[0],
					       rq*.rq_usedLen,
					       READ_BODY_TIMEOUT);
		    fi;
		incase is_body:
		    receivedMessage(InputRequest);
		esac;
	    else
		/* a spurious return from serial read - just redo it */
		if DEBUG then
		    errorString("read no bytes\n");
		fi;
		ReadNoBytes := ReadNoBytes + 1;
		if not SerialIgnoreCD then
		    checkDoIO(ReadSerReq, SDCMD_QUERY, "RNB query");
		    if ReadSerReq*.ios_Status & CIAF_COMCD ~= 0 then
			errorString("MUD: serial carrier lost 2 - aborting\n");
			killAll();
			doExit();
		    fi;
		fi;
		case InputState
		incase is_packet:
		    if os = os_syn or os = os_enq then
			startTwoByteRead(SIMPLE_READ_TIMEOUT);
		    else
			startTwoByteRead(MAIN_READ_TIMEOUT);
		    fi;
		incase is_header:
		    readRetry();
		    startNormalReadRequest(&InputBuffer[0], HEADER_LEN,
					   READ_HEADER_TIMEOUT);
		incase is_body:
		    readRetry();
		    startNormalReadRequest(&InputRequest*.rq_u.ru_bytes[0],
					   InputRequest*.rq_usedLen,
					   READ_BODY_TIMEOUT);
		esac;
	    fi;
	fi;
    fi;

    ul := ReadTimerBit;
    if bits & ul ~= 0x0 then
	if DEBUG then
	    errorString("read timer signal\n");
	fi;
	ignore SetSignal(0x0, ul);
	if GetMsg(ReadTimerPort) ~= nil and ReadTimerBusy then
	    /* a read timer has gone off */
	    if DEBUG then
		errorString("read timer message back\n");
	    fi;
	    ReadTimerBusy := false;
	    if ReadSerBusy then
		abortRead();
	    fi;
	    if not SerialIgnoreCD then
		checkDoIO(ReadSerReq, SDCMD_QUERY, "RTO query");
		if ReadSerReq*.ios_Status & CIAF_COMCD ~= 0 then
		    errorString("MUD: serial carrier lost 3 - aborting\n");
		    killAll();
		    doExit();
		fi;
	    fi;
	    case InputState
	    incase is_packet:
		if os = os_syn or os = os_enq then
		    startTwoByteRead(SIMPLE_READ_TIMEOUT);
		else
		    startTwoByteRead(MAIN_READ_TIMEOUT);
		fi;
	    incase is_header:
		readRetry();
		startNormalReadRequest(&InputBuffer[0], HEADER_LEN,
				       READ_HEADER_TIMEOUT);
	    incase is_body:
		readRetry();
		startNormalReadRequest(&InputRequest*.rq_u.ru_bytes[0],
				       InputRequest*.rq_usedLen,
				       READ_BODY_TIMEOUT);
	    esac;
	fi;
    fi;
corp;

/*
 * connectToServer - any initial conversation is over - establish a binary
 *	protocol connection to the server on the other end.
 */

proc connectToServer()void:

    NeedAck := false;
    NeedNak := false;
    NeedWait := false;
    TransmitSequence := 0;
    ReceiveSequence := 1;
    PacketSent := 0;
    AckSent := 0;
    NakSent := 0;
    WaitTimedOut := 0;
    PacketReceived := 0;
    AckReceived := 0;
    NakReceived := 0;
    ReadNoBytes := 0;
    WriteFailed := 0;
    ReadAttemptCount := 0;
    WriteAttemptCount := 0;

    InputState := is_packet;
    OutputState := os_syn;
    startTwoByteRead(SIMPLE_READ_TIMEOUT);
    sendSyn();
corp;

/*
 * serialStats - show the current packet statistics.
 */

proc serialStats()void:

    putString("::Serial port protocol statistics:\n");
    putString("::  Packets TX: ");
    putUnsigned(PacketSent);
    putString("  ACKs TX: ");
    putUnsigned(AckSent);
    putString("  NAKs TX: ");
    putUnsigned(NakSent);
    putString("\n::  Packets RX: ");
    putUnsigned(PacketReceived);
    putString("  ACKs RX: ");
    putUnsigned(AckReceived);
    putString("  NAKs RX: ");
    putUnsigned(NakReceived);
    putString("\n::  Waits timed out: ");
    putUnsigned(WaitTimedOut);
    putString("  Read no bytes: ");
    putUnsigned(ReadNoBytes);
    putString("  Writes failed: ");
    putUnsigned(WriteFailed);
    newLine();
corp;

/* some little utility routines to make things more readable */

proc createSerIO(*MsgPort_t port)*IOExtSer_t:

    pretend(CreateExtIO(port, sizeof(IOExtSer_t)), *IOExtSer_t)
corp;

proc deleteSerIO(*IOExtSer_t req)void:

    DeleteExtIO(&req*.ios_IOSer.io_io, sizeof(IOExtSer_t));
corp;

proc createTimerIO(*MsgPort_t port)*timerequest_t:

    pretend(CreateExtIO(port, sizeof(timerequest_t)), *timerequest_t)
corp;

proc deleteTimerIO(*timerequest_t req)void:

    DeleteExtIO(&req*.tr_node, sizeof(timerequest_t));
corp;

/*
 * closeSerialHandler - final close of serial device.
 */

proc clearQueue(register *Node_t ln)void:
    register *Request_t rq;

    while ln ~= nil do
	rq := pretend(ln, *Request_t);
	ln := ln*.ln_Succ;
	myFreeRequest(rq);
    od;
corp;

proc closeSerialHandler(*char device; ulong unit)void:

    killAll();

    SaveParams.ios_IOSer := ReadSerReq*.ios_IOSer;
    ReadSerReq* := SaveParams;
    checkDoIO(ReadSerReq, SDCMD_SETPARAMS, "close");

    clearQueue(InputQueue);
    clearQueue(OutputQueue);
    CloseDevice(&ReadTimerReq*.tr_node);
    CloseDevice(&ReadSerReq*.ios_IOSer.io_io);
    deleteTimerIO(WriteTimerReq);
    deleteTimerIO(ReadTimerReq);
    deleteSerIO(WriteSerReq);
    deleteSerIO(ReadSerReq);
    DeletePort(WriteTimerPort);
    DeletePort(ReadTimerPort);
    DeletePort(WriteSerPort);
    DeletePort(ReadSerPort);
    if ODUActive then
	ODUActive := false;
	FreeDevUnit(device, unit);
	CloseOwnDevUnitLibrary();
    fi;
corp;

/*
 * doSetParams - fill in the data for an SDCMD_SETPARAMS, and do it.
 */

proc doSetParams(register *IOExtSer_t req)void:

    req* := SaveParams;
    req*.ios_RBufLen := 2048;
    req*.ios_Baud := SerialBaud;
    req*.ios_BrkTime := 1000000;
    req*.ios_ReadLen := 8;
    req*.ios_WriteLen := 8;
    req*.ios_StopBits := 1;
    req*.ios_SerFlags := SERF_XDISABLED | SERF_RAD_BOOGIE;
    checkDoIO(req, SDCMD_SETPARAMS, "setParams");
corp;

/*
 * openSerialHandler - initial open of serial device. Return 'true' if worked.
 */

proc openSerialHandler(*char device; ulong unit, retryLimit;
		       bool ready, shared, _7Wire)ulong:
    register *IOExtSer_t req;
    register ushort flags, serBits;
    *char ODUResult;
    bool ok;

    ODUActive := false;
    if OpenOwnDevUnitLibrary(0) ~= nil then
	ODUResult := AttemptDevUnit(device, unit, "MUD", 0);
	if ODUResult ~= nil then
	    if ODUResult* ~= ODUERR_LEADCHAR then
		errorString("Cannot open port - in use by ");
		errorString(ODUResult);
		errorChar('\n');
	    fi;
	    CloseOwnDevUnitLibrary();
	    return(0);
	fi;
	ODUActive := true;
    fi;

    RetryLimit := retryLimit;
    flags := SERF_XDISABLED | SERF_RAD_BOOGIE;
    if _7Wire then
	flags := flags | SERF_7WIRE;
    fi;
    if shared then
	flags := flags | SERF_SHARED;
    fi;
    ReadSerPort := CreatePort(nil, 0);
    if ReadSerPort ~= nil then
      WriteSerPort := CreatePort(nil, 0);
      if WriteSerPort ~= nil then
	ReadTimerPort := CreatePort(nil, 0);
	if ReadTimerPort ~= nil then
	  WriteTimerPort := CreatePort(nil, 0);
	  if WriteTimerPort ~= nil then
	    ReadSerReq := createSerIO(ReadSerPort);
	    if ReadSerReq ~= nil then
	      WriteSerReq := createSerIO(WriteSerPort);
	      if WriteSerReq ~= nil then
		ReadTimerReq := createTimerIO(ReadTimerPort);
		if ReadTimerReq ~= nil then
		  WriteTimerReq := createTimerIO(WriteTimerPort);
		  if WriteTimerReq ~= nil then
		    ReadSerReq*.ios_SerFlags := flags;
		    if OpenDevice(device, unit,
				  &ReadSerReq*.ios_IOSer.io_io, 0) = 0
		    then
		      if OpenDevice(TIMERNAME, UNIT_VBLANK,
				    &ReadTimerReq*.tr_node, 0) = 0
		      then
			MyTask := FindTask(nil);
			SaveParams := ReadSerReq*;
			WriteSerReq* := ReadSerReq*;
			WriteSerReq*.ios_IOSer.io_io.io_Message.mn_ReplyPort :=
			    WriteSerPort;
			WriteTimerReq*.tr_node.io_Device :=
			    ReadTimerReq*.tr_node.io_Device;
			WriteTimerReq*.tr_node.io_Unit :=
			    ReadTimerReq*.tr_node.io_Unit;
			WriteTimerReq*.tr_node.io_Flags :=
			    ReadTimerReq*.tr_node.io_Flags;

			ok := true;
			if ready then
			  if SaveParams.ios_ReadLen ~= 8 or
			    SaveParams.ios_WriteLen ~= 8
			  then
			    errorString(
				"Sorry, 8 bits per character required "
				"for binary connection.\n");
			    ok := false;
			  fi;
			else
			  req := ReadSerReq;
			  checkDoIO(req, CMD_CLEAR, "setParams 1");
			  doSetParams(req);
			  checkDoIO(req, CMD_CLEAR, "setParams 2");
			fi;
		
			InputQueue := nil;
			OutputQueue := nil;
			ReadSerBusy := false;
			ReadTimerBusy := false;
			WriteSerBusy := false;
			WriteTimerBusy := false;

			ReadSerBit := 1 << ReadSerPort*.mp_SigBit;
			ReadTimerBit := 1 << ReadTimerPort*.mp_SigBit;
			WriteSerBit := 1 << WriteSerPort*.mp_SigBit;
			WriteTimerBit := 1 << WriteTimerPort*.mp_SigBit;

			if ok then
			  return(ReadSerBit  | ReadTimerBit |
				 WriteSerBit | WriteTimerBit);
			fi;
		      fi;
		      CloseDevice(&ReadSerReq*.ios_IOSer.io_io);
		    fi;
		    deleteTimerIO(WriteTimerReq);
		  fi;
		  deleteTimerIO(ReadTimerReq);
		fi;
		deleteSerIO(WriteSerReq);
	      fi;
	      deleteSerIO(ReadSerReq);
	    fi;
	    DeletePort(WriteTimerPort);
	  fi;
	  DeletePort(ReadTimerPort);
	fi;
	DeletePort(WriteSerPort);
      fi;
      DeletePort(ReadSerPort);
    fi;
    0
corp;

/*
 * serialSendMessage - send a message to the server.
 */

proc serialSendMessage(register *Request_t rq)void:
    register **Node_t tail;

    rq*.rq_message.mn_Node.ln_Succ := nil;
    tail := &OutputQueue;
    while tail* ~= nil do
	tail := &tail**.ln_Succ;
    od;
    rq*.rq_message.mn_Node.ln_Pred := tail*;
    tail* := &rq*.rq_message.mn_Node;
    if not WriteSerBusy and not WriteTimerBusy then
	writeOneRequest();
    fi;
corp;

/*
 * serialGetMessage - try to get a message from the server. Return nil if
 *	we do not have one yet.
 */

proc serialGetMessage()*Request_t:
    register *Node_t ln;

    ln := InputQueue;
    if ln ~= nil then
	InputQueue := ln*.ln_Succ;
    fi;
    pretend(ln, *Request_t)
corp;

/*
 * serialPutChar - output a single character.
 */

proc serialPutChar(char ch)void:

    setupWrite(&ch, 1);
    ignore DoIO(&WriteSerReq*.ios_IOSer.io_io);
corp;

/*
 * serialConversation - fake a simple terminal program to allow connection.
 *	Return 'false' if we do not want to try connecting.
 */

proc serialConversation(*MsgPort_t intuitionPort;
			register *char dialString)bool:
    register *IOExtSer_t serReq @ dialString;
    register *timerequest_t timerReq;
    register *Message_t m;
    register ulong intuitionBit, bits, len, i;
    bool result, wantExit;

    killAll();
    clearQueue(InputQueue);
    clearQueue(OutputQueue);
    ReadSerBusy := false;
    ReadTimerBusy := false;

    setTerminalMode();
    if dialString* ~= '\e' then
	Delay(50);
	while dialString* ~= '\e' do
	    serialPutChar(dialString*);
	    dialString := dialString + sizeof(char);
	od;
	serialPutChar('\r');
    fi;
    WantForceConnect := false;
    serReq := ReadSerReq;
    intuitionBit := 1 << intuitionPort*.mp_SigBit;
    ReadSerBusy := false;
    while true do
	if not ReadSerBusy then
	    checkDoIO(serReq, SDCMD_QUERY, "term");
	    len := serReq*.ios_IOSer.io_Actual;
	    if len = 0 then
		len := 1;
	    elif len > INPUT_BUFFER_SIZE then
		len := INPUT_BUFFER_SIZE;
	    fi;
	    setupRead(&InputBuffer[0], len);
	    /* if there were bytes out there already, this will come back
	       right away, but if not, we will wait for a character */
	    SendIO(&serReq*.ios_IOSer.io_io);
	    ReadSerBusy := true;
	fi;
	bits :=
	    ReadSerBit | intuitionBit | SIGBREAKF_CTRL_C | SIGBREAKF_CTRL_F;
	bits := Wait(bits);
	if bits & intuitionBit ~= 0 then
	    wantExit := false;
	    while
		m := GetMsg(intuitionPort);
		m ~= nil
	    do
		if not handleMessage(pretend(m, *IntuiMessage_t), false) then
		    /* removeMenus already called */
		    if ReadSerBusy then
			abortRead();
		    fi;
		    wantExit := true;
		    result := false;
		elif WantForceConnect then
		    WantForceConnect := false;
		    if ReadSerBusy then
			abortRead();
		    fi;
		    clearTerminalMode();
		    putString("\nCONNECTING TO REMOTE END\n");
		    connectToServer();
		    wantExit := true;
		    result := true;
		fi;
	    od;
	fi;
	if wantExit then
	    return(result);
	fi;
	if bits & ReadSerBit ~= 0 then
	    if GetMsg(ReadSerPort) ~= nil then
		ReadSerBusy := false;
		len := serReq*.ios_IOSer.io_Actual;
		if len = 2 and InputBuffer[0] = SYN and InputBuffer[1] = SYN
		then
		    clearTerminalMode();
		    putString("\nREMOTE END DETECTED\n");
		    connectToServer();
		    return(true);
		fi;
		if len = 1 and InputBuffer[0] = SYN then
		    Delay(5);
		    checkDoIO(serReq, SDCMD_QUERY, "term2");
		    if serReq*.ios_IOSer.io_Actual = 1 then
			setupRead(&InputBuffer[0], 1);
			ignore DoIO(&serReq*.ios_IOSer.io_io);
			if InputBuffer[0] = SYN then
			    clearTerminalMode();
			    putString("\nREMOTE END DETECTED\n");
			    connectToServer();
			    return(true);
			fi;
			forceChars(pretend(&InputBuffer[0], *char), 1);
		    fi;
		elif len ~= 0 then
		    forceChars(pretend(&InputBuffer[0], *char), len);
		fi;
	    fi;
	fi;
	if bits & SIGBREAKF_CTRL_C ~= 0 then
	    if ReadSerBusy then
		abortRead();
	    fi;
	    return(false);
	fi;
	if bits & SIGBREAKF_CTRL_F ~= 0 then
	    screenToFront();
	fi;
    od;
    false
corp;

/*
 * serialReconfig - called when user fiddles with settings via menus
 */

proc serialReconfig()void:
    register *IOExtSer_t req;

    if ReadSerBusy then
	abortRead();
	ReadSerBusy := false;
    fi;
    req := ReadSerReq;
    checkDoIO(req, CMD_CLEAR, "reconfig first clear");
    doSetParams(req);
    checkDoIO(req, CMD_CLEAR, "reconfig second clear");
corp;

/*
 * serialSendBreak - send out a BREAK condition.
 */

proc serialSendBreak()void:

    checkDoIO(WriteSerReq, SDCMD_BREAK, "break");
corp;

/*
 * serialForceConnect - user wants to force a protocol startup
 */

proc serialForceConnect()void:

    WantForceConnect := true;
corp;
