#drinc:exec/memory.g
#drinc:exec/nodes.g
#drinc:exec/ports.g
#drinc:exec/io.g
#drinc:graphics/gfxbase.g
#drinc:devices/audio.g
#drinc:devices/narrator.g
#drinc:util.g

#iff.g
#iff8SVX.g
#iffSMUS.g
#sound.g
#funcs.g
#soundFuncs.g
#globals.g

/*
 * Amiga MUD
 *
 * Copyright (c) 1997 by Chris Gray
 */

/*
 * sound.d - code for handling sound (voice, audio)
 */

ulong MAX_AUDIO = 1 << 17;

type
    NarrationList_t = struct {
	*NarrationList_t nl_next;
	*narrator_rb_t nl_request;
	ulong nl_id;
	*char nl_data;
	uint nl_length;
    },

    SoundList_t = struct {
	*SoundList_t sl_next;
	*Sound_t sl_sound;
	*IOAudio_t sl_request1, sl_request2;	/* 1 started before 2 */
	arbptr sl_fc;
	ulong sl_id;
	*short sl_dataNext;
	ulong sl_lengthLeft;
	uint sl_cycles;
	bool sl_2Active;
    };

*MsgPort_t SoundPort;
ulong SoundBit;
*NarrationList_t ActiveNarrations, FreeNarrations;
*SoundList_t ActiveSounds, FreeSounds;
ulong Clock;

uint VoiceVolume, SoundVolume, MusicVolume, VRate, VPitch;
byte VMode, VSex, VVolume;
bool NarratorOpen, SoundOpen;

uint AUDIO_ALLOC_COUNT = 4;
[AUDIO_ALLOC_COUNT] byte AudioAllocMasks := (1, 2, 4, 8);

proc resetVoice()void:

    VoiceVolume := 10000;
    VRate := DEFRATE;
    VPitch := DEFPITCH;
    VMode := DEFMODE;
    VSex := DEFSEX;
    VVolume := DEFVOL;
corp;

proc soundDefaults()void:

    SoundVolume := 10000;
    MusicVolume := 10000;
    resetVoice();
corp;

proc soundInit()bool:

    Clock :=
	if GfxBase*.gb_DisplayFlags & PAL ~= 0 then 3546895 else 3579545 fi;
    soundDefaults();
    ActiveNarrations := nil;
    FreeNarrations := nil;
    NarratorOpen := false;
    ActiveSounds := nil;
    FreeSounds := nil;
    SoundOpen := false;
    SoundPort := CreatePort(nil, 0);
    if SoundPort = nil then
	return(false);
    fi;
    SoundBit := 1 << SoundPort*.mp_SigBit;
    addEffectsBit(SoundBit);
    true
corp;

proc abortSoundPlay(register *SoundList_t sl)void:
    register *IOAudio_t ioa;

    ioa := sl*.sl_request1;
    if CheckIO(&ioa*.ioa_Request) = nil then
	ignore AbortIO(&ioa*.ioa_Request);
    fi;
    ignore WaitIO(&ioa*.ioa_Request);
    if sl*.sl_2Active then
	ioa := sl*.sl_request2;
	if CheckIO(&ioa*.ioa_Request) = nil then
	    ignore AbortIO(&ioa*.ioa_Request);
	fi;
	ignore WaitIO(&ioa*.ioa_Request);
    fi;
    ioa*.ioa_Request.io_Command := ADCMD_FREE;
    ioa*.ioa_Request.io_Flags := 0;
    ioa*.ioa_Length := 0;
    BeginIO(&ioa*.ioa_Request);
    ignore WaitIO(&ioa*.ioa_Request);
    unlockEffectFile(sl*.sl_fc);
    sl*.sl_next := FreeSounds;
    FreeSounds := sl;
corp;

proc abortNarration(register *NarrationList_t nl)void:
    register *narrator_rb_t nw;

    nw := nl*.nl_request;
    if CheckIO(&nw*.nw_message.io_io) = nil then
	ignore AbortIO(&nw*.nw_message.io_io);
    fi;
    ignore WaitIO(&nw*.nw_message.io_io);
    FreeMem(nl*.nl_data, nl*.nl_length);
    nl*.nl_next := FreeNarrations;
    FreeNarrations := nl;
corp;

proc soundTerm(bool partial)void:
    register *SoundList_t sl;
    register *NarrationList_t nl @ sl;

    while
	sl := ActiveSounds;
	sl ~= nil
    do
	ActiveSounds := sl*.sl_next;
	abortSoundPlay(sl);
    od;

    while
	nl := ActiveNarrations;
	nl ~= nil
    do
	ActiveNarrations := nl*.nl_next;
	abortNarration(nl);
    od;

    if partial then
	soundDefaults();
    else
	if SoundOpen then
	    CloseDevice(&FreeSounds*.sl_request1*.ioa_Request);
	fi;
	while
	    sl := FreeSounds;
	    sl ~= nil
	do
	    FreeSounds := sl*.sl_next;
	    DeleteExtIO(&sl*.sl_request1*.ioa_Request, sizeof(IOAudio_t));
	    DeleteExtIO(&sl*.sl_request2*.ioa_Request, sizeof(IOAudio_t));
	    FreeMem(sl, sizeof(SoundList_t));
	od;

	if NarratorOpen then
	    CloseDevice(&FreeNarrations*.nl_request*.nw_message.io_io);
	fi;
	while
	    nl := FreeNarrations;
	    nl ~= nil
	do
	    FreeNarrations := nl*.nl_next;
	    DeleteExtIO(&nl*.nl_request*.nw_message.io_io,
			sizeof(narrator_rb_t));
	    FreeMem(nl, sizeof(NarrationList_t));
	od;

	DeletePort(SoundPort);
    fi;
corp;

proc setVoiceParams(uint rate, pitch; ushort mode, sex, volume)void:

    VRate := rate;
    VPitch := pitch;
    VMode := mode;
    VSex := sex;
    VVolume := (make(volume, ulong) * VoiceVolume + 5000) / 10000;
corp;

proc setVoiceVolume(uint w)void:

    VoiceVolume := w;
    VVolume := (DEFVOL * make(VoiceVolume, ulong) + 5000) / 10000;
corp;

/*
 * doNarrate - go ahead and narrate something.
 */

proc doNarrate(*char cp; uint len; ulong id)void:
    register *NarrationList_t nl;
    register *narrator_rb_t nw;

    nl := FreeNarrations;
    if nl = nil then
	nl := mudAlloc(sizeof(NarrationList_t), 0x0);
	if nl = nil then
	    noMemory("speech record");
	    return;
	fi;
	nw := pretend(CreateExtIO(SoundPort, sizeof(narrator_rb_t)),
		      *narrator_rb_t);
	if nw = nil then
	    FreeMem(nl, sizeof(NarrationList_t));
	    noMemory("speech request");
	    return;
	fi;
	nl*.nl_request := nw;
	if NarratorOpen then
	    nw* := ActiveNarrations*.nl_request*;
	fi;
    else
	FreeNarrations := nl*.nl_next;
	nw := nl*.nl_request;
    fi;

    if not NarratorOpen then
	nw*.nw_ch_masks := &AudioAllocMasks[0];
	nw*.nw_nm_masks := AUDIO_ALLOC_COUNT;
	nw*.nw_sampfreq := DEFFREQ;
	nw*.nw_mouths := 0;
	if OpenDevice("narrator.device", 0, &nw*.nw_message.io_io, 0) ~= 0 then
	    DeleteExtIO(&nw*.nw_message.io_io, sizeof(narrator_rb_t));
	    FreeMem(nl, sizeof(NarrationList_t));
	    putString("::Can't open narrator.device for speech!::\n");
	    voiceOff();
	    return;
	fi;
	NarratorOpen := true;
    fi;

    nl*.nl_id := id;
    nl*.nl_data := mudAlloc(len, MEMF_PUBLIC);
    if nl*.nl_data = nil then
	nl*.nl_next := FreeNarrations;
	FreeNarrations := nl;
	noMemory("speech buffer");
	return;
    fi;
    nl*.nl_length := len;
    BlockCopy(nl*.nl_data, cp, len);

    nl*.nl_next := ActiveNarrations;
    ActiveNarrations := nl;

    nw*.nw_message.io_Data := nl*.nl_data;
    nw*.nw_message.io_Length := len;
    nw*.nw_message.io_io.io_Command := CMD_WRITE;
    nw*.nw_rate := VRate;
    nw*.nw_pitch := VPitch;
    nw*.nw_mode := VMode;
    nw*.nw_sex := VSex;
    nw*.nw_volume := VVolume;
    SendIO(&nw*.nw_message.io_io);
corp;

proc setSoundVolume(uint volume)void:

    SoundVolume := volume;
corp;

proc playSound1(arbptr p, fc; ulong id; uint cycles)bool:
    *Sound_t sn @ p;
    register *SoundList_t sl;
    register *IOAudio_t ioa1, ioa2;
    *short data;
    register ulong totalLength, thisLength;

    sl := FreeSounds;
    if sl = nil then
	sl := mudAlloc(sizeof(SoundList_t), 0x0);
	if sl = nil then
	    noMemory("sound record");
	    return(false);
	fi;
	ioa1 := pretend(CreateExtIO(SoundPort, sizeof(IOAudio_t)), *IOAudio_t);
	if ioa1 = nil then
	    FreeMem(sl, sizeof(SoundList_t));
	    noMemory("audio request");
	    return(false);
	fi;
	sl*.sl_request1 := ioa1;
	ioa2 := pretend(CreateExtIO(SoundPort, sizeof(IOAudio_t)), *IOAudio_t);
	if ioa2 = nil then
	    DeleteExtIO(&ioa1*.ioa_Request, sizeof(IOAudio_t));
	    FreeMem(sl, sizeof(SoundList_t));
	    noMemory("audio request");
	    return(false);
	fi;
	sl*.sl_request2 := ioa2;
	if SoundOpen then
	    /* SoundOpen, but nothing on FreeSounds, so must be at least
	       one sound actively playing. Doing the copy this way gets
	       us all the OpenDevice info as well. */
	    ioa1* := ActiveSounds*.sl_request1*;
	    ioa2* := ioa1*;
	fi;
    else
	FreeSounds := sl*.sl_next;
	ioa1 := sl*.sl_request1;
	ioa2 := sl*.sl_request2;
    fi;

    if not SoundOpen then
	ioa1*.ioa_Data := nil;
	ioa1*.ioa_Length := 0;
	if OpenDevice(AUDIONAME, 0, &ioa1*.ioa_Request, 0) ~= 0 then
	    DeleteExtIO(&ioa1*.ioa_Request, sizeof(IOAudio_t));
	    DeleteExtIO(&ioa2*.ioa_Request, sizeof(IOAudio_t));
	    FreeMem(sl, sizeof(SoundList_t));
	    putString("::can't open audio device!::\n");
	    return(false);
	fi;
	/* Data copied into second request below. */
	SoundOpen := true;
    fi;

    sl*.sl_sound := sn;
    sl*.sl_fc := fc;
    sl*.sl_id := id;
    sl*.sl_2Active := false;

    ioa1*.ioa_Data := pretend(&AudioAllocMasks[0], *short);
    ioa1*.ioa_Length := AUDIO_ALLOC_COUNT;
    ioa1*.ioa_AllocKey := 0;
    ioa1*.ioa_Request.io_Command := ADCMD_ALLOCATE;
    ioa1*.ioa_Request.io_Flags := ADIOF_NOWAIT;
    ioa1*.ioa_Request.io_Message.mn_Node.ln_Pri := -50;
    BeginIO(&ioa1*.ioa_Request);
    if WaitIO(&ioa1*.ioa_Request) ~= 0 then
	/* No sound channel available. */
	sl*.sl_next := FreeSounds;
	FreeSounds := sl;
	return(false);
    fi;

    ioa1*.ioa_Request.io_Command := CMD_WRITE;
    ioa1*.ioa_Request.io_Flags := ADIOF_PERVOL;
    ioa1*.ioa_Period := Clock / sn*.sn_v8h.v8h_samplesPerSecond;
    ioa1*.ioa_Volume :=
	((sn*.sn_v8h.v8h_volume >> 10) * SoundVolume + 5000) / 10000;
    ioa2* := ioa1*;
    totalLength := sn*.sn_sampleLength;
    data := sn*.sn_data;
    totalLength := totalLength & (-2);
    thisLength :=
	if totalLength > MAX_AUDIO then MAX_AUDIO else totalLength fi;
    ioa1*.ioa_Data := data;
    ioa1*.ioa_Length := thisLength;
    totalLength := totalLength - thisLength;
    data := data + thisLength;
    if totalLength ~= 0 then
	/* Cycling will be done by us, not by audio.device */
	sl*.sl_cycles := cycles;
	ioa1*.ioa_Cycles := 1;
	BeginIO(&ioa1*.ioa_Request);
	sl*.sl_2Active := true;
	/* Most setup of this 'ioa2' is done by the struct copy, above. */
	ioa2*.ioa_Cycles := 1;
	thisLength :=
	    if totalLength > MAX_AUDIO then MAX_AUDIO else totalLength fi;
	ioa2*.ioa_Data := data;
	ioa2*.ioa_Length := thisLength;
	BeginIO(&ioa2*.ioa_Request);
	totalLength := totalLength - thisLength;
	data := data + thisLength;
    else
	/* Cycling will be done by audio.device */
	sl*.sl_cycles := 1;
	ioa1*.ioa_Cycles := cycles;
	BeginIO(&ioa1*.ioa_Request);
    fi;
    sl*.sl_dataNext := data;
    sl*.sl_lengthLeft := totalLength;
    sl*.sl_next := ActiveSounds;
    ActiveSounds := sl;
    true
corp;

proc playSound(arbptr p, fc; ulong id; uint cycles)void:

    lockEffectFile(fc);
    if not playSound1(p, fc, id, cycles) then
	unlockEffectFile(fc);
    fi;
corp;

proc setMusicVolume(uint volume)void:

    MusicVolume := volume;
corp;

proc playMusic(register arbptr p; arbptr fc; ulong id)void:
    register *Music_t mu @ p;
    register *Instrument_t ins;
    register uint i;

    ins := &mu*.mu_instruments[0];
    i := 0;
    while i ~= mu*.mu_instrumentCount do
	putString("Using instrument '");
	putString(ins*.ins_name);
	putString("'\n");
	ins := ins + sizeof(Instrument_t);
	i := i + 1;
    od;
    putString("Imagine the music playing...\n");
corp;

proc handleSoundSignal(ulong bits)void:
    register *Message_t m;
    register **SoundList_t psl;
    register *SoundList_t sl;
    register *IOAudio_t ioa @ m;
    register **NarrationList_t pnl @ psl;
    register *NarrationList_t nl @ sl;
    register ulong thisLength;
    bool soundCompleted, lostChannel;

    while
	m := GetMsg(SoundPort);
	m ~= nil
    do
	/* ioa := pretend(m, IOAudio_t); */
	psl := &ActiveSounds;
	while
	    sl := psl*;
	    sl ~= nil and sl*.sl_request1 ~= ioa and sl*.sl_request2 ~= ioa
	do
	    psl := &sl*.sl_next;
	od;
	if sl ~= nil then
	    soundCompleted := false;
	    lostChannel := false;
	    if sl*.sl_request1 = ioa then
		if ioa*.ioa_Request.io_Error ~= 0 then
		    /* Most likely this is IOERR_ALLOCFAILED, but we don't
		       really care - we shut down the sound in any case. */
		    if sl*.sl_2Active then
			sl*.sl_2Active := false;
			sl*.sl_request1 := sl*.sl_request2;
			sl*.sl_request2 := ioa;
			ioa := sl*.sl_request1;
			if CheckIO(&ioa*.ioa_Request) = nil then
			    ignore AbortIO(&ioa*.ioa_Request);
			fi;
			/* No WaitIO - we are already inside a GetMsg loop */
		    else
			soundCompleted := true;
			lostChannel := true;
		    fi;
		else
		    if sl*.sl_2Active then
			sl*.sl_request1 := sl*.sl_request2;
			sl*.sl_request2 := ioa;
			thisLength := sl*.sl_lengthLeft;
			if thisLength ~= 0 then
			    if thisLength > MAX_AUDIO then
				thisLength := MAX_AUDIO;
			    fi;
			    ioa*.ioa_Request.io_Command := CMD_WRITE;
			    ioa*.ioa_Request.io_Flags := 0;
			    ioa*.ioa_Cycles := 1;
			    ioa*.ioa_Data := sl*.sl_dataNext;
			    ioa*.ioa_Length := thisLength;
			    BeginIO(&ioa*.ioa_Request);
			    sl*.sl_lengthLeft :=
				sl*.sl_lengthLeft - thisLength;
			    sl*.sl_dataNext := sl*.sl_dataNext + thisLength;
			else
			    if sl*.sl_cycles ~= 1 then
				/* Need to start another cycle */
				ioa*.ioa_Request.io_Command := CMD_WRITE;
				ioa*.ioa_Request.io_Flags := 0;
				ioa*.ioa_Cycles := 1;
				ioa*.ioa_Data := sl*.sl_sound*.sn_data;
				ioa*.ioa_Length := MAX_AUDIO;
				BeginIO(&ioa*.ioa_Request);
				if sl*.sl_cycles ~= 0 then
				    /* 0 means repeat forever */
				    sl*.sl_cycles := sl*.sl_cycles - 1;
				fi;
				sl*.sl_lengthLeft :=
				    sl*.sl_sound*.sn_sampleLength - MAX_AUDIO;
				sl*.sl_dataNext :=
				    sl*.sl_sound*.sn_data + MAX_AUDIO;
			    else
				sl*.sl_2Active := false;
			    fi;
			fi;
		    else
			soundCompleted := true;
		    fi;
		fi;
	    else
		/* Just ignore this - if our 2nd request has come back before
		   our first, then we must have lost our channel, and we can
		   expect our first request back right away. */
		sl*.sl_2Active := false;
	    fi;
	    if soundCompleted then
		eDone(0, sl*.sl_id);
		if not lostChannel then
		    ioa*.ioa_Request.io_Command := ADCMD_FREE;
		    ioa*.ioa_Request.io_Flags := 0;
		    ioa*.ioa_Length := 0;
		    BeginIO(&ioa*.ioa_Request);
		    ignore WaitIO(&ioa*.ioa_Request);
		fi;
		unlockEffectFile(sl*.sl_fc);
		psl* := sl*.sl_next;
		sl*.sl_next := FreeSounds;
		FreeSounds := sl;
	    fi;
	else
	    pnl := &ActiveNarrations;
	    while
		nl := pnl*;
		nl ~= nil and nl*.nl_request ~= pretend(m, *narrator_rb_t)
	    do
		pnl := &nl*.nl_next;
	    od;
	    if nl ~= nil then
		eDone(1, nl*.nl_id);
		pnl* := nl*.nl_next;
		FreeMem(nl*.nl_data, nl*.nl_length);
		nl*.nl_next := FreeNarrations;
		FreeNarrations := nl;
	    else
		putString("\n*** Unknown sound request returned! ***\n\n");
	    fi;
	fi;
    od;
corp;

proc abortSound(byte kind; register ulong id)void:
    register **NarrationList_t pnl;
    register **SoundList_t psl @ pnl;
    register *NarrationList_t nl;
    register *SoundList_t sl @ nl;

    if id ~= 0 then
	case kind
	incase 0:   /* normal sound */
	    psl := &ActiveSounds;
	    while
		sl := psl*;
		sl ~= nil and sl*.sl_id ~= id
	    do
		psl := &sl*.sl_next;
	    od;
	    if sl ~= nil then
		psl* := sl*.sl_next;
		abortSoundPlay(sl);
	    fi;
	incase 1:   /* narration */
	    pnl := &ActiveNarrations;
	    while
		nl := pnl*;
		nl ~= nil and nl*.nl_id ~= id
	    do
		pnl := &nl*.nl_next;
	    od;
	    if nl ~= nil then
		pnl* := nl*.nl_next;
		abortNarration(nl);
	    fi;
	incase 2:   /* music */
	    ;
	esac;
    fi;
corp;
