#drinc:exec/memory.g
#drinc:graphics/gfx.g
#drinc:graphics/rastport.g
#drinc:intuition/window.g
#drinc:intuition/image.g
#drinc:intuition/gadget.g
#drinc:devices/inputevent.g
#drinc:libraries/dos.g
#drinc:util.g

#text.g
#graphics.g
#globals.g
#funcs.g

/*
 * Amiga MUD
 *
 * Copyright (c) 1997 by Chris Gray
 */

/*
 * text.d - handle text input/output to the text window.
 */

/* forward declaration needed */

extern tx_putChar1(char ch)void;

Gadget_t ActualScrollGadget;

[MAX_TEXT_COLUMNS] char TextBuff;	/* the current output text line */
uint
    TextColumn, 			/* current text column */
    TextLinePos,			/* next pos in TextBuff */
    TextWordPos,			/* pos in TextBuff of "word" */
    TextHeight, 			/* TextXXXXHeight */
    TextOutputRow,			/* bottom of window in current mode */
    TextInputBottom,			/* bottom of rect for input line */
    TextInputTop,			/* top of rect for input line */
    TextLines,				/* TextXXXXLines */
    OutputLines,			/* TextLines - 1 (the input line) */
    PropFlags;				/* cache a copy */
HistoryType_t HistoryType;
ScrollType_t ScrollType;
bool
    AtNewLine,				/* last char was a newline */
    LastWasOutput,			/* last action was output char */
    EchoOn,				/* display typed characters */
    SendingLine,			/* currently inside 'sendLine' */
    WantSendLine,			/* another line ready to go */
    BrightBold; 			/* make it stand out */

[MAX_TEXT_COLUMNS + 1] char CookedBuffer;/* buffer of edited line input */
[MAX_TEXT_COLUMNS] char PromptBuffer;	/* the current prompt */
[MAX_TEXT_COLUMNS] char UndoBuffer;	/* for the ^Y undo */
uint
    CookedCount,			/* total chars in the input line */
    CookedPos,				/* current cursor pos in input */
    PromptLen;				/* length of the current prompt */

*char OutputHistory;
ulong OutputHistorySize, InputHistorySize;
ulong OutputHistoryPos;
uint OutputHistoryLines;
int WindowTopLine;	/* negative when not enough lines in history */
*char InputHistory;
ulong InputHistoryPos;
*char HistoryPoint;
bool LastWasHistory, LastWasNext;

PropInfo_t ScrollInfo;
Image_t ScrollImage := (
    0, 0, 0, 0, 0, nil, 0, 0, nil
);
ulong ScrollPot, ScrollBody;		/* so can avoid extra redraws */
bool ScrollPresent;

Handle_t LogFd;
bool LogInputOnly;

enum {
    es_none,
    es_gotEsc,
    es_gotLSq
} EscapeState;
bool LastWasNewline;

/*
 * tx_setValues - setup values based on current visibility.
 */

proc tx_setValues()void:

    if PictureShown then
	TextHeight := TextHalfHeight;
	TextOutputRow := TextHalfOutputRow;
	TextInputBottom := TextHalfInputBottom;
	TextInputTop := TextHalfInputTop;
	TextLines := TextHalfLines;
    else
	TextHeight := TextFullHeight;
	TextOutputRow := TextFullOutputRow;
	TextInputBottom := TextFullInputBottom;
	TextInputTop := TextFullInputTop;
	TextLines := TextFullLines;
    fi;
    OutputLines := TextLines - 1;
corp;

/*
 * tx_setScrollStuff - set the height of the scroll gadget and its position.
 */

proc tx_setScrollStuff(bool amEditing)void:
    register *Gadget_t g;

    g := ScrollGadget;
    if OldStyle then
	g*.g_LeftEdge := TextWidth - ScrollWidth;
	g*.g_Height :=
	    if amEditing then
		TextFullHeight - TopBorder + 5
	    else
		TextHeight - TopBorder + 5
	    fi;
    else
	g*.g_LeftEdge := - ScrollWidth;
	g*.g_Height :=
	    if Mode = md_oneWindow then
		- (TopBorder + ScreenDelta + SIZE_HEIGHT)
	    elif Mode = md_twoWindow then
		- (TopBorder + SIZE_HEIGHT)
	    else
		- TopBorder
	    fi;
    fi;
corp;

/*
 * tx_scrollInit - initialize the scrollbar structures.
 */

proc tx_scrollInit()void:
    register *PropInfo_t pi;
    register *Gadget_t g @ pi;

    PropFlags :=
	if OldStyle then
	    AUTOKNOB | FREEVERT
	else
	    AUTOKNOB | FREEVERT | PROPNEWLOOK
	fi;
    pi := &ScrollInfo;
    pi*.pi_Flags := PropFlags;
    pi*.pi_HorizPot := 0;
    pi*.pi_VertPot := 0;
    pi*.pi_HorizBody := 0;
    pi*.pi_VertBody := 0;
    pi*.pi_CWidth := 0;
    pi*.pi_CHeight := 0;
    pi*.pi_HPotRes := 0;
    pi*.pi_VPotRes := 0;
    pi*.pi_LeftBorder := 0;
    pi*.pi_TopBorder := 0;
    g := ScrollGadget;
    g*.g_NextGadget := nil;
    g*.g_TopEdge := if OldStyle then TextTop - 5 else TextTop fi;
    g*.g_Width := ScrollWidth;
    tx_setScrollStuff(false);
    g*.g_Flags :=
	if OldStyle then
	    GADGHCOMP | GADGIMAGE
	else
	    GADGHCOMP | GADGIMAGE | GRELRIGHT | GRELHEIGHT
	fi;
    g*.g_Activation := FOLLOWMOUSE | GADGIMMEDIATE | RELVERIFY | RIGHTBORDER;
    g*.g_GadgetType := PROPGADGET;
    g*.g_GadgetRender.gImage := &ScrollImage;	/* *must* be provided */
    g*.g_SelectRender.gImage := nil;
    g*.g_GadgetText := nil;
    g*.g_MutualExclude := 0x0;
    g*.g_SpecialInfo.gProp := &ScrollInfo;
    g*.g_GadgetID := 1;
corp;

/*
 * tx_outputMove - move cursor to given text row/column.
 */

proc tx_outputMove(uint column, row)void:

    Move(TextRastPort, column * CHAR_WIDTH + LeftBorder,
	 row * CHAR_HEIGHT + TextTop + CHAR_BASELINE);
corp;

/*
 * tx_inputMove - move cursor to the given column in the input line.
 */

proc tx_inputMove(uint column)void:

    Move(TextRastPort, column * CHAR_WIDTH + LeftBorder,
	 TextOutputRow + CHAR_HEIGHT);
corp;

/*
 * tx_toggleCursor - toggle a cursor at the current position.
 */

proc tx_toggleCursor()void:
    register *RastPort_t rp;

    if not Editing then
	rp := TextRastPort;
	tx_inputMove(PromptLen + CookedPos);
	SetDrMd(rp, COMPLEMENT);
	RectFill(rp,
		 rp*.rp_cp_x, rp*.rp_cp_y - CHAR_BASELINE,
		 rp*.rp_cp_x + CHAR_WIDTH - 1,
		 rp*.rp_cp_y + (CHAR_HEIGHT - CHAR_BASELINE - 1)
	);
	if TerminalMode then
	    SetDrMd(rp, JAM2);
	else
	    SetDrMd(rp, JAM1);
	fi;
	LastWasOutput := false;
    fi;
corp;

/*
 * tx_repaint - repaint the given slot of the text window.
 *	NOTE: when we break long lines/words up, we insert newlines into the
 *	output history buffer, so the "lines" in that buffer are always
 *	short enough to fit on an actual screen line, when they are added.
 *	If the window is shrunk, then we just clip them.
 */

proc tx_repaint(uint windowTop, windowBottom)void:
    register *RastPort_t rp;
    register *char p, q;
    register uint rectTop, lineLen @ rectTop;

    if not Editing then
	rp := TextRastPort;

	/* first, clear the rectangle we need to redraw */

	rectTop := TextTop + windowTop * CHAR_HEIGHT;
	SetAPen(rp, ErasePen);
	RectFill(rp, LeftBorder, rectTop, RightEdge,
		 rectTop + (windowBottom - windowTop + 1) * CHAR_HEIGHT - 1);
	SetAPen(rp, TextPen);

	if windowTop + WindowTopLine < 0 then
	    windowTop := - WindowTopLine;
	fi;
	if windowBottom + WindowTopLine >= 0 then
	    /* scan backwards in the history to find the needed lines */

	    q := OutputHistory;
	    p := q + OutputHistoryPos;
	    p := p - sizeof(char);	    /* point at the last newline */
	    lineLen := OutputHistoryLines - WindowTopLine - 1;
	    while
		while
		    p := p - sizeof(char);
		    p* ~= '\n' and p ~= q
		do
		od;
		lineLen ~= windowTop and p ~= q
	    do
		lineLen := lineLen - 1;
	    od;
	    if p* = '\n' then
		p := p + sizeof(char);		/* point past the newline */
	    else
		windowTop := lineLen;
	    fi;

	    /* now, paint the text in the rectangle */

	    if windowTop <= windowBottom then
		while
		    lineLen := 0;
		    q := p;
		    while p* ~= '\n' do
			lineLen := lineLen + 1;
			p := p + sizeof(char);
		    od;
		    tx_outputMove(0, windowTop);
		    if lineLen > TextColumns then
			lineLen := TextColumns;
		    fi;
		    Text(rp, q, lineLen);
		    p := p + sizeof(char);	/* skip the newline */
		    windowTop ~= windowBottom
		do
		    windowTop := windowTop + 1;
		od;
	    fi;
	fi;
	LastWasOutput := true;
    fi;
corp;

/*
 * tx_redrawScroll - redraw the scrollbar on the right.
 */

proc tx_redrawScroll()void:
    ulong newPot, newBody;

    if OutputHistoryLines <= OutputLines then
	newPot := 0;
	newBody := 0xffff;
    else
	newPot := make(WindowTopLine, long) * 0xffff /
	    (OutputHistoryLines - OutputLines);
	newBody := make(OutputLines, ulong) * 0xffff / OutputHistoryLines;
    fi;
    if newPot ~= ScrollPot or newBody ~= ScrollBody then
	NewModifyProp(ScrollGadget, TextWindow, nil, PropFlags,
		      0, newPot, 0, newBody, 1);
	ScrollPot := newPot;
	ScrollBody := newBody;
    fi;
corp;

/*
 * tx_doAnyScroll - handle whatever we need to do.
 */

proc tx_doAnyScroll(int dx, dy; uint xmin, ymin, xmax, ymax)void:

    ScrollRaster(TextRastPort, dx, dy, xmin, ymin, xmax, ymax);
    /* Gack! */
    if Mode = md_twoWindow then
	WantRefresh := true;
    fi;
corp;

/*
 * tx_doScrollWindow - scroll the visible text window by the indicated
 *	number of text lines. Negative scrolls forward, positive backwards.
 */

proc tx_doScrollWindow(int lines)void:

    tx_doAnyScroll(0, lines * CHAR_HEIGHT,
		   LeftBorder, TextTop, RightEdge, TextInputTop - 1);
corp;

/*
 * tx_outputHistory - add a character to the output history buffer.
 */

proc tx_outputHistory(char ch)void:
    register *char p, q;
    register ulong i;
    bool wantScroll;

    wantScroll := false;
    i := OutputHistoryPos;
    if i = OutputHistorySize then
	p := OutputHistory;
	q := p;
	while i ~= 0 and p* ~= '\n' do
	    i := i - sizeof(char);
	    p := p + sizeof(char);
	od;
	if i ~= 0 then
	    i := i - sizeof(char);
	    p := p + sizeof(char);
	fi;
	OutputHistoryPos := i;
	while i ~= 0 do
	    i := i - sizeof(char);
	    q* := p*;
	    q := q + sizeof(char);
	    p := p + sizeof(char);
	od;
	OutputHistoryLines := OutputHistoryLines - 1;
	if WindowTopLine = 0 then
	    if not Editing then
		tx_doScrollWindow(1);
		tx_repaint(OutputLines - 1, OutputLines - 1);
	    fi;
	else
	    WindowTopLine := WindowTopLine - 1;
	fi;
	i := OutputHistoryPos;
	wantScroll := true;
    fi;
    (OutputHistory + i)* := ch;
    i := i + sizeof(char);
    OutputHistoryPos := i;
    if ch = '\n' then
	if WindowTopLine = OutputHistoryLines - OutputLines then
	    WindowTopLine := WindowTopLine + 1;
	fi;
	OutputHistoryLines := OutputHistoryLines + 1;
	wantScroll := true;
    fi;
    if wantScroll and ScrollPresent then
	tx_redrawScroll();
    fi;
corp;

/*
 * tx_historySetup - start the history buffers in a useable state.
 */

proc tx_historySetup()void:

    /* Output history done in tx_init0, so that scrollbar can be done. */
    InputHistory* := ' ';
    HistoryPoint := InputHistory + sizeof(char);
    HistoryPoint* := '\e';
    InputHistoryPos := 2 * sizeof(char);
    LastWasHistory := false;
    LastWasNext := false;
corp;

/*
 * tx_drawBorders - draw the pretty borders we use. Used here on init and in
 *	the main intuition code on activate/inactive. Note that this code
 *	is only used when NewLook is false, so we can use the old-style
 *	constant pen values.
 */

proc tx_drawBorders(uint height)void:
    register *RastPort_t rp;
    register uint i, j, i2;
    [4] uint PEN_MAP = (CST_TEXT_PEN, LIGHT_PEN, DARK_PEN, ERASE_PEN);

    if height = 0 then
	if Editing then
	    height := TextFullHeight;
	else
	    height := TextHeight;
	fi;
    fi;
    rp := TextRastPort;
    j := height - 1;
    for i from 0 upto 3 do
	i2 := i * 2;
	SetAPen(rp, PEN_MAP[i]);
	Move(rp, i2, i + TITLE_BAR);
	Draw(rp, TextWidth - OLD_SCROLL_WIDTH - 1 - i2, i + TITLE_BAR);
	Draw(rp, TextWidth - OLD_SCROLL_WIDTH - 1 - i2, j);
	Move(rp, TextWidth - OLD_SCROLL_WIDTH - 2 - i2, i + TITLE_BAR);
	Draw(rp, TextWidth - OLD_SCROLL_WIDTH - 2 - i2, j);
	Move(rp, TextWidth - OLD_SCROLL_WIDTH - 1 - i2, j);
	Draw(rp, i2, j);
	Draw(rp, i2, i + TITLE_BAR);
	Move(rp, i2 + 1, j);
	Draw(rp, i2 + 1, i + TITLE_BAR);
	j := j - 1;
    od;
    SetAPen(rp, TextPen);
corp;

/*
 * tx_scrollReconfig - setup the scroll bar for the current window size.
 */

proc tx_scrollReconfig()void:

    tx_setScrollStuff(false);
    if OutputHistoryLines <= OutputLines then
	ScrollPot := 0;
	ScrollBody := 0xffff;
    else
	ScrollPot := make(WindowTopLine, ulong) * 0xffff /
	    (OutputHistoryLines - OutputLines);
	ScrollBody := make(OutputLines, ulong) * 0xffff / OutputHistoryLines;
    fi;
    ScrollInfo.pi_VertPot := ScrollPot;
    ScrollInfo.pi_VertBody := ScrollBody;
corp;

/*
 * tx_reconfigure - set up the text stuff for the given mode.
 */

proc tx_reconfigure()void:

    if ScrollPresent then
	ignore RemoveGadget(TextWindow, ScrollGadget);
    fi;
    tx_scrollReconfig();
    if OldStyle then
	/* draw the complete border */
	tx_drawBorders(0);
    fi;
    ignore AddGadget(TextWindow, ScrollGadget, 0);
    RefreshGadgets(ScrollGadget, TextWindow, nil);
    ScrollPresent := true;
corp;

/*
 * tx_resetVars - reset variables associated with text. Called on init, and
 *	from Intuition.d on {set|clear}TerminalMode.
 */

proc tx_resetVars()void:

    TextColumn := 0;
    TextLinePos := 0;
    TextWordPos := 0;
    AtNewLine := false;
    EchoOn := true;
    SendingLine := false;
    WantSendLine := false;
    CookedCount := 0;
    CookedPos := 0;
    PromptLen := 0;
    UndoBuffer[0] := '\e';
    EscapeState := es_none;
    LastWasNewline := false;
corp;

proc tx_setHistoryType(HistoryType_t historyType)void:

    HistoryType := historyType;
corp;

proc tx_setScrollType(ScrollType_t scrollType)void:

    ScrollType := scrollType;
corp;

proc tx_setHistorySizes(ulong outputHistorySize, inputHistorySize)void:

    OutputHistorySize := outputHistorySize;
    InputHistorySize := inputHistorySize;
corp;

/*
 * tx_init0 - initialize text processing stuff. This routine is
 *	called early, before the text window is open. It should set up
 *	everything needed for the scroll gadget, so that that gadget
 *	can be set into the text window when it is opened.
 */

proc tx_init0()void:

    tx_setValues();
    tx_resetVars();
    ScrollGadget := &ActualScrollGadget;
    tx_scrollInit();
    OutputHistoryPos := 0;
    OutputHistoryLines := 0;
    WindowTopLine := - OutputLines;
    tx_scrollReconfig();
    ScrollPresent := true;
corp;

/*
 * tx_clipCooked - clip the input line, in case window was resized.
 */

proc tx_clipCooked()void:
    register uint limit;

    limit := TextColumns - PromptLen - 1;
    if CookedCount > limit then
	CookedCount := limit
    fi;
    if CookedPos > limit then
	CookedPos := limit;
    fi;
corp;

/*
 * tx_reInit - any reset needed on a user window resizing.
 */

proc tx_reInit()void:
    uint oldOutputLines;

    oldOutputLines := OutputLines;
    tx_setValues();
    tx_clipCooked();
    WindowTopLine := WindowTopLine + oldOutputLines - OutputLines;
    if WindowTopLine < 0 then
	if OutputHistoryLines < OutputLines then
	    /* Not a full window, go negative, and only have partial fill. */
	    WindowTopLine := OutputHistoryLines - OutputLines;
	else
	    WindowTopLine := 0;
	fi;
    fi;
    tx_redrawScroll();
corp;

/*
 * tx_init1 - called when we can actually start rendering in the text window.
 */

proc tx_init1()bool:

    BrightBold := false;
    LogFd := 0;
    LogInputOnly := false;
    HistoryType := ht_new;
    ScrollType := st_input;
    OutputHistory := AllocMem(OutputHistorySize * sizeof(char), 0x00);
    if OutputHistory = nil then
	errorString("Cannot allocate output history buffer\n");
	return(false);
    fi;
    InputHistory := AllocMem(InputHistorySize * sizeof(char), 0x00);
    if InputHistory = nil then
	FreeMem(OutputHistory, OutputHistorySize * sizeof(char));
	errorString("Cannot allocate input history buffer\n");
	return(false);
    fi;
    tx_historySetup();
    tx_reconfigure();
    if OldStyle then
	SetAPen(TextRastPort, ERASE_PEN);
	RectFill(TextRastPort, 0, TextHalfHeight ,
	    TextWidth - 1, TextFullHeight - BottomBorder - 1);
	SetAPen(TextRastPort, CST_TEXT_PEN);
    fi;
    tx_toggleCursor();
    true
corp;

/*
 * tx_term - shut down all the stuff here. Assumes tx_init has been called.
 */

proc tx_term()void:

    FreeMem(InputHistory, InputHistorySize * sizeof(char));
    FreeMem(OutputHistory, OutputHistorySize * sizeof(char));
corp;

/*
 * tx_scrollWindow - actual work of scrolling the text part of the window.
 */

proc tx_scrollWindow(register int where)void:
    register int delta, wtl, olm1;

    wtl := WindowTopLine;
    olm1 := OutputLines - 1;
    if where < wtl and wtl - where < olm1 then
	/* do a true scroll */
	delta := wtl - where;
	WindowTopLine := where;
	tx_doScrollWindow(- delta);
	tx_repaint(0, delta - 1);
    elif where > wtl and where - wtl < olm1 then
	/* do a true scroll */
	delta := where - wtl;
	WindowTopLine := where;
	tx_doScrollWindow(delta);
	tx_repaint(olm1 + 1 - delta, olm1);
    else
	/* just redraw the whole thing */
	WindowTopLine := where;
	tx_repaint(0, olm1);
    fi;
corp;

/*
 * tx_scroll - user action requiring text history scroll.
 */

proc tx_scroll()void:
    register int where, space;

    space := OutputHistoryLines - OutputLines;
    if space >= 0 then
	where := make(ScrollInfo.pi_VertPot, ulong) * space / 0xffff;
	if where > space then
	    where := space;
	fi;
	if where ~= WindowTopLine then
	    tx_scrollWindow(where);
	fi;
    fi;
corp;

/*
 * ed_scroll - first step of doing it for editing (need ScrollInfo).
 */

proc ed_scroll()void:

    ed_scroll2(ScrollInfo.pi_VertPot);
corp;

/*
 * tx_scrollLines - scroll a given number of lines up or down.
 */

proc tx_scrollLines(bool scrollUp, scrollLines)void:
    register int lines, pos, limit;

    lines := if scrollLines then OutputLines - 1 else 1 fi;
    if scrollUp then
	/* scroll back into the history */
	if WindowTopLine > 0 then
	    pos := WindowTopLine - lines;
	    if pos < 0 then
		pos := 0;
	    fi;
	    tx_scrollWindow(pos);
	    tx_redrawScroll();
	fi;
    else
	/* scroll forward towards the present */
	limit := OutputHistoryLines - WindowTopLine - OutputLines;
	if lines > limit then
	    lines := limit;
	fi;
	if lines > 0 then
	    tx_scrollWindow(WindowTopLine + lines);
	    tx_redrawScroll();
	fi;
    fi;
corp;

/*
 * tx_scrollToBottom - scroll to the bottom of the output history.
 */

proc tx_scrollToBottom()void:
    int where;

    where := OutputHistoryLines - OutputLines;
    if where ~= WindowTopLine then
	tx_scrollWindow(where);
	/* force the redraw, since they are not up-to-date here */
	ScrollPot := 0;
	ScrollBody := 0;
	tx_redrawScroll();
    fi;
corp;

/*
 * tx_appendToInputHistory - append the given line to the input history buffer.
 */

proc tx_appendToInputHistory()void:
    register *char buffer, p;
    register ulong n;

    CookedBuffer[CookedCount] := '\e';
    buffer := &CookedBuffer[0];
    if buffer* ~= '\e' then
	/* Don't bother appending empty lines to the input history. */
	while
	    if InputHistoryPos = InputHistorySize - sizeof(char) then
		p := InputHistory + 2 * sizeof(char);
		n := sizeof(char);
		while p* ~= '\e' do
		    p := p + sizeof(char);
		    n := n + sizeof(char);
		od;
		p := p + sizeof(char);
		BlockCopy(InputHistory + 2 * sizeof(char), p,
			  InputHistorySize - n - 2 * sizeof(char));
		InputHistoryPos := InputHistoryPos - n;
	    fi;
	    (InputHistory + InputHistoryPos)* := buffer*;
	    InputHistoryPos := InputHistoryPos + sizeof(char);
	    buffer* ~= '\e'
	do
	    buffer := buffer + sizeof(char);
	od;
    fi;
corp;

/*
 * tx_refreshInputLine - put the bottom (input) line back.
 */

proc tx_refreshInputLine()void:
    register *RastPort_t rp;
    uint len;

    rp := TextRastPort;
    SetAPen(rp, ErasePen);
    RectFill(rp, LeftBorder, TextInputTop, RightEdge, TextInputBottom);
    if PromptLen ~= 0 then
	SetAPen(rp, PromptPen);
	tx_inputMove(0);
	len := PromptLen;
	if len > TextWidth - 10 then
	    len := TextWidth - 10;
	fi;
	Text(rp, &PromptBuffer[0], len);
    fi;
    SetAPen(rp, TextPen);
corp;

/*
 * tx_displayInput - display what is currently in CookedBuffer.
 */

proc tx_displayInput()void:

    tx_inputMove(PromptLen);
    tx_clipCooked();
    if EchoOn then
	Text(TextRastPort, &CookedBuffer[0], CookedCount);
    fi;
corp;

/*
 * tx_refresh - redraw the entire text window contents.
 */

proc tx_refresh()void:

    tx_repaint(0, OutputLines - 1);
    tx_refreshInputLine();
    tx_displayInput();
    tx_toggleCursor();
corp

/*
 * tx_clearInputLine - clear out the input region.
 */

proc tx_clearInputLine()void:

    tx_refreshInputLine();
    CookedCount := 0;
    CookedPos := 0;
corp;

/*
 * tx_historyPrevious - go to the previous line in the input history.
 */

proc tx_historyPrevious()void:
    register *char historyPoint, p;
    register uint cookedCount;

    tx_clearInputLine();
    if LastWasHistory then
	historyPoint := HistoryPoint;
	if historyPoint ~= InputHistory + sizeof(char) then
	    while
		historyPoint := historyPoint - sizeof(char);
		historyPoint* ~= '\e'
	    do
	    od;
	fi;
    else
	historyPoint := InputHistory + InputHistoryPos - sizeof(char);
    fi;
    if historyPoint ~= InputHistory + sizeof(char) then
	while
	    historyPoint := historyPoint - sizeof(char);
	    historyPoint* ~= '\e'
	do
	od;
	historyPoint := historyPoint + sizeof(char);
	p := &CookedBuffer[0];
	cookedCount := 0;
	while historyPoint* ~= '\e' do
	    p* := historyPoint*;
	    p := p + sizeof(char);
	    historyPoint := historyPoint + sizeof(char);
	    cookedCount := cookedCount + 1;
	od;
	CookedCount := cookedCount;
	CookedPos := cookedCount;
	if EchoOn then
	    tx_displayInput();
	fi;
    fi;
    LastWasNext := false;
    LastWasHistory := true;
    HistoryPoint := historyPoint;
corp;

/*
 * tx_historyNext - move down to the next line in the input history.
 */

proc tx_historyNext()void:
    register *char historyPoint, p;
    register uint cookedCount;

    tx_clearInputLine();
    historyPoint := HistoryPoint;
    if LastWasHistory and
	historyPoint ~= InputHistory + InputHistoryPos - sizeof(char)
    then
	historyPoint := historyPoint + sizeof(char);
	p := &CookedBuffer[0];
	cookedCount := 0;
	while historyPoint* ~= '\e' do
	    p* := historyPoint*;
	    p := p + sizeof(char);
	    historyPoint := historyPoint + sizeof(char);
	    cookedCount := cookedCount + 1;
	od;
	CookedCount := cookedCount;
	CookedPos := cookedCount;
	tx_displayInput();
	if EchoOn then
	    LastWasNext := true;
	fi;
	LastWasHistory := true;
	HistoryPoint := historyPoint;
    else
	LastWasHistory := false;
    fi;
corp;

/*
 * tx_historySearch - search backwards in the input history.
 */

proc tx_historySearch()void:
    register *char historyPoint, p;
    register uint cookedPos, n;
    register char ch1, ch2;

    cookedPos := CookedPos;
    if cookedPos ~= 0 then
	tx_clearInputLine();
	CookedPos := cookedPos;
	if LastWasHistory then
	    historyPoint := HistoryPoint;
	    if historyPoint ~= InputHistory + sizeof(char) then
		while
		    historyPoint := historyPoint - sizeof(char);
		    historyPoint* ~= '\e'
		do
		od;
	    fi;
	else
	    historyPoint := InputHistory + InputHistoryPos - sizeof(char);
	fi;
	while
	    n := cookedPos;
	    if historyPoint = InputHistory + sizeof(char) then
		false
	    else
		while
		    historyPoint := historyPoint - sizeof(char);
		    historyPoint* ~= '\e'
		do
		od;
		historyPoint := historyPoint + sizeof(char);
		p := &CookedBuffer[0];
		while
		    if n = 0 then
			false
		    else
			ch1 := historyPoint*;
			if ch1 >= 'A' and ch1 <= 'Z' then
			    ch1 := ch1 + ('a' - 'A');
			fi;
			ch2 := p*;
			if ch2 >= 'A' and ch2 <= 'Z' then
			    ch2 := ch2 + ('a' - 'A');
			fi;
			ch1 = ch2
		    fi
		do
		    n := n - 1;
		    historyPoint := historyPoint + sizeof(char);
		    p := p + sizeof(char);
		od;
		n ~= 0
	    fi
	do
	    while
		historyPoint := historyPoint - sizeof(char);
		historyPoint* ~= '\e'
	    do
	    od;
	od;
	if n = 0 then
	    /* Back up to the beginning of the input line, so that we display
	       it in the case it was originally entered. */
	    while cookedPos ~= 0 do
		historyPoint := historyPoint - sizeof(char);
		p := p - sizeof(char);
		cookedPos := cookedPos - 1;
	    od;
	    /* copy the retrieved history line into CookedBuffer */
	    while historyPoint* ~= '\e' do
		p* := historyPoint*;
		p := p + sizeof(char);
		historyPoint := historyPoint + sizeof(char);
		cookedPos := cookedPos + 1;
	    od;
	    CookedCount := cookedPos;
	    tx_displayInput();
	else
	    tx_clearInputLine();
	fi;
	LastWasNext := false;
	LastWasHistory := true;
	HistoryPoint := historyPoint;
    fi;
corp;

/*
 * tx_scrollInInput - scroll horizontally in the input line.
 *	Parameters are the signed character count to scroll and the
 *	character position of the left edge of the scroll rectangle,
 *	which does not include the prompt.
 */

proc tx_scrollInInput(int chars; uint leftChar)void:

    tx_doAnyScroll(chars * CHAR_WIDTH, 0,
		   LeftBorder + (PromptLen + leftChar) * CHAR_WIDTH,
		   TextInputTop, RightEdge, TextInputBottom);
corp;

/*
 * tx_insertCharacter - insert the given character into the input line.
 */

proc tx_insertCharacter(char ch)void:
    register *RastPort_t rp;

    if (PromptLen + CookedCount) < TextColumns - 1 and
	(ch >= IECODE_ASCII_FIRST + '\e' and
	 ch <= IECODE_ASCII_LAST + '\e' or
	 ch >= IECODE_LATIN1_FIRST + '\e' and
	 ch <= IECODE_LATIN1_LAST + '\e')
    then
	rp := TextRastPort;
	if CookedPos ~= CookedCount then
	    tx_scrollInInput(-1, CookedPos);
	    BlockCopyB(&CookedBuffer[CookedCount],
		&CookedBuffer[CookedCount - 1],
		CookedCount - CookedPos);
	fi;
	if EchoOn then
	    Text(rp, &ch, 1);
	else
	    Move(rp, rp*.rp_cp_x + CHAR_WIDTH, rp*.rp_cp_y);
	fi;
	CookedBuffer[CookedPos] := ch;
	CookedCount := CookedCount + 1;
	CookedPos := CookedPos + 1;
	LastWasHistory := false;
    fi;
corp;

/*
 * tx_inputLine - processing for an input line to go to the server.
 */

proc tx_inputLine(*char buffer; uint len)void:

    /* do this BEFORE 'sendLine', so that it doesn't change things
       (like prompt and echo) on us */
    putText(&PromptBuffer[0], PromptLen);
    if EchoOn then
	if LogFd ~= 0 and LogInputOnly then
	    ignore Write(LogFd, buffer, len);
	fi;
	putText(buffer, len);
    fi;
    if LogFd ~= 0 and LogInputOnly then
	ignore Write(LogFd, "\n", 1);
    fi;
    tx_putChar1('\n');
    sendLine(buffer, len);
corp;

/*
 * tx_deleteInInput - scroll input line left to delete some characters.
 */

proc tx_deleteInInput(register uint colStart, count)void:

    tx_scrollInInput(count, colStart);
    BlockCopy(&CookedBuffer[colStart], &CookedBuffer[colStart + 1],
	      CookedCount - colStart);
    CookedCount := CookedCount - count;
    CookedPos := CookedPos - count;
corp;

/*
 * tx_cookChar - take a character and treat it as an input character during
 *	normal text input. This involves echoing it, and handling some line
 *	editing, etc.
 */

proc tx_cookChar(char ch; bool wantInput)void:
    char
	CNTL_A = '\(0x01)',
	CNTL_B = '\(0x02)',
	CNTL_E = '\(0x05)',
	CNTL_F = '\(0x06)',
	CNTL_H = '\(0x08)',
	CNTL_I = '\(0x09)',
	CNTL_K = '\(0x0b)',
	CNTL_M = '\(0x0d)',
	CNTL_N = '\(0x0e)',
	CNTL_P = '\(0x10)',
	CNTL_R = '\(0x12)',
	CNTL_U = '\(0x15)',
	CNTL_W = '\(0x17)',
	CNTL_X = '\(0x18)',
	CNTL_Y = '\(0x19)',
	DEL = '\(0x7f)';
    register *RastPort_t rp;
    register uint pos;

    if ScrollType = st_input or ScrollType = st_either then
	tx_scrollToBottom();
    fi;
    tx_toggleCursor();
    rp := TextRastPort;
    case ch
    incase CNTL_A:
	/* cursor to beginning of line */
	CookedPos := 0;
    incase CNTL_B:
	/* cursor left one */
	if CookedPos ~= 0 then
	    CookedPos := CookedPos - 1;
	fi;
    incase CNTL_E:
	/* cursor to end of line */
	CookedPos := CookedCount;
    incase CNTL_F:
	/* cursor right one */
	if CookedPos ~= CookedCount then
	    CookedPos := CookedPos + 1;
	fi;
    incase CNTL_H:
	/* delete previous character */
	if CookedPos ~= 0 then
	    tx_deleteInInput(CookedPos - 1, 1);
	fi;
	LastWasHistory := false;
    incase CNTL_I:
	/* move forward to next 8 column boundary */
	CookedPos := (CookedPos + 8) & (-8);
	if CookedPos > CookedCount then
	    CookedPos := CookedCount;
	fi;
    incase CNTL_K:
	/* delete from cursor to end of line, save in UndoBuffer */
	BlockCopy(&UndoBuffer[0], &CookedBuffer[CookedPos],
		  CookedCount - CookedPos);
	CookedCount := CookedPos;
	CookedBuffer[CookedCount] := '\e';
	SetAPen(TextRastPort, ErasePen);
	RectFill(TextRastPort,
		 (PromptLen + CookedPos) * CHAR_WIDTH + LeftBorder,
		 TextInputTop, RightEdge, TextInputBottom);
	SetAPen(rp, TextPen);
    incase CNTL_M:
	/* carriage-return - send the current line */
	if not SendingLine and wantInput then
	    /* Note of explanation. If we are currently parsing something
	       out of the edit buffer, and the user goes back to the normal
	       window (Editing false), and then tries to enter another
	       input line, the line does not enter. This is because
	       'wantInput' will be false, because the parser in
	       mud.library does not need any input from us currently. */
	    SendingLine := true;
	    while
		if EchoOn then
		    case HistoryType
		    incase ht_full:
			tx_appendToInputHistory();
		    incase ht_new:
			if not LastWasHistory or
			    HistoryPoint ~=
				InputHistory + InputHistoryPos - sizeof(char)
			then
			    tx_appendToInputHistory();
			fi;
		    incase ht_minimal:
			if not LastWasHistory then
			    tx_appendToInputHistory();
			fi;
		    esac;
		fi;
		HistoryPoint := InputHistory + InputHistoryPos - sizeof(char);
		pos := CookedCount;
		tx_clearInputLine();
		/* Explanation: tx_inputLine calls into the code in
		   mud.library, which sends to the server. But, since it
		   is sort-of synchronous, in that it will respond to
		   messages from the server while doing that, it also
		   checks for and responds to IDCMP messages, which can
		   result in us getting back to this code again. We do not
		   want to recurse into tx_inputLine, etc., hence this
		   funny code here. */
		tx_inputLine(&CookedBuffer[0], pos);
		LastWasHistory := false;
		WantSendLine
	    do
		WantSendLine := false;
	    od;
	    SendingLine := false;
	else
	    WantSendLine := true;
	fi;
    incase CNTL_N:
	/* go to next line in input history */
	tx_historyNext();
    incase CNTL_P:
	/* go to previous line in input history */
	tx_historyPrevious();
    incase CNTL_R:
	/* search backwards in history */
	tx_historySearch();
    incase CNTL_U:
	/* delete from previous to beginning of line */
	if CookedPos ~= 0 then
	    tx_deleteInInput(0, CookedPos);
	fi;
	LastWasHistory := false;
    incase CNTL_W:
	/* delete the word to the left of the cursor */
	pos := CookedPos;
	if pos ~= 0 then
	    pos := pos - 1;
	    if pos ~= 0 then
		if CookedBuffer[pos] = ' ' then
		    pos := pos - 1;
		fi;
		while pos ~= 0 and CookedBuffer[pos] ~= ' ' do
		    pos := pos - 1;
		od;
		if CookedBuffer[pos] = ' ' then
		    pos := pos + 1;
		fi;
	    fi;
	    tx_deleteInInput(pos, CookedPos - pos);
	fi;
    incase CNTL_X:
	/* erase entire input line */
	tx_clearInputLine();
	LastWasHistory := false;
    incase CNTL_Y:
	/* insert from the undo buffer and move to end of line */
	pos := 0;
	while UndoBuffer[pos] ~= '\e' do
	    tx_insertCharacter(UndoBuffer[pos]);
	    pos := pos + 1;
	od;
	CookedPos := CookedCount;
    incase DEL:
	/* delete the current character */
	if CookedPos ~= CookedCount then
	    tx_deleteInInput(CookedPos, 1);
	    /* compensate for subtract in 'tx_deleteInInput' */
	    CookedPos := CookedPos + 1;
	fi;
	LastWasHistory := false;
    default:
	tx_insertCharacter(ch);
    esac;
    tx_toggleCursor();
corp;

/*
 * tx_specialKey - handle a special keypress.
 *	Convention for history - 'HistoryPoint' points to the \e at the
 *	end of the line we have just displayed.
 */

proc tx_specialKey(uint ch)void:

    if ScrollType = st_input or ScrollType = st_either then
	tx_scrollToBottom();
    fi;
    tx_toggleCursor();
    case ch
    incase KEY_LEFT:
	/* cursor left one */
	if CookedPos ~= 0 then
	    CookedPos := CookedPos - 1;
	fi;
    incase KEY_RIGHT:
	/* cursor right one */
	if CookedPos ~= CookedCount then
	    CookedPos := CookedPos + 1;
	fi;
    incase KEY_SHIFT | KEY_LEFT:
	/* cursor to beginning of line */
	CookedPos := 0;
    incase KEY_SHIFT | KEY_RIGHT:
	/* cursor to end of line */
	CookedPos := CookedCount;
    incase KEY_UP:
	/* previous line in history */
	tx_historyPrevious();
    incase KEY_DOWN:
	/* next line in history */
	tx_historyNext();
    incase KEY_SHIFT | KEY_UP:
	/* search backwards in history */
	tx_historySearch();
    incase KEY_SHIFT | KEY_DOWN:
	/* go to the end of the input history and clear the input line */
	HistoryPoint := InputHistory + InputHistoryPos - 2 * sizeof(char);
	tx_clearInputLine();
    default:
	sendKey(ch);
    esac;
    tx_toggleCursor();
corp;

/*
 * tx_redraw - redraw the text display. 'full' is set when we are called
 *	from 'ed_hide'.
 */

proc tx_redraw(bool full)void:
    uint drawTo;

    tx_setValues();
    tx_reconfigure();
    /* It appears that when a SizeWindow is done such that the window is
       being enlarged, the text that was in the old small frame is put at
       the bottom of the new larger frame. So, we don't want to repaint
       that portion, since it shows up as visible flicker if we do so. */
    if PictureShown then
	if full and OldStyle then
	    /* Coming from editing - have to erase bottom of text window */
	    SetAPen(TextRastPort, ERASE_PEN);
	    RectFill(TextRastPort, 0, TextHalfHeight,
		     TextWidth - 1, TextFullHeight - BottomBorder - 1);
	fi;
	tx_repaint(0, TextHalfLines - 2);
    else
	if full or FullRedraw then
	    drawTo := TextFullLines - 2;
	else
	    drawTo := LineDifference - 1;
	    if TextHalfLines > drawTo then
		/* We have to draw into the top part of the window, where
		   space has been exposed for us to draw into, but we also
		   have to draw enough to overwrite all of the half-window
		   contents that is there - Intuition *copies* the window
		   data down, it doesn't scroll it down and clear. */
		drawTo := TextHalfLines;
	    fi;
	fi;
	tx_repaint(0, drawTo);
    fi;
    if full or FullRedraw or Mode = md_oneWindow or Mode = md_twoWindow or
	(Mode = md_oneScreen or Mode = md_twoScreen) and PictureShown
    then
	/* When hiding the picture, Intuition grows the picture in such
	   a way that the bottom of the view is OK, so we do not need to
	   redraw the input line. When shrinking the window, we do,
	   however. */
	tx_refreshInputLine();
	tx_displayInput();
	tx_toggleCursor();
    fi;
corp;

/*
 * tx_doBigScroll - helper for tx_graphicsOn and tx_graphicsOff. Scroll
 *	the commonly visible text (that which stays across modes) either
 *	towards the origin ('down' false) or away from it.
 */

proc tx_doBigScroll(bool down)void:
    register int delta;

    delta := TextFullHeight - TextHalfHeight;
    if down then
	delta := -delta;
    fi;
    ScrollRaster(TextRastPort, 0, delta,
		 LeftBorder, TextTop, RightEdge, TextFullInputBottom);
corp;

/*
 * tx_graphicsOn - text stuff needed when user turns graphics on.
 */

proc tx_graphicsOn()void:

    WindowTopLine := WindowTopLine + LineDifference;
    if OldStyle then
	/* no text redraw needed - it will all be scrolled in */
	tx_toggleCursor();
	tx_doBigScroll(false);
	SetAPen(TextRastPort, ERASE_PEN);
	RectFill(TextRastPort, 0, TextHalfLines * CHAR_HEIGHT + TextTop,
	    TextWidth - 1, TextFullInputBottom);
	SetAPen(TextRastPort, CST_TEXT_PEN);
	tx_setValues();
	tx_reconfigure();
	RefreshGadgets(ScrollGadget, TextWindow, nil);
	tx_toggleCursor();
    fi;
corp;

/*
 * tx_graphicsOff - text stuff needed when user turns graphics off.
 */

proc tx_graphicsOff()void:

    if OldStyle then
	tx_toggleCursor();
	tx_doBigScroll(true);
	tx_setValues();
	if WindowTopLine < make(LineDifference, int) then
	    WindowTopLine := OutputHistoryLines - OutputLines;
	    tx_reconfigure();
	    RefreshGadgets(ScrollGadget, TextWindow, nil);
	    tx_repaint(0, TextFullLines - 2);
	else
	    WindowTopLine := WindowTopLine - LineDifference;
	    tx_reconfigure();
	    RefreshGadgets(ScrollGadget, TextWindow, nil);
	    tx_repaint(0, LineDifference);
	fi;
	tx_toggleCursor();
    else
	/* The tx_setValues call is done in tx_redraw, but we need the
	   new value of OutputLines here, so we end up doing it twice. */
	tx_setValues();
	if WindowTopLine < make(LineDifference, int) then
	    WindowTopLine := OutputHistoryLines - OutputLines;
	else
	    WindowTopLine := WindowTopLine - LineDifference;
	fi;
    fi;
corp;

/*
 * tx_flushLine - flush the current buffered stuff to the indicated
 *	length. Also add it all, and a newline, to the history buffer.
 */

proc tx_flushLine(register uint len)void:
    register *char p;
    register *RastPort_t rp;

    p := &TextBuff[0];
    if LogFd ~= 0 and not LogInputOnly then
	TextBuff[len] := '\n';
	ignore Write(LogFd, p, len + 1);
    fi;
    if len ~= 0 then
	if WindowTopLine = OutputHistoryLines - OutputLines and not Editing
	then
	    /* only display the line if the bottom line is currently within
	       the viewing window selected by the user */
	    rp := TextRastPort;
	    if not LastWasOutput then
		LastWasOutput := true;
		Move(rp, LeftBorder, TextOutputRow);
	    fi;
	    if BrightBold then
		SetAPen(rp, ErasePen);
		SetBPen(rp, TextPen);
		SetDrMd(rp, JAM2);
	    fi;
	    if len > TextColumns then
		len := TextColumns;
	    fi;
	    Text(rp, p, len);
	    if BrightBold then
		SetAPen(rp, TextPen);
		SetBPen(rp, ErasePen);
		SetDrMd(rp, JAM1);
	    fi;
	fi;
	while len ~= 0 do
	    len := len - 1;
	    tx_outputHistory(p*);
	    p := p + sizeof(char);
	od;
    fi;
    tx_outputHistory('\n');
    AtNewLine := true;
corp;

/*
 * tx_putChar1 - output a character to the text output screen.
 */

proc tx_putChar1(register char ch)void:
    register *char p, q;
    register uint i;

    if ScrollType = st_output or ScrollType = st_either then
	if Editing then
	    WindowTopLine := OutputHistoryLines - OutputLines;
	else
	    tx_scrollToBottom();
	fi;
    fi;
    if AtNewLine then
	/* we delay the scroll until the first character after the newline,
	   to keep the most data on the screen in the normal cases */
	AtNewLine := false;
	if WindowTopLine = OutputHistoryLines - OutputLines and not Editing
	then
	    /* only do the scroll if the user is viewing the last line of
	       the display, otherwise just leave it still - the user will
	       see the new output when he windows down to it. */
	    tx_doScrollWindow(1);
	    Move(TextRastPort, LeftBorder, TextOutputRow);
	fi;
    fi;
    if ch = '\n' then
	tx_flushLine(TextLinePos);
	/* Have to delay the 'checkRefresh' call to this point, rather
	   than just after the scroll call, since the 'tx_flushLine' call
	   gets the various counters correct to allow a tx_refresh call
	   to display the correct data. */
	checkRefresh();
	TextLinePos := 0;
	TextColumn := 0;
	TextWordPos := 0;
    else
	if TextColumn >= TextColumns then
	    /* we are already at the last column we want to use */
	    if TextWordPos ~= 0 and ch ~= ' ' and ch ~= '\t' and
		TextBuff[TextLinePos - 1] ~= ' '
	    then
		/* we have more than one word in the buffer and the current
		   output character is not one we can break at - do a word wrap
		   at the end of the previous word, and shuffle what we have
		   of the current word to the beginning of the buffer. */
		tx_flushLine(TextWordPos - 1);
		TextColumn := TextLinePos - TextWordPos;
		if TextColumn ~= 0 then
		    p := &TextBuff[TextWordPos];
		    q := &TextBuff[0];
		    for i from TextColumn - 1 downto 0 do
			q* := p*;
			q := q + sizeof(char);
			p := p + sizeof(char);
		    od;
		fi;
		TextWordPos := 0;
		TextLinePos := TextColumn;
	    else
		/* one word won't fit on the line - break it right here */
		tx_flushLine(TextLinePos);
		TextColumn := 0;
		TextWordPos := 0;
		TextLinePos := 0;
		if ch = ' ' or ch = '\t' then
		    /* if the word EXACTLY fit, discard any spacing */
		    ch := '\e';
		fi;
	    fi;
	fi;
	if ch ~= '\e' then
	    if ch = '\t' then
		/* 8 column tab */
		while
		    TextBuff[TextLinePos] := ' ';
		    TextLinePos := TextLinePos + 1;
		    TextColumn := TextColumn + 1;
		    (TextColumn + 8) & (8 - 1) ~= 0
		do
		od;
	    else
		if ch ~= ' ' and
		    (TextLinePos = 0 or TextBuff[TextLinePos - 1] = ' ')
		then
		    /* current character is the beginning of a word */
		    TextWordPos := TextLinePos;
		fi;
		TextBuff[TextLinePos] := ch;
		TextLinePos := TextLinePos + 1;
		TextColumn := TextColumn + 1;
	    fi;
	fi;
    fi;
corp;

/*
 * newLine - a new line on text output.
 */

proc newLine()void:

    /* These assignments to LastWasOutput are needed because the state of
       the rastport can change without us knowing about it, e.g. when the
       user is doing a Workbench-to-front with the LEFT-AMIGA-N combo. */
    LastWasOutput := false;
    tx_putChar1('\n');
corp;

/*
 * putChar - user interface to tx_putChar1.
 */

proc putChar(char ch)void:

    LastWasOutput := false;
    tx_putChar1(ch);
corp;

/*
 * putString - push a whole string through the text output. This handles
 *	automatic word wrap, newlines, indents, pagination, tabs, etc.
 */

proc putString(register *char st)void:

    LastWasOutput := false;
    while st* ~= '\e' do
	tx_putChar1(st*);
	st := st + sizeof(char);
    od;
corp;

/*
 * putText - put a buffer of text to the text output window. Do all appropriate
 *	word wrapping, etc.
 */

proc putText(register *char p; register uint len)void:
    byte BRIGHT_BOLD = 0x01;

    LastWasOutput := false;
    while len ~= 0 do
	len := len - 1;
	if p* = '\(BRIGHT_BOLD)' then
	    BrightBold := not BrightBold;
	else
	    tx_putChar1(p*);
	fi;
	p := p + sizeof(char);
    od;
corp;

/*
 * putUnsigned - output a 32 bit unsigned decimal value.
 */

proc putUnsigned(register ulong n)void:
    register *char p;
    [12] char buffer;

    p := &buffer[11];
    p* := '\e';
    while
	p := p - sizeof(char);
	p* := n % 10 + '0';
	n := n / 10;
	n ~= 0
    do
    od;
    putString(p);
corp;

/*
 * setEcho - enable/disable echoing (also storing to history)
 */

proc setEcho(bool flag)void:

    EchoOn := flag;
corp;

/*
 * setPrompt - set a prompt for input.
 */

proc setPrompt(register *char pr)void:
    register *char p;

    if not SendingLine then
	tx_toggleCursor();
    fi;
    PromptLen := 0;
    p := &PromptBuffer[0];
    while pr* ~= '\e' do
	p* := pr*;
	p := p + sizeof(char);
	pr := pr + sizeof(char);
	PromptLen := PromptLen + 1;
    od;
    /* Some are up to 36 characters in mud.library */
    if PromptLen > 50 then
	PromptLen := 50;
    fi;
    if not Editing then
	if EchoOn then
	    tx_refreshInputLine();
	    tx_displayInput();
	else
	    tx_clearInputLine();
	fi;
	if not SendingLine then
	    tx_toggleCursor();
	fi;
    fi;
corp;

/*
 * tx_doForceScroll - scroll the visible text window by the indicated
 *	number of text lines. Negative scrolls forward, positive backwards.
 *	This variant, for forceChars, uses the entire display area, not
 *	omitting the "input line".
 */

proc tx_doForceScroll(int lines)void:

    tx_doAnyScroll(0, lines * CHAR_HEIGHT,
		   LeftBorder, TextTop, RightEdge, TextInputBottom);
corp;

/*
 * forceChars - output a bufferful of characters and force them to appear.
 */

proc forceChars(register *char p; register ulong len)void:
    register *RastPort_t rp;
    register ulong count;
    register uint column, lastColumn;
    register char ch;
    char ESCAPE = '\(0x1b)';

    tx_toggleCursor();
    rp := TextRastPort;
    column := CookedPos;
    lastColumn := column;
    count := 0;
    while len ~= 0 do
	ch := p*;
	p := p + sizeof(char);
	len := len - 1;
	if EscapeState = es_none then
	    if ch = '\r' then
		if count ~= 0 then
		    tx_inputMove(lastColumn);
		    Text(rp, &CookedBuffer[lastColumn], count);
		    count := 0;
		fi;
		/* Do all this even if column = 0, so empty lines are kept. */
		CookedBuffer[column] := '\n';
		if LogFd ~= 0 then
		    ignore Write(LogFd, &CookedBuffer[0], column + 1);
		fi;
		for lastColumn from 0 upto column do
		    tx_outputHistory(CookedBuffer[lastColumn]);
		od;
		if column = 0 and not LastWasNewline then
		    /* Make empty lines scroll up */
		    tx_doForceScroll(1);
		fi;
		column := 0;
		lastColumn := 0;
		Move(rp, LeftBorder, TextOutputRow);
	    elif ch = '\n' then
		if count ~= 0 then
		    tx_inputMove(lastColumn);
		    Text(rp, &CookedBuffer[lastColumn], count);
		    count := 0;
		    lastColumn := column;
		fi;
		tx_doForceScroll(1);
	    elif ch = '\b' then
		if column ~= 0 then
		    if lastColumn = column then
			lastColumn := lastColumn - 1;
		    fi;
		    column := column - 1;
		    /* destructive backspace on screen */
		    tx_inputMove(column);
		    Text(rp, " ", 1);
		fi;
		if count ~= 0 then
		    count := count - 1;
		fi;
	    elif ch = '\t' then
		while
		    if column = TextColumns then
			tx_inputMove(lastColumn);
			Text(rp, &CookedBuffer[lastColumn], count);
			count := 0;
			if LogFd ~= 0 then
			    ignore Write(LogFd, &CookedBuffer[0], column);
			fi;
			for lastColumn from 0 upto column - 1 do
			    tx_outputHistory(CookedBuffer[lastColumn]);
			od;
			tx_outputHistory('\n');
			column := 0;
			lastColumn := 0;
			tx_doForceScroll(1);
			tx_inputMove(0);
		    fi;
		    count := count + 1;
		    CookedBuffer[column] := ' ';
		    column := column + 1;
		    column ~= TextColumns and column & 7 ~= 0
		do
		od;
	    elif ch = ESCAPE then
		EscapeState := es_gotEsc;
	    elif ch - '\e' ~= 0x7f and (ch - '\e') & 0x7f + '\e' >= ' ' then
		if column = TextColumns then
		    tx_inputMove(lastColumn);
		    Text(rp, &CookedBuffer[lastColumn], count);
		    count := 0;
		    if LogFd ~= 0 then
			ignore Write(LogFd, &CookedBuffer[0], column);
		    fi;
		    for lastColumn from 0 upto column - 1 do
			tx_outputHistory(CookedBuffer[lastColumn]);
		    od;
		    tx_outputHistory('\n');
		    column := 0;
		    lastColumn := 0;
		    tx_doForceScroll(1);
		    tx_inputMove(0);
		fi;
		count := count + 1;
		CookedBuffer[column] := ch;
		column := column + 1;
	    fi;
	    LastWasNewline := ch = '\n';
	else
	    /* In some phase of ANSI escape sequence processing. */
	    if EscapeState = es_gotEsc then
		/* Should check for a '[' here, but what the heck! */
		EscapeState := es_gotLSq;
	    else
		/* es_gotLSq */
		if not (ch = ';' or ch >= '0' and ch <= '9') then
		    EscapeState := es_none;
		fi;
	    fi;
	fi;
    od;
    if count ~= 0 then
	tx_inputMove(lastColumn);
	Text(rp, &CookedBuffer[lastColumn], count);
    fi;
    CookedPos := column;
    tx_toggleCursor();
corp;

/*
 * logMode - set the logging mode.
 */

proc logMode(bool inputOnly)void:

    LogInputOnly := inputOnly;
corp;

proc buildPath(*char buffer, dir, name)void:

    if dir = nil or dir* = '\e' then
	CharsCopy(buffer, name);
    else
	CharsCopy(buffer, dir);
	dir := dir + (CharsLen(dir) - 1) * sizeof(char);
	if dir* ~= ':' and dir* ~= '/' then
	    CharsConcat(buffer, "/");
	fi;
	CharsConcat(buffer, name);
    fi;
corp;

/*
 * logOn - open the specified log file and start using it.
 */

proc logOn(*char dir, fileName)bool:
    [500] char name;
    register Handle_t fd;

    buildPath(&name[0], dir, fileName);
    fd := Open(&name[0], MODE_OLDFILE);
    if fd ~= 0 then
	/* the file exists, but we don't know what kind yet */
	Close(fd);
	fd := Open(&name[0], MODE_READWRITE);
	if fd ~= 0 then
	    ignore Seek(fd, 0, OFFSET_END);
	else
	    /* might be something like PIPE: or PRT: */
	    fd := Open(&name[0], MODE_OLDFILE);
	fi;
    else
	fd := Open(&name[0], MODE_NEWFILE);
    fi;
    if fd = 0 then
	putString("::Can't open log file \"");
	putString(&name[0]);
	putString("\" - logging not turned on.\n");
	false
    else
	putString("::Logging to file \"");
	putString(&name[0]);
	putString("\".\n");
	LogFd := fd;
	true
    fi
corp;

/*
 * logOff - turn off logging and close the file.
 */

proc logOff()void:

    Close(LogFd);
    LogFd := 0;
    putString("::Logging turned off.\n");
corp;

/*
 * tx_snapshot - dump the output history buffer to a file.
 */

proc tx_snapshot(*char dir, fileName)void:
    [500] char name;
    Handle_t fd;

    buildPath(&name[0], dir, fileName);
    fd := Open(&name[0], MODE_NEWFILE);
    if fd = 0 then
	putString("::Can't open snapshot file \"");
	putString(&name[0]);
	putString("\" - text snapshot not made.\n");
    else
	if Write(fd, OutputHistory, OutputHistoryPos) ~= OutputHistoryPos then
	    putString("::Error writing to snapshot file \"");
	else
	    putString("::Text snapshot made to file \"");
	fi;
	putString(&name[0]);
	putString("\".\n");
	Close(fd);
    fi;
corp;
